module cubeio_data
  use cubetools_parameters
  use cubeio_messaging

  type cubeio_data_t
    ! Dimensions
    integer(kind=pixe_k) :: nx = 0  ! Number of X pixels
    integer(kind=pixe_k) :: ny = 0  ! Number of Y pixels
    integer(kind=chan_k) :: nc = 0  ! Number of channels
    logical :: iscplx = .false.     ! R*4 or C*4?
    real(kind=sign_k),    allocatable :: r4(:,:,:)  ! [nx,ny,nc] (LMV) or [nc,nx,ny] (VLM) R*4 data cube
    complex(kind=sign_k), allocatable :: c4(:,:,:)  ! [nx,ny,nc] (LMV) or [nc,nx,ny] (VLM) C*4 data cube
    ! Flag
    integer(kind=code_k) :: ready = code_buffer_none  ! Is the cubeio_data_t ready for use?
  contains
    procedure :: memsize => cubeio_data_memsize
  end type cubeio_data_t

  public :: cubeio_data_t
  public :: cubeio_data_reallocate,cubeio_data_free
  private

contains

  subroutine cubeio_data_reallocate(cubset,data,iscplx,nx,ny,nc,order,error)
    use gkernel_interfaces
    use cubetools_setup_types
    !-------------------------------------------------------------------
    ! (Re)allocate a cubeio_data_t
    ! Do nothing when the array sizes did not changed
    !-------------------------------------------------------------------
    type(cube_setup_t),   intent(in)    :: cubset
    type(cubeio_data_t),  intent(inout) :: data
    logical,              intent(in)    :: iscplx
    integer(kind=pixe_k), intent(in)    :: nx
    integer(kind=pixe_k), intent(in)    :: ny
    integer(kind=chan_k), intent(in)    :: nc
    integer(kind=4),      intent(in)    :: order
    logical,              intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='REALLOCATE>CUBE>DATA'
    integer(kind=4) :: ier
    integer(kind=index_length) :: odim(3),ndim(3)
    logical :: calloc
    !
    call cubeio_message(ioseve%trace,rname,'Welcome')
    !
    ! Sanity checks
    if (nx.le.0) then
      call cubeio_message(seve%e,rname,'Number of X pixels is null or negative')
      error = .true.
    endif
    if (ny.le.0) then
      call cubeio_message(seve%e,rname,'Number of Y pixels is null or negative')
      error = .true.
    endif
    if (nc.le.0) then
      call cubeio_message(seve%e,rname,'Number of channels is null or negative')
      error = .true.
    endif
    if (error)  return
    !
    select case (order)
    case (code_cube_imaset)
      ndim(1) = nx
      ndim(2) = ny
      ndim(3) = nc
    case (code_cube_speset)
      ndim(1) = nc
      ndim(2) = nx
      ndim(3) = ny
    case default
      call cubeio_message(seve%e,rname,'Unsupported data order')
      error = .true.
      return
    end select
    !
    ! Allocation or reallocation?
    if (data%iscplx) then
      calloc = allocated(data%c4)
    else
      calloc = allocated(data%r4)
    endif
    if (calloc) then
      ! Reallocation?
      if (data%iscplx) then
        odim(:) = shape(data%c4)
      else
        odim(:) = shape(data%r4)
      endif
      if ((data%iscplx.eqv.iscplx) .and. all(odim.eq.ndim)) then
        ! Same type and same size => Nothing to be done!
        call cubeio_message(ioseve%alloc,rname,'Data array already allocated with correct size')
        goto 100
      else  ! Different type or different size => reallocation
        call cubeio_message(ioseve%alloc,rname,'Reallocating data array')
        call cubeio_data_free(data,error)
        if (error)  return
      endif
    else
      ! Allocation
      call cubeio_message(ioseve%alloc,rname,'Creating data array')
    endif
    !
    ! Reallocate memory of the right size
    if (iscplx) then
      allocate(data%c4(ndim(1),ndim(2),ndim(3)),stat=ier)
    else
      allocate(data%r4(ndim(1),ndim(2),ndim(3)),stat=ier)
    endif
    if (failed_allocate(rname,'data array',ier,error)) return
    !
  100 continue
    ! Operation success
    data%nx = nx
    data%ny = ny
    data%nc = nc
    data%iscplx = iscplx
    data%ready = code_buffer_none
    !
  end subroutine cubeio_data_reallocate

  subroutine cubeio_data_free(data,error)
    !---------------------------------------------------------------------
    ! Free a 'cubeio_data_t' instance
    !---------------------------------------------------------------------
    type(cubeio_data_t), intent(inout) :: data   !
    logical,             intent(inout) :: error  !
    !
    if (data%iscplx) then
      if (allocated(data%c4))  deallocate(data%c4)
    else
      if (allocated(data%r4))  deallocate(data%r4)
    endif
    !
    data%iscplx = .false.
    data%nx = 0
    data%ny = 0
    data%nc = 0
    data%ready = code_buffer_none
  end subroutine cubeio_data_free

  function cubeio_data_memsize(data)
    !-------------------------------------------------------------------
    ! Return the memory footprint in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_data_memsize
    class(cubeio_data_t), intent(in) :: data
    !
    cubeio_data_memsize = 0
    !
    if (allocated(data%r4)) &
      cubeio_data_memsize = cubeio_data_memsize + size(data%r4,kind=size_length)*4
    !
    if (allocated(data%c4)) &
      cubeio_data_memsize = cubeio_data_memsize + size(data%c4,kind=size_length)*8
    !
  end function cubeio_data_memsize

end module cubeio_data
