module cubefit_function_spectral_shell
  use fit_minuit
  !
  use cubefit_messaging
  use cubefit_spectral_parameters
  use cubefit_spectral_obs
  !
  integer(kind=npar_k),parameter :: nparline = 4
  integer(kind=npar_k),parameter :: iarea   = 1
  integer(kind=npar_k),parameter :: ifreq   = 2
  integer(kind=npar_k),parameter :: ifwzl   = 3
  integer(kind=npar_k),parameter :: ihorn   = 4
  !
  ! Contrary to other methods this ones uses the frequency axis
  !
  public cubefit_function_spectral_shell_init, cubefit_function_spectral_shell_minimize
  public cubefit_function_spectral_shell_extract, cubefit_function_spectral_shell_residuals
  public cubefit_function_spectral_shell_npar, cubefit_function_spectral_shell_user2par
  public cubefit_function_spectral_shell_par2spec, cubefit_function_spectral_shell_spec2par
  public cubefit_function_spectral_shell_iterate,cubefit_function_spectral_shell_wind2par
  public cubefit_function_spectral_shell_flags, cubefit_function_spectral_shell_doprofile
  public cubefit_function_spectral_shell_units
  private
  !
contains
  !
  subroutine cubefit_function_spectral_shell_init(par,obs,minuit,error)
    !------------------------------------------------------------------------
    ! 
    !------------------------------------------------------------------------
    type(spectral_pars_t), intent(inout) :: par
    type(spectral_obs_t),  intent(in)    :: obs
    type(fit_minuit_t),    intent(inout) :: minuit
    logical,               intent(inout) :: error
    !
    real(kind=coor_k) :: freq,fwzl
    real(kind=sign_k) :: val,area
    character(len=mess_l) :: mess
    integer(kind=4) :: ipar,iline
    integer(kind=chan_k) :: ichan
    character(len=*), parameter :: rname='SPECTRAL>SHELL>INIT'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')    
    !     
    ! Starting values
    if (par%nline.eq.0) then ! Automatic guess
       par%leaders(:) = 0
       par%flag(:,:) = 0
       par%errs(:) = 0
       area=0.
       freq=0.
       fwzl=0.
       do ichan=obs%ifirst+1,obs%ilast-1
          if (obs%wfit(ichan).ne.0 .and. abs(obs%spec%t(ichan)).gt.obs%sigbase) then
             val = obs%spec%t(ichan)
             area = area + val
             freq = freq + val * obs%spec%f(ichan)
             fwzl = fwzl + val * obs%spec%f(ichan)**2
          endif
       enddo
       if (area.ne.0) then
          freq = freq / area
          fwzl = fwzl / area
          par%pars(ifwzl) = obs%deltaf * sqrt (abs(fwzl-freq**2)*8.*alog(2.))
          par%pars(ifreq) = freq
          par%pars(iarea) = area * obs%deltaf
       else
          call cubefit_message(seve%e,rname,'Null area found, give explicit guesses')
          error =  .true.
          return
       endif
       minuit%nu=nparline
    else ! Manipulate initial guesses from user
       minuit%nu=nparline*par%nline
    endif
    minuit%nu=minuit%nu+nparline
    !
    ! User feedback on par%parsameters used, useful for debug only...
    call cubefit_message(fitseve%others,rname,'Input Parameters:  Area Position Fwzl Horn')
    ipar = 0
    do iline=1,max(par%nline,1)
       write (mess,'(5x,4(5x,1pg11.4))') par%pars(ipar+iarea),par%pars(ipar+ifreq),par%pars(ipar+ifwzl),&
            par%pars(ipar+ihorn)
       call cubefit_message(fitseve%others,rname,mess)
       ipar=ipar+nparline
    enddo
    !
    ! Set up Parameters
    ! Areas
    if (par%leaders(iarea).eq.0) then
       minuit%u(iarea)   = 1.0
       minuit%werr(iarea)= 0.0
    else
       ipar = (nparline-1)*par%leaders(iarea)+iarea
       minuit%u(iarea)=par%pars(ipar)
       if (par%flag(par%leaders(iarea),iarea).eq.4) then
          minuit%werr(iarea)=0.
       else
          ! minuit%werr(K) = obs%sigbase*sqrt(abs(par%pars(k+2))*obs%deltaf)
          minuit%werr(iarea)=obs%sigbase*obs%deltaf*3.d0
          if (par%errs(ipar).ne.0) minuit%werr(iarea)=par%errs(ipar)
          if (minuit%u(iarea).ne.0.d0) then
             minuit%alim(iarea)=min(0.d0,8.d0*minuit%u(iarea))
             minuit%blim(iarea)=max(0.d0,8.d0*minuit%u(iarea))
          else
             minuit%lcode(iarea)=1
          endif
       endif
    endif
    !
    ! Frequencies
    if (par%leaders(ifreq).eq.0) then
       minuit%u(ifreq)=0.
       minuit%werr(ifreq)=0.
    else
       ipar = (nparline-1)*par%leaders(ifreq)+ifreq
       minuit%u(ifreq)=par%pars(ipar)
       if (par%flag(par%leaders(ifreq),ifreq).eq.4) then
          minuit%werr(ifreq)=0.
       else
          minuit%werr(ifreq)=obs%deltaf
          if (par%errs(ipar).ne.0) minuit%werr(ifreq)=par%errs(ipar)
          minuit%alim(ifreq)=minuit%u(ifreq)-0.1*obs%spec%n*obs%deltaf
          minuit%blim(ifreq)=minuit%u(ifreq)+0.1*obs%spec%n*obs%deltaf
       endif
    endif
    !
    ! Line Widths
    if (par%leaders(ifwzl).eq.0) then
       minuit%u(ifwzl)=1.
       minuit%werr(ifwzl)=0.
    else
       ipar = (nparline-1)*par%leaders(ifwzl)+ifwzl
       minuit%u(ifwzl)=abs(par%pars(ipar))
       if (par%flag(par%leaders(ifwzl),ifwzl).eq.4) then
          minuit%werr(ifwzl)=0.
       else
          minuit%werr(ifwzl)=obs%deltaf
          if (par%errs(ipar).ne.0) minuit%werr(ifwzl)=par%errs(ipar)
          minuit%alim(ifwzl)=obs%deltaf
          minuit%blim(ifwzl)=0.5*obs%spec%n*obs%deltaf
       endif
    endif
    !
    ! Horn to Center
    if (par%leaders(ihorn).eq.0) then
       minuit%u(ihorn)=1.
       minuit%werr(ihorn)=0.d0
    else
       ipar = (nparline-1)*par%leaders(ihorn)+ihorn
       minuit%u(ihorn)=par%pars(ipar)
       if (par%flag(par%leaders(ihorn),ihorn).eq.4) then
          minuit%werr(ihorn)=0.d0
       else
          minuit%werr(ihorn)=0.05
          if (par%errs(ipar).ne.0) minuit%werr(ihorn)=par%errs(ipar)
          minuit%alim(ihorn)=-1.0
          minuit%blim(ihorn)=100.
       endif
    endif
    ! Set up parameters for Secondary Variables
    ipar=5
    do iline=1,max(par%nline,1)
       ! Area
       minuit%u(ipar)=par%pars(ipar-nparline)
       if (par%flag(iline,iarea).eq.0 .or. par%nline.eq.0) then
          minuit%werr(ipar)=obs%sigbase*obs%deltaf*3.0
          if (par%errs(ipar-nparline).ne.0) minuit%werr(ipar)=par%errs(ipar-nparline)
          if (minuit%u(ipar).ne.0.) then
             minuit%alim(ipar)=min(0.d0,8.d0*minuit%u(ipar))
             minuit%blim(ipar)=max(0.d0,8.d0*minuit%u(ipar))
          else
             minuit%lcode(ipar)=1             ! No Boundaries if minuit%u(ipar)=0.
          endif
       else
          minuit%werr(ipar)=0.d0
          if (iline.eq.par%leaders(iarea)) minuit%u(ipar)=1.d0
       endif
       ipar=ipar+1
       !
       ! Velocity
       minuit%u(ipar)=par%pars(ipar-nparline)
       if (par%flag(iline,ifreq).eq.0 .or. par%nline.eq.0) then
          minuit%werr(ipar)=obs%deltaf
          if (par%errs(ipar-nparline).ne.0) minuit%werr(ipar)=par%errs(ipar-nparline)
          minuit%alim(ipar)=minuit%u(ipar)-0.1*obs%spec%n*obs%deltaf
          minuit%blim(ipar)=minuit%u(ipar)+0.1*obs%spec%n*obs%deltaf
       else
          minuit%werr(ipar)=0.d0
          if (iline.eq.par%leaders(ifreq)) minuit%u(ipar)=0.
       endif
       ipar=ipar+1
       !
       ! Line Width
       minuit%u(ipar)=abs(par%pars(ipar-nparline))
       if (par%flag(iline,ifwzl).eq.0 .or. par%nline.eq.0) then
          minuit%werr(ipar)=obs%deltaf
          if (par%errs(ipar-nparline).ne.0) minuit%werr(ipar)=par%errs(ipar-nparline)
          minuit%alim(ipar)=obs%deltaf
          minuit%blim(ipar)=0.5*obs%spec%n*obs%deltaf
       else
          minuit%werr(ipar)=0.d0
          if (iline.eq.par%leaders(ifwzl)) minuit%u(ipar)=1.
       endif
       ipar=ipar+1
       !
       ! Horn / Center ratio
       minuit%u(ipar)=par%pars(ipar-nparline)
       if (par%flag(iline,ihorn).eq.0 .or. par%nline.eq.0) then
          minuit%werr(ipar)=0.05
          if (par%errs(ipar-nparline).ne.0) minuit%werr(ipar)=par%errs(ipar-nparline)
          minuit%alim(ipar)=-1.0
          minuit%blim(ipar)=100.
       else
          minuit%werr(ipar)=0.d0
          if (iline.eq.par%leaders(ihorn)) minuit%u(ipar)=1.
       endif
       ipar=ipar+1
    enddo
  end subroutine cubefit_function_spectral_shell_init
  !
  subroutine cubefit_function_spectral_shell_minimize(npar,grad,chi2,pars,iflag,obs)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    integer(kind=npar_k),   intent(in)    :: npar        ! Number of parameters
    real(kind=grad_k),      intent(out)   :: grad(npar)  ! Gradientes
    real(kind=chi2_k),      intent(out)   :: chi2        ! chi squared
    real(kind=para_k),      intent(in)    :: pars(npar)  ! Parameter values
    integer(kind=4),        intent(in)    :: iflag       ! Code operation
    type(spectral_obs_t),   intent(inout) :: obs         ! Observation
    !
    integer(kind=chan_k) :: ichan,nline
    real(kind=coor_k) :: xfreq,eps
    real(kind=sign_k) :: sumpred,diff
    real(kind=para_k), allocatable :: repars(:,:)
    real(kind=para_k) :: areasca,freqoff,fwzlsca,hornsca
    real(kind=grad_k), allocatable :: regrad(:,:),scagrad(:)
    integer(kind=line_k) :: iline
    integer(kind=npar_k) :: ipar
    logical :: dograd
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>MINIMIZE'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    ! Final computations
    if (iflag.eq.minuit_rms_flag) then
       call obs%sigma(cubefit_function_spectral_shell_profile,.true.)
       return
    endif
    !
    !
    ! The profile is written as the sum
    ! 	f(Ai,Ni,Vi,Hi)
    ! where Vi = VV.vi, Ai = AA.ai, Ni = NN+ni and Hi = HH*hi
    dograd = iflag.eq.minuit_gra_flag
    nline = max(obs%par%nline,1)
    allocate(repars(nparline,nline),regrad(nparline,nline),scagrad(nparline))
    !
    areasca = pars(iarea)
    freqoff = pars(ifreq)
    fwzlsca = pars(ifwzl)
    hornsca = pars(ihorn)
    scagrad(iarea) = 0.0
    scagrad(ifreq) = 0.0
    scagrad(ifwzl) = 0.0
    scagrad(ihorn) = 0.0
    do iline=1,nline
       ipar = iline*nparline
       repars(iarea,iline) = pars(ipar+iarea)*areasca
       regrad(iarea,iline) = 0.0
       repars(ifreq,iline) = pars(ipar+ifreq)+freqoff
       regrad(ifreq,iline) = 0.0
       repars(ifwzl,iline) = pars(ipar+ifwzl)*fwzlsca
       regrad(ifwzl,iline) = 0.0
       repars(ihorn,iline) = pars(ipar+ihorn)*hornsca
       regrad(ihorn,iline) = 0.0
    enddo
    !
    chi2 = 0.0
    eps = abs(obs%spec%f(obs%ifirst+1)-obs%spec%f(obs%ifirst))
    do ichan=obs%ifirst, obs%ilast
       if (obs%wfit(ichan).ne.0) then
          xfreq = obs%spec%f(ichan)
          sumpred = 0.0
          do iline=1,nline
             call cubefit_function_spectral_shell_one (xfreq,eps,repars(:,iline),dograd,sumpred,regrad(:,iline))
          enddo
          diff = sumpred - obs%spec%t(ichan)
          chi2 = chi2 + diff**2
          diff = 2.0*diff
          do iline = 1, nline
             do ipar = 1, nparline
                regrad(ipar,iline) = diff*regrad(ipar,iline)
                regrad(ipar,iline) = regrad(ipar,iline) + regrad(ipar,iline)
             enddo
             scagrad(iarea) = scagrad(iarea) + regrad(iarea,iline)*repars(iarea,iline)
             scagrad(ifreq) = scagrad(ifreq) + regrad(ifreq,iline)
             scagrad(ifwzl) = scagrad(ifwzl) + regrad(ifwzl,iline)*repars(ifwzl,iline)
             scagrad(ihorn) = scagrad(ihorn) + regrad(ihorn,iline)*repars(ihorn,iline)
          enddo
       endif
    enddo
    !
    grad(iarea) = scagrad(iarea)
    grad(ifreq) = scagrad(ifreq)
    grad(ifwzl) = scagrad(ifwzl)
    grad(ihorn) = scagrad(ihorn)
    !
    do iline=1,nline
       ipar = iline*nparline
       grad(ipar+iarea) = regrad(iarea,iline)*areasca
       grad(ipar+ifreq) = regrad(ifreq,iline)
       grad(ipar+ifwzl) = regrad(ifwzl,iline)*fwzlsca
       grad(ipar+ihorn) = regrad(ihorn,iline)*hornsca
    enddo
  end subroutine cubefit_function_spectral_shell_minimize
  !
  subroutine cubefit_function_spectral_shell_one(xfreq,eps,pars,dograd,pred,grad)
    use cubetools_nan
    !----------------------------------------------------------------------
    ! Computes the contribution of a Shell-like profile in the current
    ! channel to the square sum, and to the gradients relative to the
    ! parameters also if DOGRAD is .true.. Highly optimised ?
    !----------------------------------------------------------------------
    real(kind=coor_k), intent(in)    :: xfreq          ! Abscissa
    real(kind=coor_k), intent(in)    :: eps            ! Freq resolution
    real(kind=para_k), intent(in)    :: pars(nparline) ! Input parameters
    logical,           intent(in)    :: dograd         ! Compute contribution to gradients
    real(kind=sign_k), intent(inout) :: pred           ! Output value of Shell function
    real(kind=grad_k), intent(out)   :: grad(nparline) ! Gradients
    ! logical,      intent(out) :: error   ! Logical error flag -> deactivated, no way to signal error
    ! Local
    real(kind=4) :: predloc,area,voff,fwzl,horn,arg,aarg,arg0,arg1,arg2,arg3,arg4
    integer(kind=npar_k) :: ipar
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>ONE'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    area = pars(iarea)
    voff = pars(ifreq)-xfreq
    fwzl = pars(ifwzl)
    horn = pars(ihorn)
    if (fwzl.eq.0.or.area.eq.0) then
       call cubefit_message(fitseve%others,rname,'zero valued area or width')
       ! Tell minuit this is a bad choice
       pred    = gr4nan
       grad(:) = pred
       return
    endif
    arg  = voff/fwzl
    arg0 = 1.-0.5*eps/fwzl
    arg1 = 1.+0.5*eps/fwzl
    aarg = abs(arg)
    if (aarg .lt. arg0) then
       arg2 = arg**2
       predloc = area*1.5/fwzl/(3.+horn)*(1.+horn*arg2)
       if (dograd) then
          arg3        =  1./(1.+horn*arg2)
          arg4        =  arg2*arg3
          grad(iarea) =  predloc/area
          grad(ifreq) = -predloc*arg3*2.*horn*arg/fwzl
          grad(ifwzl) = -predloc/fwzl*(1.+2.*horn*arg4)
          grad(ihorn) =  predloc*(-1./(3.+horn)+arg4)
       endif
    elseif (aarg .lt. arg1) then
       arg2 = arg0**2
       predloc   = area*1.5/fwzl/(3.+horn)*(1.+horn*arg2)*(aarg-arg1)/(arg0-arg1)
       if (dograd) then
          grad(iarea) =  predloc/area
          grad(ifreq) = -predloc/(aarg-arg1)/fwzl
          if (arg.le.0) grad(ifreq) = -grad(ifreq)
          grad(ifwzl) = -predloc/fwzl*(1.-1./(arg1-aarg)-2.*horn*arg0*(1.-arg0)/(1.+horn*arg2))
          grad(ihorn) =  predloc*(-1./(3.+horn)+arg2/(1.+horn*arg2))
       endif
    else
       predloc = 0.
       if (dograd) then
          do ipar=1,nparline
             grad(ipar) = 0.
          enddo
       endif
    endif
    pred = pred + predloc
  end subroutine cubefit_function_spectral_shell_one
  !
  subroutine cubefit_function_spectral_shell_extract(minuit,obs,par,error)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    type(fit_minuit_t),    intent(inout) :: minuit
    type(spectral_obs_t),  intent(inout) :: obs
    type(spectral_pars_t), intent(inout) :: par
    logical,               intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    integer(kind=npar_k) :: ipar
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>EXTRACT'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    ! Update Parameters and errors
    ipar=0
    do iline=1,max(par%nline,1)
       par%pars(ipar+iarea)=minuit%u(ipar+iarea+nparline)*minuit%u(iarea)
       if (iline.eq.par%leaders(iarea)) then
          par%errs(ipar+iarea) = minuit%werr(iarea)
       else
          par%errs(ipar+iarea) = minuit%werr(ipar+iarea+nparline)
       endif
       par%pars(ipar+ifreq)=minuit%u(ipar+ifreq+nparline)+minuit%u(ifreq)
       if (iline.eq.par%leaders(ifreq)) then
          par%errs(ipar+ifreq) = minuit%werr(ifreq)
       else
          par%errs(ipar+ifreq) = minuit%werr(ipar+ifreq+nparline)
       endif
       par%pars(ipar+ifwzl)=minuit%u(ipar+ifwzl+nparline)*minuit%u(ifwzl)
       if (iline.eq.par%leaders(ifwzl)) then
          par%errs(ipar+ifwzl) = minuit%werr(ifwzl)
       else
          par%errs(ipar+ifwzl) = minuit%werr(ipar+ifwzl+nparline)
       endif
       par%pars(ipar+ihorn)=minuit%u(ipar+ihorn+nparline)*minuit%u(ihorn)
       if (iline.eq.par%leaders(ihorn)) then
          par%errs(ipar+ihorn) = minuit%werr(ihorn)
       else
          par%errs(ipar+ihorn) = minuit%werr(ipar+ihorn+nparline)
       endif
       ipar=ipar+nparline
    enddo
  end subroutine cubefit_function_spectral_shell_extract
  !
  subroutine cubefit_function_spectral_shell_residuals(obs,spec,error)
    use cubemain_spectrum_real
    !------------------------------------------------------------------------
    ! Compute the residuals of a shell fit
    !------------------------------------------------------------------------
    type(spectral_obs_t), intent(inout) :: obs          
    type(spectrum_t),     intent(inout) :: spec         
    logical,              intent(inout) :: error
    !
    integer(kind=chan_k) :: ichan
    real(kind=coor_k) :: xfreq
    real(kind=sign_k) :: pred
    character(len=*), parameter :: rname = "SPECTRAL>SHELL>RESIDUALS"
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    do ichan=1, spec%n
       xfreq = obs%spec%f(ichan)
       pred = cubefit_function_spectral_shell_profile(obs,xfreq,0)
       spec%t(ichan) = obs%spec%t(ichan) - pred
    end do
  end subroutine cubefit_function_spectral_shell_residuals
  !
  subroutine cubefit_function_spectral_shell_doprofile(iline,obs,spec,error)
    use cubemain_spectrum_real
    !------------------------------------------------------------------------
    ! Compute the doprofile of a shell fit
    !------------------------------------------------------------------------
    integer(kind=line_k), intent(in)    :: iline
    type(spectral_obs_t), intent(inout) :: obs          
    type(spectrum_t),     intent(inout) :: spec         
    logical,              intent(inout) :: error
    !
    integer(kind=chan_k) :: ichan
    real(kind=coor_k) :: xfreq
    real(kind=sign_k) :: pred
    character(len=*), parameter :: rname = "SPECTRAL>SHELL>DOPROFILE"
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    do ichan=1, spec%n
       xfreq = obs%spec%f(ichan)
       pred = cubefit_function_spectral_shell_profile(obs,xfreq,iline)
       spec%t(ichan) = pred
    end do
  end subroutine cubefit_function_spectral_shell_doprofile
  !
  subroutine cubefit_function_spectral_shell_user2par(flag,pars,par,error)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    integer(kind=flag_k),  intent(in)    :: flag(:)
    real(kind=para_k),     intent(in)    :: pars(:)
    type(spectral_pars_t), intent(inout) :: par
    logical,               intent(inout) :: error
    !
    integer(kind=npar_k) :: ipar,jpar
    integer(kind=line_k) :: iline
    integer(kind=4) :: ifatal
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>USER2PAR'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    par%leaders(:) = 0
    par%flag(:,:)  = 0
    par%errs(:)    = 0.
    !
    jpar = 1
    do iline=1,par%nline
       do ipar=1,nparline
          par%flag(iline,ipar) = flag(jpar)
          par%pars(jpar)       = pars(jpar)
          jpar=jpar+1
       enddo ! ipar
    enddo ! iline
    !
    ifatal = 0
    call par%check_line(iarea,error)
    if (error) ifatal=ifatal+1
    call par%check_line(ifreq,error)
    if (error) ifatal=ifatal+1
    call par%check_line(ifwzl,error)
    if (error) ifatal=ifatal+1
    call par%check_line(ihorn,error)
    if (error) ifatal=ifatal+1
    !
    if (par%leaders(iarea).ne.0 .and. par%leaders(ifwzl).ne.0 .and.&
         & par%leaders(iarea).ne.par%leaders(ifwzl)) ifatal=ifatal+1
    if (par%leaders(iarea).ne.0 .and. par%leaders(ihorn).ne.0 .and.&
         & par%leaders(iarea).ne.par%leaders(ihorn)) ifatal=ifatal+1
    if (par%leaders(iarea).ne.0 .and. par%leaders(ifreq).ne.0 .and.&
         & par%leaders(iarea).ne.par%leaders(ifreq)) ifatal=ifatal+1
    if (par%leaders(ifwzl).ne.0 .and. par%leaders(ifreq).ne.0 .and.&
         & par%leaders(ifwzl).ne.par%leaders(ifreq)) ifatal=ifatal+1
    if (par%leaders(ihorn).ne.0 .and. par%leaders(ifreq).ne.0 .and.&
         & par%leaders(ihorn).ne.par%leaders(ifreq)) ifatal=ifatal+1
    if (par%leaders(ihorn).ne.0 .and. par%leaders(ifwzl).ne.0 .and.&
         & par%leaders(ihorn).ne.par%leaders(ifwzl)) ifatal=ifatal+1
    if (ifatal.ne.0) then
       write(mess,'(i0,a)') ifatal,' Fatal Errors on Parameters'
       call cubefit_message(seve%e,rname,'Parameters flags are inconsistent')
       error = .true.
       return
    endif
  end subroutine cubefit_function_spectral_shell_user2par
  !
  subroutine cubefit_function_spectral_shell_par2spec(par,spec,error)
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(spectral_pars_t), intent(in)    :: par
    type(spectrum_t),      intent(inout) :: spec
    logical,               intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    integer(kind=chan_k) :: ichan
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>PAR2SPEC'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    ichan = spec_ndaps
    do iline=1,max(par%nline,1)
       ichan = ichan+1
       spec%t(ichan) = par%flag(iline,iarea) 
       ichan = ichan+1
       spec%t(ichan) = par%pars((iline-1)*nparline+iarea)
       ichan = ichan+1
       spec%t(ichan) = par%errs((iline-1)*nparline+iarea)
       ichan = ichan+1
       spec%t(ichan) = par%flag(iline,ifreq)
       ichan = ichan+1
       spec%t(ichan) = par%pars((iline-1)*nparline+ifreq)
       ichan = ichan+1
       spec%t(ichan) = par%errs((iline-1)*nparline+ifreq)
       ichan = ichan+1
       spec%t(ichan) = par%flag(iline,ifwzl)
       ichan = ichan+1
       spec%t(ichan) = par%pars((iline-1)*nparline+ifwzl)
       ichan = ichan+1
       spec%t(ichan) = par%errs((iline-1)*nparline+ifwzl)
       ichan = ichan+1
       spec%t(ichan) = par%flag(iline,ihorn)
       ichan = ichan+1
       spec%t(ichan) = par%pars((iline-1)*nparline+ihorn)
       ichan = ichan+1
       spec%t(ichan) = par%errs((iline-1)*nparline+ihorn)
    enddo
  end subroutine cubefit_function_spectral_shell_par2spec
  !
  subroutine cubefit_function_spectral_shell_spec2par(spec,par,error)
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(spectrum_t),      intent(in)    :: spec
    type(spectral_pars_t), intent(inout) :: par
    logical,               intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    integer(kind=chan_k) :: ichan
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>SPEC2PAR'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    ichan = spec_ndaps
    do iline=1,max(par%nline,1)
       ichan = ichan+1
       par%flag(iline,iarea)              = nint(spec%t(ichan),flag_k)
       ichan = ichan+1                     
       par%pars((iline-1)*nparline+iarea) = spec%t(ichan) 
       ichan = ichan+1                     
       par%errs((iline-1)*nparline+iarea) = spec%t(ichan) 
       ichan = ichan+1                     
       par%flag(iline,ifreq)              = nint(spec%t(ichan),flag_k)
       ichan = ichan+1                     
       par%pars((iline-1)*nparline+ifreq) = spec%t(ichan) 
       ichan = ichan+1                     
       par%errs((iline-1)*nparline+ifreq) = spec%t(ichan) 
       ichan = ichan+1                     
       par%flag(iline,ifwzl)              = nint(spec%t(ichan),flag_k)
       ichan = ichan+1                     
       par%pars((iline-1)*nparline+ifwzl) = spec%t(ichan) 
       ichan = ichan+1                     
       par%errs((iline-1)*nparline+ifwzl) = spec%t(ichan)
       ichan = ichan+1                     
       par%flag(iline,ihorn)              = nint(spec%t(ichan),flag_k)
       ichan = ichan+1                     
       par%pars((iline-1)*nparline+ihorn) = spec%t(ichan) 
       ichan = ichan+1                     
       par%errs((iline-1)*nparline+ihorn) = spec%t(ichan) 
    enddo
  end subroutine cubefit_function_spectral_shell_spec2par
  !
  subroutine cubefit_function_spectral_shell_iterate(par,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(spectral_pars_t), intent(inout) :: par
    logical,               intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    integer(kind=npar_k) :: ipar
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>ITERATE'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    ipar = 0
    do iline=1,par%nline
       if (par%flag(iline,ipar+iarea).eq.3) &
            par%pars(ipar+iarea)=par%pars(ipar+iarea)/par%pars(nparline*(par%leaders(iarea)-1)+iarea)
       if (par%flag(iline,ipar+ifreq).eq.3) &
            par%pars(ipar+ifreq)=par%pars(ipar+ifreq)/par%pars(nparline*(par%leaders(ifreq)-1)+ifreq)
       if (par%flag(iline,ipar+ifwzl).eq.3) &
            par%pars(ipar+ifwzl)=par%pars(ipar+ifwzl)/par%pars(nparline*(par%leaders(ifwzl)-1)+ifwzl)
       if (par%flag(iline,ipar+ihorn).eq.3) then
          if (par%pars(nparline*(par%leaders(ihorn)-1)+ihorn).eq.0) then
             par%flag(iline,ipar+ihorn) = 1.0
          else
             par%pars(ipar+ifwzl)=par%pars(ipar+ifwzl)/par%pars(nparline*(par%leaders(ihorn)-1)+ihorn)
          endif
       endif
       ipar = ipar+nparline
    enddo
  end subroutine cubefit_function_spectral_shell_iterate
  !
  subroutine cubefit_function_spectral_shell_wind2par(obs,wind,par,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(spectral_obs_t),  intent(in)    :: obs
    integer(kind=chan_k),  intent(in)    :: wind(:)
    type(spectral_pars_t), intent(inout) :: par
    logical,               intent(inout) :: error
    !
    integer(kind=chan_k) :: first,last
    integer(kind=line_k) :: iline
    real(kind=para_k) :: area,freq,fwzl
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>WIND2PAR'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    par%errs(:)     = 0
    par%leaders(:)  = 0
    do iline=1,par%nline
       first = wind(2*iline-1)
       last  = wind(2*iline)
       par%flag(iline,:)   = 0
       !
       call obs%est_shell(first,last,area,freq,fwzl,error)
       if (error) return
       if (fwzl.lt.obs%deltaf) fwzl = obs%deltaf
       par%pars((iline-1)*nparline+iarea) = area 
       par%pars((iline-1)*nparline+ifreq) = freq 
       par%pars((iline-1)*nparline+ifwzl) = fwzl
       par%pars((iline-1)*nparline+ihorn) = 0
    end do
  end subroutine cubefit_function_spectral_shell_wind2par
  !
  !----------------------------------------------------------------------
  !
  function cubefit_function_spectral_shell_profile(obs,xfreq,iline) result(shell)
    !----------------------------------------------------------------------
    ! Compute the value of a shell or a sum of shells at xfreq
    !----------------------------------------------------------------------
    type(spectral_obs_t),   intent(in)    :: obs      ! Observation
    real(kind=coor_k),      intent(in)    :: xfreq    ! Coordinate to compute value
    integer(kind=line_K),   intent(in)    :: iline    ! Which shell is to be computed
    !
    real(kind=sign_k) :: shell
    !
    integer(kind=line_k) :: ifirst,ilast,jline
    real(kind=para_k) :: locpars(nparline)
    real(kind=para_k) :: nullgrad(nparline)
    logical :: dograd
    !
    !
    dograd = .false.
    shell=0.
    if (iline.eq.0) then
       ifirst = 1
       ilast = max(obs%par%nline,1)
    else
       ifirst = iline
       ilast  = iline
    endif
    !
    do jline = ifirst,ilast
       locpars(:) = obs%par%pars((jline-1)*nparline+1:jline*nparline)
       if (locpars(iarea).ne.0 .and. locpars(ifwzl).ne.0) then
          call cubefit_function_spectral_shell_one(xfreq,obs%deltaf,locpars,dograd,shell,nullgrad)
       endif
    enddo
  end function cubefit_function_spectral_shell_profile
  !
  subroutine cubefit_function_spectral_shell_flags(ipar,lineflag,parflag,error)
    use cubedag_allflags
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    integer(kind=npar_k), intent(in)    :: ipar
    type(flag_t),         intent(out)   :: lineflag
    type(flag_t),         intent(out)   :: parflag
    logical,              intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>FLAGS'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    iline  = ipar/nparline+1
    select case(mod(ipar,nparline))
    case(1)
       parflag = flag_area
    case(2)
       parflag = flag_frequency
    case(3)
       parflag = flag_fwzl
    case(0)
       parflag = flag_horn
       iline  = iline-1
    end select
    call cubefit_line2flag(iline,lineflag,error)
    if (error)  return
  end subroutine cubefit_function_spectral_shell_flags
  !
  subroutine cubefit_function_spectral_shell_units(ipar,unit,error)
    use cubetools_unit
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    integer(kind=npar_k), intent(in)    :: ipar
    character(len=*),     intent(out)   :: unit
    logical,              intent(inout) :: error
    !
    integer(kind=line_k) :: iline
    character(len=*), parameter :: rname = 'SPECTRAL>SHELL>UNITS'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    iline  = ipar/nparline+1
    select case(mod(ipar,nparline))
    case(1)
       unit = strg_add//unit_freq_name(3)
    case(2)
       unit = unit_freq_name(3) ! MHz
    case(3)
       unit = unit_freq_name(3) ! MHz
    case(0)
       unit = '---' ! No unit for horn ratio parameter
    end select
  end subroutine cubefit_function_spectral_shell_units
  !
  function cubefit_function_spectral_shell_npar(nline) result(npar)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    integer(kind=line_k), intent(in) :: nline
    !
    integer(kind=npar_k) :: npar
    !
    npar = nparline*nline
  end function cubefit_function_spectral_shell_npar
end module cubefit_function_spectral_shell
