
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
extern int optind;
extern char *optarg;
extern int getopt (int argc, char *const *argv, const char *shortopts);
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_SIGINFO_H
#include <siginfo.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "version.h"
#include "sigfifo.h"
#include "sigformat.h"
#include "pidfile.h"
#include "tagdb.h"
#include "gensig.h"

#ifdef HAVE_PRINTF_H
#include "sigparams.h"
#endif

void showsyntax()
{
#ifdef HAVE_GETOPT_LONG
   printf("%s\n"
"Usage: gensig [-dhkrsv] [<long options>] [tagfile [tagfile [...]]]\n"
"  -a, --add             Add specified tagfiles to search list\n"
"                        (otherwise, *only* use files specified)\n"
"  -d, --debug           Debug mode (runs in foreground, prints actions)\n"
"  -f, --nofile          Don't leave behind a regular file when we quit\n"
"  -h, --help		Print this message\n"
"  -k, --kill            Kill running gensig (nothing else)\n"
"  -m, --format          File to read for format of signature (printf-style)\n"
"  -o, --once            Once, write a signature to stdout & quit\n"
"  -q, --quiet           Don't complain if there's aready a gensig running\n"
"                        (Just fail silently)\n"
"  -r, --restart         Kill running gensig (if one exists) and restart\n"
"  -s, --sigfile         Specify signature file (FIFO) to use\n"
"  -v, --version		Print version information\n"
"", version);
#else
   printf("%s\n"
"Usage: gensig [-dhkrsv] [tagfile [tagfile [...]]]\n"
"  -a   Add specified tagfiles to search list\n"
"       (otherwise, *only* use files specified)\n"
"  -d   debug, run in foreground, print actions\n"
"  -f   DON'T leave behind a regular file when we quit\n"
"  -h   Print this message\n"
"  -k   Kill running gensig (nothing else)\n"
"  -m   File to read for format of signature (printf-style) (requires a value)\n"
"  -o   Once, write a signature to stdout & exit\n"
"  -q   Don't complain if there's aready a gensig running\n"
"       (Just fail silently)\n"
"  -r   Kill running gensig (if one exists) and restart\n"
"  -s   signature file (FIFO) to use (requires a value)\n"
"  -v   Print version information\n"
"", version);
#endif
}

/* globals & default values */
int keepgoing = 1;
int killold = 0;
int quiet = 0;
int leavefile = 1;
int default_tagfiles_also = 0;
int debug = 0;
char fifoname[BUFSIZ] = "";
char formatname[BUFSIZ] = "";
int once_and_exit = 0;

char *locate_fifo()
{
   if (*fifoname != '\0') return fifoname;
   sprintf(fifoname, "%s/.signature", getpwuid(getuid())->pw_dir);
   return fifoname;
}

/* SIGNAL HANDLERS */

RETSIGTYPE quit_signal(int signum) {
   if (debug)
   {
#ifdef HAVE_PSIGNAL
      psignal(signum, "quit_signal");
#else
# ifdef HAVE_STRSIGNAL
      if (debug) fprintf(stderr, "quit_signal: %s\n", strsignal(signum));
# else
#  ifdef SYS_SIGLIST_DECLARED
      if (debug) fprintf(stderr, "quit_signal: %s\n", sys_siglist[signum]);
#  else
      if (debug) fprintf(stderr, "quit_signal: signal %d\n", signum);
#  endif
# endif
#endif
   }

   keepgoing = 0;
   reset_fifo(locate_fifo(), leavefile);
   close_tagdb();
   clean_pidfile();
   exit(0);
}
RETSIGTYPE debug_signal(int signum)
{
#ifdef HAVE_PSIGNAL
   psignal(signum, "debug_signal");
#else
# ifdef HAVE_STRSIGNAL
   if (debug) fprintf(stderr, "debug_signal: %s\n", strsignal(signum));
# else
#  ifdef SYS_SIGLIST_DECLARED
   if (debug) fprintf(stderr, "debug_signal: %s\n", sys_siglist[signum]);
#  else
   if (debug) fprintf(stderr, "debug_signal: signal %d\n", signum);
#  endif
# endif
#endif
   return;
}

/* end signal handlers */

int main(int argc, char **argv)
{
   int tf_count = 0;
   int sigfifo_fd = -1;	/* The .signature file descriptor */
   char *tagline;
   char signature[BUFSIZ];
   int siglen;
   int rv;
   int opt_char;
   char opt_string[] = "adfhkm:oqrs:v";
#ifdef HAVE_GETOPT_LONG
   int opt_index = 0;
   static struct option long_options[] = {
		/* name, hasarg, flag, val */
		{ "add", no_argument, NULL, 'a' },
		{ "debug", no_argument, NULL, 'd' },
		{ "nofile", no_argument, NULL, 'f' },
		{ "help", no_argument, NULL, 'h' },
		{ "kill", no_argument, NULL, 'k' },
		{ "format", required_argument, NULL, 'm' },
		{ "once", no_argument, NULL, 'o' },
		{ "quiet", no_argument, NULL, 'q' },
		{ "restart", no_argument, NULL, 'r' },
		{ "version", no_argument, NULL, 'v' },
		{ "sigfile", required_argument, NULL, 's' },
		{ 0, 0, 0, 0 }
	};
#endif

   while ((opt_char =
#ifdef HAVE_GETOPT_LONG
	getopt_long(argc, argv, opt_string, long_options, &opt_index)
#else
	getopt(argc, argv, opt_string)
#endif
	) != EOF)
   {
      switch (opt_char)
      {
#ifdef HAVE_GETOPT_LONG
         /* this doesn't really do anything yet
		-- I map everything to the chars below */
         case 0:
            printf("option %s", long_options[opt_index].name);
            if (optarg) printf(" with arg %s", optarg);
            printf("\n");
            break;
#endif
         case 'a':
            default_tagfiles_also = 1;
            break;
         case 'd':
            fprintf(stderr, "Debug enabled: running in foreground.\n");
            debug = 1;
            break;
         case 'f':
            leavefile = 0;
            break;
         case 'h':
            showsyntax();
            exit(0);
            break;
         case 'k':
            killold = 1;
            keepgoing = 0;
            break;
         case 'm':
            /* init_sigformat(optarg); */
            strncpy(formatname, optarg, BUFSIZ);
            break;
         case 'o':
            once_and_exit = 1;
            break;
         case 'q':
            quiet = 1;
            break;
         case 'r':
            killold = 1;
            break;
         case 's':
            strncpy(fifoname, optarg, BUFSIZ);
            break;
         case 'v':
            printf("%s\n", version);
            exit(0);
            break;
         case '?':
            showsyntax();
            exit(0);
            break;
         default:
            fprintf(stderr,
		"?? getopt returned character code 0x%x ??\n", opt_char);
      }
   }

   init_tagdb();

   if (optind < argc)
   {
      while (optind < argc)
      {
         tf_count += open_tagfile(argv[optind]);
         optind++;
      }

   }

   if (default_tagfiles_also || (tf_count == 0))
   {
      tf_count += open_default_tagfiles();
   }

   /* fork if we're not debugging... */

   if (!debug && !once_and_exit && keepgoing)
   {
      if ((rv = fork()))
      {
         if (rv < 0)
         perror("fork");
         exit(1);
      }
      setsid();
   }

   if ( (*fifoname) == '\0' )
   {
      sprintf(fifoname, "%s/.signature", getpwuid(getuid())->pw_dir);
   }

   if ( (*formatname) == '\0' )
   {
      sprintf(formatname, "%s.tmpl", fifoname);
   }

   init_sigformat(formatname);

#ifdef HAVE_PRINTF_H
   init_sigparams();
#endif

   if (once_and_exit)
   {
      tagline = get_random_tagline(0);
      printf(get_sigformat(), tagline?tagline:"");

      keepgoing = 0;
   }
   else
   {
      rv = make_pidfile(fifoname);

      if (rv < 0)
      {
         fprintf(stderr, "Error creating pidfile, quitting...\n");
         exit(1);
      }
      else if (rv != getpid())
      {
         /* another gensig running... what to do? */

         if (quiet)
         {
            exit(0);	/* exit quietly */
         }
         else if (killold)
         {
            kill(rv, SIGINT);

            if (keepgoing)	/* try one more time only */
            {
               sleep(1);

               rv = make_pidfile(NULL);

               if (rv < 0)
               {
                  fprintf(stderr, "Error creating pidfile, quitting...\n");
                  exit(1);
               }
               else if (rv != getpid())
               {
                  fprintf(stderr, "Other gensig (%d) wouldn't go away!\n", rv);
                  exit(0);
               }
            }
         }
         else
         {
            fprintf(stderr, "Another gensig is running as pid %d\n", rv);
            exit(0);
         }
      }
      else	/* rv == getpid() */ if (!keepgoing)
      {
         clean_pidfile();
      }
   }

   /* if we're not supposed to do anything, just exit here... */
   if (!keepgoing)
   {
      exit(0);
   }

   /* close unused files */

   if (debug == 0)
   {
      close(0);
      close(1);
      close(2);
   }

   signal(SIGHUP, debug_signal);
   signal(SIGINT, quit_signal);
   signal(SIGTERM, quit_signal);
   signal(SIGPIPE, debug_signal);
   signal(SIGALRM, debug_signal);

   if (setup_fifo(fifoname))
   {
      quit_signal(0);
   }

   /* Main loop. Basically, anything that should cause the
    * process to exit will set 'keepgoing' to zero.
    * This applies mainly to signal handlers, since the stuff
    * inside the loop can just 'break'.
    */
   while (keepgoing)
   {
      if (debug) fprintf(stderr, "Entering open\n");
      if ((sigfifo_fd = open(locate_fifo(), O_WRONLY)) < 0)
      {
         if (errno == EINTR)
         {
            /* interrupted -- we'll try again */
            if (debug) fprintf(stderr, "Open interrupted...\n");
         }
         else
         {
            if (debug) fprintf(stderr, "Can't open %s\n", locate_fifo());
            perror("open fifo");
            keepgoing = 0;
         }
      }
      if (debug) fprintf(stderr, "Open OK\n");

      if ((sigfifo_fd >= 0) && keepgoing)
      {

#ifdef HAVE_PRINTF_H
	 foreach_sigparams();
#endif

	 if (debug) fprintf(stderr, "Parameter status computation done");

         tagline = get_random_tagline(1);

         if (debug) fprintf(stderr, "Got tagline, writing signature\n");

         siglen = sprintf(signature, get_sigformat(), tagline?tagline:"");

         write(sigfifo_fd, signature, siglen);

         if (debug) fprintf(stderr, "Signature written\n");

         close(sigfifo_fd);
         usleep(200000);
      }
   }

   if (debug) fprintf(stderr, "Cleaning up and exiting...\n");
   /* Clean up and exit */
   close(sigfifo_fd);
   reset_fifo(locate_fifo(), leavefile);
   close_tagdb();
   clean_pidfile();
   exit(0);

}

