/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "tableIO_hdf.hh"
#include "table.hh"
#include <cstring>
#include <iostream>

#ifndef H5_NO_NAMESPACE
using namespace H5;
#endif

static StrType str64(PredType::C_S1);
static StrType str4(PredType::C_S1);

namespace table {

    //  Enumerator of supported data types.
    enum xdata_type {
	x_string,
	x_ilwdch,
	x_ilwduc,
	x_short,
	x_int,
	x_float,
	x_double
    };

    //  Data type table element.
    struct ttype {
	const char* _id;
	int         _align;
	int         _length;
	const H5SPACE::AtomType* _pred;
    };

    //  Table of supported data types and their translation to hdf5 atoms
    //  Note that the order of table elements must follow the xdata_type
    //  enumerator.
    ttype typetab[] = {
	{"string", 1, 64, &str64}, 
	{"ilwd:char", 1, 64, &str64}, 
	{"ilwd:char_u", 1, 64, &str64}, 
	{"int_2", 2, 2, &PredType::NATIVE_SHORT}, 
	{"int_4s", 4, 4, &PredType::NATIVE_INT}, 
	{"real_4", 4, 4, &PredType::NATIVE_FLOAT},
	{"real_8", 8, 8, &PredType::NATIVE_DOUBLE},
	{0, 0, 0, &PredType::NATIVE_CHAR}
    };

    //----------------------------------  Pointer union
    union typ_ptr {
	char*   c;
	short*  s;
	int*    i;
	float*  f;
	double* d;
    };

} //namespace table

using namespace table;
using namespace std;  

//======================================  tableIO_hdf Constructor
tableIO_hdf::tableIO_hdf(void) {
    str64.setSize(64);
    str64.setStrpad(H5T_STR_NULLTERM);
    str4.setSize(4);
    str4.setStrpad(H5T_STR_NULLTERM);
}

//======================================  tableIO_hdf destructor
tableIO_hdf::~tableIO_hdf(void) {
}

//======================================  inline function to force alignment
inline size_t
force_align(size_t off, size_t align) {
    if (!align) return off;
    return (off + align - 1) & (~align + 1);
}

//======================================  calculate the hdf compound structure.
void 
tableIO_hdf::plan_data(const Table& tab) {
    //----------------------------------  Clear out any data alread in the plan
    size_t nCol = tab.size();
    _plan.clear();
    _plan.reserve(nCol);
    _plan_size=0;

    //----------------------------------  Loop over columns: add one plan 
    //                                    entry per column
    size_t max_align = 0;
    for (size_t i=0; i<nCol; i++) {
	const Column& col_i(tab[i]);
	layout col_layout;
	string col_type = col_i.getXType();
	int typ=0;
	while (typetab[typ]._id) {
	    if (col_type == typetab[typ]._id) break;
	    typ++;
	}
	if (!typetab[typ]._id) {
	    throw runtime_error(
		       string("tableIO_hdf: Unknown external data type: ") +
		       col_type);
	}
	col_layout._name   = col_i.refTitle();
	col_layout._type   = typ;
	col_layout._length = typetab[typ]._length;
	size_t align = typetab[typ]._align;
	_plan_size = force_align(_plan_size, align);
	if (max_align < align) max_align = align;
	col_layout._offset = _plan_size;
	_plan_size        += col_layout._length;
	_plan.push_back(col_layout);
    }
    _plan_size = force_align(_plan_size, max_align);

    //----------------------------------  Construct an hdf5 compound data-type
    if (debug()) cerr << "Construct table data structure, size=" 
		      << _plan_size << endl;
    _dataType = new CompType(_plan_size);
    for (size_t i=0; i<nCol; i++) {
	int dtype = _plan[i]._type;
	if (debug()) {
	    cerr << "Add data element, name=" << _plan[i]._name
		 << ", type=" << typetab[dtype]._id 
		 << ", offset=" << _plan[i]._offset << endl;
	}
	_dataType->insertMember(_plan[i]._name, _plan[i]._offset, 
				*typetab[dtype]._pred);
    }
}

//======================================  Read an hdf5 table
void 
tableIO_hdf::readTable(Table& tabl, const string& path, const string& name) {

    hsize_t nCol = tabl.size();
    if (!nCol) {
	cerr << "hdf file reading must have a pre-defined set of columns."
	     << endl;
	cerr << "Use addColumn command to add columns to read." << endl;
	throw runtime_error("tableIO_hdf::readTable - No columns defined");
    }

    //----------------------------------  Plan data columns
    plan_data(tabl);

    //----------------------------------  Open the file
    H5File file(path, H5F_ACC_RDONLY);

    //----------------------------------  Open a data set
    DataSet dataset = file.openDataSet(name);

    //----------------------------------  Get the number of entries.
    hsize_t dim;
    DataSpace dspace = dataset.getSpace();
    int nDim = dspace.getSimpleExtentDims( &dim, NULL);
    if (debug()) {
	cerr << "File: " << path << " table: " << name << " contains "
	     << nDim << " dimensions and " << dim << " elements. " << endl;
    }
    if (nDim != 1) {
	string msg("tableIO_hdf::readTable - Multi-dimensional table: ");
	msg += name;
	throw runtime_error(msg);
    }

    //----------------------------------  Allocate space, read in the data.
    if (!dim) return;
    char* pData = new char[dim*_plan_size];
    dataset.read(pData, *_dataType);
    for (hsize_t j=0; j<nCol; j++) {
	tabl[j].reserve(dim);
    }

    //----------------------------------  Copy it all to the table;
    typ_ptr pij;
    for (hsize_t i=0; i<dim; i++) {
	tabl.addRow(i, Cell(""));
	hsize_t roff = i * _plan_size;
	for (hsize_t j=0; j<nCol; j++) {
	    hsize_t typ = _plan[j]._type;
	    pij.c = pData + roff + _plan[j]._offset;
	    if (debug() > 2) {
		cerr << "data: " << _plan[j]._name << "[" << i << "] type: " 
		     << typetab[typ]._id << endl;
	    }

	    //--------------------------  Set the cell
	    switch (typ) {
	    case x_string:
	    case x_ilwdch:
	    case x_ilwduc:
		tabl[j][i] = Cell(pij.c);
		break;
	    case x_short:
		tabl[j][i] = Cell(*pij.s);
		break;
	    case x_int:
		tabl[j][i] = Cell(*pij.i);
		break;
	    case x_float:
		tabl[j][i] = Cell(*pij.f);
		break;
	    case x_double:
		tabl[j][i] = Cell(*pij.d);
		break;
	    }
	}
	if (debug() > 3) {
	    size_t k=tabl.getNRows() - 1;
	    cerr << "row " << k << ":";
	    for (size_t j=0; j<nCol; j++) cerr << " " << tabl[j][k].getString();
	    cerr << endl;
	}
    }
    delete[] pData;
}

//======================================  Write a table to an hdf5 file
void 
tableIO_hdf::writeTable(const Table& tabl, const string& path, 
			const string& name) {
    _tableName = name;

    //------------------------------------  Set up the data organization.
    plan_data(tabl);

    //------------------------------------  Fill array with data
    hsize_t nRows = tabl.getNRows();
    hsize_t nCols = tabl.size();
    char* pData = new char[nRows*_plan_size];
    typ_ptr pij;
    for (hsize_t i=0; i<nRows; i++) {
	hsize_t roff = i * _plan_size;
	for (hsize_t j=0; j<nCols; j++) {
	    hsize_t typ = _plan[j]._type;
	    pij.c = pData + roff + _plan[j]._offset;
	    switch (typ) {
	    case x_string:
	    case x_ilwdch:
	    case x_ilwduc:
		if (tabl[j][i].getString().size() < 64) {
		    strcpy(pij.c, tabl[j][i].getString().c_str());
		} else {
		    memcpy(pij.c, tabl[j][i].getString().c_str(), 63);
		    pij.c[63] = 0;
		}
		break;
	    case x_short:
		*pij.s = tabl[j][i].getNumeric();
		break;
	    case x_int:
		*pij.i = tabl[j][i].getNumeric();
		break;
	    case x_float:
		*pij.f = tabl[j][i].getNumeric();
		break;
	    case x_double:
		*pij.d = tabl[j][i].getNumeric();
		break;
	    }
	}
    }

    //------------------------------  Dump it all
    if (debug() > 1) {
	for (hsize_t i=0; i<nRows; i++) {
	    cout << "Row " << i << " {" << endl;
	    hsize_t roff = i * _plan_size;
	    for (hsize_t j=0; j<nCols; j++) {
		hsize_t typ = _plan[j]._type;
		pij.c = pData + roff + _plan[j]._offset;
		cout << "    " << _plan[j]._name << ":    ";
		switch (typ) {
		case x_string:
		case x_ilwdch:
		case x_ilwduc:
		    cout << pij.c;
		    break;
		case x_short:
		    cout << *pij.s;
		    break;
		case x_int:
		    cout << *pij.i;
		    break;
		case x_float:
		    cout << *pij.f;
		    break;
		case x_double:
		    cout << *pij.d;
		    break;
		}
		cerr << endl;
	    }
	    cerr << "}" << endl;
	}
    }

    //------------------------------------  Open the hdf5 file.
    H5File hd_file(path, H5F_ACC_TRUNC);

    //------------------------------------  Build the data space
    DataSpace dspace(1, &nRows);

    //-----------------------------------  Write table to the dataset;
    DataSet def_ds(hd_file.createDataSet(name, *_dataType, dspace));
    def_ds.write(pData, *_dataType );
    delete[] pData;
}
