//////////////////////////////////////////////////////////////////////////
//  									//
//  WindowIterator							//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include "events/WindowIterator.hh"
#include "events/Event.hh"


namespace events {

//______________________________________________________________________________
   BasicWindowIterator::BasicWindowIterator (
                     const ConstIterator& beg, 
                     const ConstIterator& end, 
                     int order,
                     const TimeWindow& window)
   : mInline (true), mWindow (window, order), mFirstTime (true),
   mPermMax (0), mPermI (0) 
   {
      mState.push_back (InputState (beg, end));
      InitWindow (); 
   }

//______________________________________________________________________________
   BasicWindowIterator::BasicWindowIterator (
                     const ConstIterator& beg1, 
                     const ConstIterator& end1, 
                     const ConstIterator& beg2, 
                     const ConstIterator& end2, 
                     const TimeWindow& window)
   : mInline (false), mWindow (window, 2), mFirstTime (true),
   mPermMax (0), mPermI (0) 
   {
      mState.push_back (InputState (beg1, end1));
      mState.push_back (InputState (beg2, end2));
      InitWindow (); 
   }

//______________________________________________________________________________
   BasicWindowIterator::BasicWindowIterator (
                     const ConstIterator& beg1, 
                     const ConstIterator& end1, 
                     const ConstIterator& beg2, 
                     const ConstIterator& end2, 
                     const ConstIterator& beg3, 
                     const ConstIterator& end3, 
                     const TimeWindow& window)
   : mInline (false), mWindow (window, 3), mFirstTime (true),
   mPermMax (0), mPermI (0) 
   {
      mState.push_back (InputState (beg1, end1));
      mState.push_back (InputState (beg2, end2));
      mState.push_back (InputState (beg3, end3));
      InitWindow (); 
   }

//______________________________________________________________________________
   BasicWindowIterator::BasicWindowIterator (
                     const InputStateList& state, 
                     const TimeWindow& window)
   : mInline (false), mWindow (window, state.size()), 
   mState (state), mFirstTime (true), mPermMax (0), mPermI (0) 
   {
      InitWindow (); 
   }

//______________________________________________________________________________
   void BasicWindowIterator::Increment ()
   {
      // inline
      if (mInline) {
         if ((mWindow.GetOrder() <= 0) || (mState.size() != 1) ||
            (mState[0].mCur == mState[0].mEnd)) {
            return;
         }
         // more permutations?
         if (++mPermI < mPermMax) {
            SetPermutation();
            return;
         }
         // find next event/coincidence
         do {
            // move current to next event
            if (mFirstTime) mFirstTime = false;
            else ++mState[0].mCur; // avoid ++ initially
            if (mState[0].mCur == mState[0].mEnd) { // reached the end
               mWindow.Clear();
               return;
            }
            // add new events to list
            Time start = mState[0].mCur->GetTime() + mWindow.GetOffset();
            Time stop = start + mWindow.GetWidth();
            while ((mState[0].mNext != mState[0].mEnd) && 
                  (mState[0].mNext->GetTime() < stop)) {
               mWindow.PushBack (const_cast<Event*>(&*mState[0].mNext));
               ++mState[0].mNext;
            }
            // remove old events from list
            while (!mWindow.Empty() && (mWindow.GetTimeFirst() < start)) {
               mWindow.PopFront();
            }
            // Determine how many permuations
            mPermMax = GetPermutations();
            mPermI = 0;
            if (mPermMax > 0) {
               SetPermutation();
            }
         } while (mPermMax == 0);
      }
      
      // multiple input streams
      else {
         if ((mWindow.GetOrder() <= 0) ||
            (mWindow.GetOrder() != (int)mState.size()) ||
            (mState[0].mCur == mState[0].mEnd)) {
            return;
         }
         // more permutations with current main event?
         if (!mFirstTime) {
            for (int i = mState.size() - 1; i > 0; --i) {
               ++mState[i].mCur;
               if (mState[i].mCur != mState[i].mNext) {
                  mWindow.SetCurrent 
                     (i, const_cast<Event*>(&*mState[i].mCur));
                  return;
               }
            // reset
               mState[i].mCur = mState[i].mFirst;
               mWindow.SetCurrent 
                  (i, const_cast<Event*> (&*mState[i].mCur));
            }
         }
         // look for next coincident event
         bool emptylist;
         do {
            // move current to next event
            if (mFirstTime) mFirstTime = false;
            else ++mState[0].mCur; // avoid ++ initially
            if (mState[0].mCur == mState[0].mEnd) { // reached the end
               mWindow.Clear();
               return;
            }
            mWindow.SetCurrent 
               (0, const_cast<Event*> (&*mState[0].mCur));
            // add new events to list
            Time start = mState[0].mCur->GetTime() + mWindow.GetOffset();
            Time stop = start + mWindow.GetWidth();
            while (true) {
               // get oldest event from all input streams
               int found = -1;
               for (int i = 0; i < (int)mState.size(); ++i) {
                  if (mState[i].mNext == mState[i].mEnd) 
                     continue;
                  if ((found == -1) || (mState[i].mNext->GetTime() < 
                                       mState[found].mNext->GetTime())) {
                     found = i;
                  }
               }
               // add to window
               if ((found >= 0) && 
                  (mState[found].mNext->GetTime() < stop)) {
                  mWindow.PushBack 
                     (const_cast<Event*>(&*mState[found].mNext));
                  ++mState[found].mNext;
               }
               else {
                  break; // done
               }
            };
            // remove old events from list
            while (!mWindow.Empty() && (mWindow.GetTimeFirst() < start)) {
               mWindow.PopFront();
            }
            // adjust first
            for (int i = 0; i < (int)mState.size(); ++i) {
               while ((mState[i].mFirst != mState[i].mEnd) &&
                     (mState[i].mFirst->GetTime() < start)) {
                  ++mState[i].mFirst;
               }
            }
            // set current in rest of event inputs
            emptylist = false;
            for (int i = 1; i < (int)mState.size(); ++i) {
               mState[i].mCur = mState[i].mFirst;
               if (mState[i].mCur == mState[i].mNext) {
                  emptylist = true;
               }
               else {
                  mWindow.SetCurrent 
                     (i, const_cast<Event*> (&*mState[i].mCur));
               }
            }
         } while (emptylist);
      }
   }

//______________________________________________________________________________
   void BasicWindowIterator::InitWindow ()
   {
      mFirstTime = true;
      mPermI = 0;
      mPermMax = -1;
      mWindow.Clear();
      bool emptylist = false;
      for (InputStateList::iterator i = mState.begin();
          i != mState.end(); ++i) {
         i->mCur = i->mBegin;
         i->mFirst = i->mBegin;
         i->mNext = i->mBegin;
         if (i->mBegin == i->mEnd) emptylist = true;
      }
      // anything to do?
      if (emptylist && !mState.empty()) {
         mState[0].mCur = mState[0].mEnd;
      }
      Increment();
   }

//______________________________________________________________________________
   int BasicWindowIterator::GetPermutations () const
   {
      // order 1 => 1 permutations
      if (mWindow.GetOrder() == 1) {
         return 1;
      }
      // count remaining events in list
      int N = mWindow.Size();
      if ((mWindow.GetOffset() <= Interval (0.)) &&
         (mWindow.GetOffset() + mWindow.GetWidth() > Interval (0.))) {
         --N;
      }
      // number of current events to choose
      int k = mWindow.GetOrder() - 1; 
      if (N < k) {
         return 0; // not enough events in list!
      }
      // calculate number of possible permutation
      int p = 1;
      for (int j = N; j > N - k; --j) {
         p *= j; // Choose 1 in N, 1 in (N-1), ... 1 in (N-k+1)
      }
      return p;
   }

//______________________________________________________________________________
   void BasicWindowIterator::SetPermutation ()
   {
      mWindow.SetCurrent (0, const_cast<Event*>(&*mState[0].mCur));
      if (mWindow.GetOrder() == 1) {
         return;
      }
      // make a list of indices
      std::vector<int> mIndices (mWindow.Size());
      int cur = -1;
      for (int j = 0; j < mWindow.Size(); ++j) {
         mIndices[j] = j;
         if (&mWindow[j] == &mWindow(0)) cur = j;
      }
      // throw away current(0) if in list
      if (cur >= 0) mIndices.erase (mIndices.begin() + cur);
      // loop over possibilities
      int M = mPermMax;
      int i = mPermI;
      for (int j = 1; j < mWindow.GetOrder(); ++j) {
         M /= mIndices.size();
         mWindow.SetCurrent (j, &mWindow[mIndices[i/M]]);
         mIndices.erase (mIndices.begin() + i/M);
         i %= M;
      }
   }

}
