#ifndef _LIGO_EVENTFUNCTION_H
#define _LIGO_EVENTFUNCTION_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Function						*/
/*                                                         		*/
/* Module Description: Defines an abstract function of an event		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Function.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

namespace events {


   class Event;
   class Argument;
   class Value;


/** This is an abstract class for defining functions on events.
   
    @memo Defines an event
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Function {
   public:
      /** Creates an event function.
          @memo Default constructor
       ******************************************************************/
      Function() {
      }
      /** Destructs an event function.
          @memo Denstructor
       ******************************************************************/
      virtual ~Function() {
      }
      /** Returns a copy of the event function. This method must be 
          overriden by all descendents.
          @memo Copy the event function
          @return event function copy
       ******************************************************************/
      virtual Function* Copy() const  = 0;
   
      /** Returns the value of the function. This function must be
          overriden by descendents. The function returns true if it could
          be evaluated and false otherwise.
   
          @memo Function value
          @param arg Event argument list
          @param val value (return)
          @return True if function could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const = 0;
      /** Function call operator. Returns the same as Evaluate 
          (should not be overriden).
   
          @memo () operator
          @param arg Event argument list
          @param val Value (return)
          @return True if function could be evaluated
       ******************************************************************/
      bool operator() (const Argument& arg, Value& val) const {
         return Evaluate (arg, val); }
      /** Function call operator. Evaluates a function on a single
          event (should not be overriden).
          @memo () operator
          @param arg Event argument list
          @param val Value (return)
          @return True if function could be evaluated
       ******************************************************************/
      bool operator() (const Event& event, Value& val) const;
   
   protected:
      /** Copy constructor. This is almost never intended, so just
          disable it from the outside.
          @memo Copy constructor
       ******************************************************************/
      Function (const Function&) {
      }
      /** Assignment operator. This is almost never intended, so just
          disable it from the outside.
          @memo Assignment operator
       ******************************************************************/
      Function& operator= (const Function&) {
         return *this; }
   
   };

}

#endif // _LIGO_EVENTFUNCTION_H

