#ifndef _LIGO_EVENTCOLUMNTYPE_H
#define _LIGO_EVENTCOLUMNTYPE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ColumnType						*/
/*                                                         		*/
/* Module Description: Defines the possible types of event columns	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ColumnType.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <complex>
#include <string>
#include "Time.hh"


namespace events {

   class Event;

   const int ColumnTypekNum = 6;


/** Event column types. This class describes the supported column types.
    Currently the following types are supported:
    \begin{verbatim}
    Time - GPS time in sec and nsec (2 x unsigned 32 bit integer)
    Real - A double precision floating point value
    Complex - A double precision complex floating point value
    Int - A 32 bit signed integer value
    String - A null terminated character array
    Event - Useful for composite events such as coincidences
    \end{verbatim}
    In the current implementation only simple types which don't
    have virtual methods and don't contain pointers are suppored.
    This is due to the fact that when an event is created no
    constructors are called but rather the whole data block is
    initialized with zeros. Upon creation of a new event data block
    the only non-zero element is the number of columns.
   
    @memo Defines the event column types
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
   ************************************************************************/
   struct ColumnType {
      /** Column value type (enum value is column order!) 
          Update Value when changing.
       ********************************************************************/
      enum Enum {
      /// Inavlid
      kInvalid = 0,
      /// Complex double (128 bit)
      kComplex,
      /// Time (stored as (uint,uint))
      kTime,
      /// Double (64 bit)
      kReal,
      /// Int (32 bit)
      kInt,
      /// String (stored as char[])
      kString,
      /// Event
      kEvent
      };
      /// Number of supported types
      static const int kNum;
   
      /// Pointer to generic data type
      typedef void* data_ptr;
      /// Const pointer to generic data type
      typedef const void* const_data_ptr;
      /// Integer type
      typedef int Int;
      /// Floating point type
      typedef double Real;
      /// Complex type
      typedef std::complex<double> Complex;
      //  Time type
      typedef Time time_type;
      /// String type (within an event a string is stored as a char[])
      typedef std::string String;
      /// Event type
      typedef events::Event Event;
   
      /// Column number type
      typedef Int ColumnNumber_t;
      /// Column event sub ID type
      typedef Int ColumnName_t;
      /// Column time type
      typedef time_type ColumnTime_t;
      /// Column ifo set type
      typedef Int ColumnIfo_t;
   
      /// Structure for common fixed columns
      struct Common_t {
         /// Column number
         ColumnNumber_t	mColumns;
         /// Event subtype
         ColumnName_t	mName;
         /// Event time
         ColumnTime_t	mTime;
         /// Interferometer set
         ColumnIfo_t	mIfo;
      }; // Check Factory::AddFixedColumn if changing!
   
      /// Column name of event column number
      static const char* const kColumnNumberName;
      /// Column name of event subtype id
      static const char* const kColumnNameName;
      /// Column name of event time
      static const char* const kColumnTimeName;
      /// Column name of event interferometer set
      static const char* const kColumnIfoName;
   
      /// Calculates the address of a column value      
      static inline data_ptr Address (data_ptr p, int ofs) {
         return (data_ptr)((char*)p + ofs); }
      /// Calculates the address of a column value      
      static inline const_data_ptr Address (const_data_ptr p, int ofs) {
         return (const_data_ptr)((const char*)p + ofs); }
   
      /** Get the name of the column type.
          @param type Value type
          @return Type name
          @memo Get the type name
       ******************************************************************/
      static std::string TypeName (Enum type);
      /** Get the size of the column type (in bytes).
          @param type Value type
          @return Type size
          @memo Get the type size
       ******************************************************************/
      static int TypeSize (Enum type);
      /** Get the required alignment size of the column type (in bytes).
          @param type Value type
          @return Alignment size
          @memo Get the alignment size
       ******************************************************************/
      static int AlignmentSize (Enum type);
      /** Constructs a column value. If no initialization object is
          specified, writes zeros for simple types and calls the 
          constructor for complex types. If an initialization object
          is specified, copies  the mmeory region for simple types
          and calls the copt constructor for complex types.
          @memo Constructs a column value
          @param type Value type
          @param data Pointer to column data
          @param init Initialization object if non zero
          @return True if successful
       ******************************************************************/
      static bool ConstructValue (Enum type, data_ptr data, 
                        const_data_ptr init = 0);
      /** Destructs a column value. Does does nothing for simple types
          and calls the Destructor for complex types.
   
          @memo Destructs a column value
          @param type Value type
          @param data Pointer to column data
          @return True if successful
       ******************************************************************/
      static bool DestructValue (Enum type, data_ptr data);
      /** Copies a value form destinmation to source.
          @memo Copy column value
          @param type Value type
          @param d1 Destination
          @param d2 Source
          @return True if successful
       ******************************************************************/
      static bool CopyValue (Enum type, data_ptr d1, 
                        const_data_ptr d2);
      /** Compares two column values.
          @memo Compares two column values
          @param type Value type
          @param d1 Pointer to first column data
          @param d2 Pointer to second column data
          @return True if equal
       ******************************************************************/
      static bool CompareValue (Enum type, const_data_ptr d1, 
                        const_data_ptr d2);
      /** Compares column value against zero.
          @memo Compares column value against zero
          @param type Value type
          @param data Pointer to event data
          @return True if zero
       ******************************************************************/
      static bool IsZeroValue (Enum type, const_data_ptr data);
   };


}

#endif // _LIGO_EVENTCOLUMNTYPE_H
