#ifndef _LIGO_EVENTCOLUMNINFO_H
#define _LIGO_EVENTCOLUMNINFO_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ColumnInfo						*/
/*                                                         		*/
/* Module Description: Defines an event column				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ColumnInfo.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <vector>
#include "events/ColumnType.hh"


namespace events {


/** Event column information. This class describes the columns of
    an event. This class is used the event layout. Event columns are
    separated between fixed and varying. Fixed columns are identical
    for every event, whereas varying columns are specific to the 
    layout of the event. By default fixed columns are the event time,
    the event subtype and the interferometer set. A forth column
    describing the number of actual column allocated for the event
    is needed to add columns to an event layout at run-time. 
    
    The column names within an event layout have to be unique.
    Column names can not contain the characters "[", "]" or ".".
    They can not contain "(", ")" unless they are of the event type 
    and contain the string "Event" together with a positive index.

    A column information contains the following information:
    \begin{verbatim}
    Name - The name of the column (case insensitive)
    Type - The column type, i.e., Real, Time, Int, Complex or String.
    CharSize - The maximum length of a string type
    Fixed - True if a fixed column
    Column - The column index of this column within an event layout
    Offset - The byte offset of column value within an event data block
    \end{verbatim}

    @memo Defines an event column
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ColumnInfo : public ColumnType {
      friend class Factory;
   
   public:
      /** Creates an empty event column information record.
          @memo Default constructor
          @param name Column name
          @param type Column type
       ******************************************************************/
      explicit ColumnInfo (const char* name = 0, 
                        Enum type = kInvalid) {
         Init (name, type); }
      /** Creates an event column.
          @memo Constructor
          @param name Column name
          @param type Column type
       ******************************************************************/
      explicit ColumnInfo (const std::string& name, 
                        Enum type = kInvalid) {
         Init (name.c_str(), type); }
   
      /** Equality operator. Compares the column names only 
          (case-insensitive).
          @memo Equality operator
       ******************************************************************/
      bool operator== (const ColumnInfo& desc) const;
      /** Inequality operator. Compares the column names only 
          (case-insensitive).
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const ColumnInfo& desc) const {
         return !(*this == desc); }
      /** Less operator. Orders the columns by type and name.
          The type order is Complex < Time < Real < Int < String.
          @memo Less operator
       ******************************************************************/
      bool operator< (const ColumnInfo& desc) const;
   
      /** Set the column name and type. This will reintialize the 
          object and reset column number and offset.
          @memo Set the column name
          @return True if valid name
       ******************************************************************/
      bool SetNameType (const char* name, Enum type);
      /** Get the column name. 
          @memo Get the column name
       ******************************************************************/
      const char* GetName () const {
         return (mType == kInvalid) ? 0 : mName.c_str(); }
      /** Get the column type. 
          @memo Get the column type
       ******************************************************************/
      Enum GetType () const {
         return mType; }
      /** Get the type name of the column. 
          @memo Get the column type name
       ******************************************************************/
      std::string GetTypeName () const {
         return ColumnType::TypeName (mType); }
      /** Get the size of the column type (in bytes).
          @memo Get the type size
       ******************************************************************/
      int GetTypeSize () const {
         return TypeSize (mType); }
      /** Get the required alignment size of the column type (in bytes).
          @memo Get the alignment size
       ******************************************************************/
      int GetTypeAlignment () const {
         return AlignmentSize (mType); }
      /** Set the column number.
          @memo Set column number
       ******************************************************************/
      void SetColumn (int col) {
         mColumn = col; }
      /** Get the column number.
          @memo Get column number
       ******************************************************************/
      int GetColumn () const {
         return mColumn; }
      /** Set the column data offset.
          @memo Set column data offset
       ******************************************************************/
      void SetOffset (int offset) {
         mOffset = offset; }
      /** Get the column data offset.
          @memo Get column data offset
       ******************************************************************/
      int GetOffset () const {
         return mOffset; }
      /** Is this a fixed column.
          @memo Fixed column?
       ******************************************************************/
      bool IsFixed () const {
         return mFixed; }
   
      /** Constructs a column value. Writes zeros for simple types
          and calls the constructor for complex types.
          @memo Constructs a column value
          @param data Pointer to event data
          @return True if successful
       ******************************************************************/
      bool Construct (data_ptr data, const_data_ptr init = 0) {
         return ConstructValue (mType, Address (data, mOffset),
                              init ? Address (init, mOffset) : 0); }
      /** Destructs a column value. Does does nothing for simple types
          and calls the Destructor for complex types.
          @memo Destructs a column value
          @param data Pointer to event data
          @return True if successful
       ******************************************************************/
      bool Destruct (data_ptr data) {
         return DestructValue (mType, Address (data, mOffset)); }
      /** Compares two column values.
          @memo Compares two column values
          @param d1 Pointer to first column data
          @param d2 Pointer to second column data
          @return True if equal
       ******************************************************************/
      bool Compare (const_data_ptr d1, const_data_ptr d2) const {
         return CompareValue (mType, Address (d1, mOffset),
                             Address (d2, mOffset)); }
      /** Compares column value against zero.
          @memo Compares column value against zero
          @param data Pointer to event data
          @return True if zero
       ******************************************************************/
      bool IsZero (const_data_ptr data) const {
         return IsZeroValue (mType, Address (data, mOffset)); }
   
      /** Check if a valid column name is specified.
          @memo Check name
          @param name Column name
          @return True if valid
       ******************************************************************/
      static bool CheckName (const char* name);
   
   protected:
      /** Initializes the column information record.
          @memo Initializes the column information record
          @param name Column name
          @param type Column type
       ******************************************************************/
      void Init (const char* name, Enum type);
   
   private:
      /** Set fixed or varying column.
          @memo Set fixed column
       ******************************************************************/
      void SetFixed (bool set = true) {
         mFixed = set; }
   
      /// Column name
      std::string	mName;
      /// Columnm type
      Enum		mType;
      /// Fixed columns?
      bool		mFixed;
      /// Column index
      int		mColumn;
      /// Data offset
      int		mOffset;
   };


/** @name ColumnInfoList
    A list of column information records.
    @memo  A list of column information records
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
   /// A list of column information records
   typedef std::vector<ColumnInfo> ColumnInfoList;
//@}

}

#endif // _LIGO_EVENTCOLUMNINFO_H
