/* -*- mode: c++; c-basic-offset: 3; -*- */
//////////////////////////////////////////////////////////////////////////
//  									//
//  Algorithm Implementation of non-template code			//
//  									//
//////////////////////////////////////////////////////////////////////////

#include "PConfig.h"
#include <time.h>
#include <stdio.h>
#include <iostream>
#include <queue>
#include <vector>
#include <functional>
#include <algorithm>
#include <cstring>
#include "events/AlgorithmUtil.hh"
#include "events/Algorithm.hh"
#include "events/Event.hh"
#include "events/Iterator.hh"
#include "events/WindowIterator.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/Argument.hh"
#include "events/List.hh"
#include "events/Chain.hh"
#include "events/Set.hh"
#include "events/Value.hh"
#include "events/Column.hh"
#include "events/Cluster.hh"
#include "TSeries.hh"
#include "Histogram1.hh"
#include "Histogram2.hh"


namespace events {

//______________________________________________________________________________
   void SetColumn (const Iterator& sourceBegin, 
                  const Iterator& sourceEnd,
                  const Column& column,
                  const Function& expression)
   {
      // create column object
      Column col (column);
      // create window iterators
      WindowIterator i (sourceBegin, sourceEnd, 1, 0.);
      WindowIterator end (sourceEnd, sourceEnd, 1, 0.);
      Value val;
      // loop over events
      for (; i != end; ++i) {
         // evaluate expression
         if (expression.Evaluate (*i, val)) {
            col.Set (i->Current(), val);
         }
      }
   }

//______________________________________________________________________________
   void SetColumn (const Iterator& sourceBegin, 
                  const Iterator& sourceEnd,
                  const Column& column,
                  const Function& expression,
                  const Condition& cond,
                  const TimeWindow& window)
   {
      // create column object
      Column col (column);
      // create window iterators
      WindowIterator i (sourceBegin, sourceEnd, 1, window);
      WindowIterator end (sourceEnd, sourceEnd, 1, window);
      // loop over events
      Value val;
      bool ret;
      for (; i != end; ++i) {
         // evaluate condition and expression
         if (cond.Evaluate (*i, ret) && ret && 
            expression.Evaluate (*i, val)) {
            col.Set (i->Current(), val);
         }
      }
   }

//______________________________________________________________________________
   Iterator Remove (const Iterator& sourceBegin, 
                   const Iterator& sourceEnd, 
                   const Condition& cond,
                   const TimeWindow& window)
   {
      // create window iterators
      Iterator last = sourceBegin;
      WindowIterator i (sourceBegin, sourceEnd, 1, window);
      WindowIterator end (sourceEnd, sourceEnd, 1, window);
      // loop over events
      for (; i != end; ++i) {
         bool ret;
         if (cond.Evaluate (*i, ret) && ret) {
            // nothing: will be removed
         }
         else {
            // swap current event with last if not the same
            if (&i->Current() != &*last) {
               std::swap (i->Current(), *last);
            }
            ++last;
         }
      }
      return last;
   }

//______________________________________________________________________________
// NOTE: the 2nd list has to be disjoint!!!
   bool VetoGate(Set& in1, Set& in2,
                Set& out1p, Set& out1f, Set& out2p, Set& out2f,
                Interval offset)
   {
      return VetoGate (in1.Begin(), in1.End(), in2.Begin(), in2.End(),
                      std::back_inserter(out1p), std::back_inserter(out1f),
                      std::back_inserter(out2p), std::back_inserter(out2f),
                      offset);
   }

//______________________________________________________________________________
   bool IsDisjoint (const ConstIterator& beg, const ConstIterator& end)
   {
      if (beg == end) {
         return true;
      }
      ConstIterator itr1 = beg;
      bool ID=true;
      Time START;
      Time START_Last = itr1->GetTime();
      double DUR;
      double DUR_Last;
      Column colDur("Duration");
      if (!colDur.Get (*itr1, DUR_Last)) { 
         fprintf (stderr, "Duration not readable!\n"); 
         return false; 
      } // Duration was not readable
      for (++itr1; ID && itr1 != end; ++itr1) {
         START = itr1->GetTime();
         if (!colDur.Get (*itr1, DUR)) { 
            fprintf (stderr, "Duration not readable!\n"); 
            return false; 
         } // Duration was not readable
         if (START < START_Last + DUR_Last) ID = false;
         START_Last = START;
         DUR_Last = DUR;
      }
      return ID;
   }

//______________________________________________________________________________
   bool IsDisjoint (Set& s)
   {
      return IsDisjoint(s.Begin(),s.End());
   }

//______________________________________________________________________________
   bool CheckOrder (const ConstIterator& begin, const ConstIterator& end)
   {
      // anything to do?
      if (begin == end) {
         return true;
      }
      // two iterators which follow each other
      ConstIterator i = begin;
      ConstIterator j = begin; ++j;
      // loop over all events
      for (; j != end; ++i, ++j) {
         // check proper timimg order
         if (!(*i < *j)) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
//    void Sort (Iterator sourceBegin, Iterator sourceEnd,
//              const Function& func, bool ascending)
//    {
//       // create an array of sort elements
//       sortarray sarr;
//       for (Iterator i = sourceBegin; i != sourceEnd; ++i) {
//          sarr.push_back (SortElement (*i, func, ascending));
//       }
//       // sort it
//       if (ascending) {
//          std::sort (sarr.begin(), sarr.end());
//       }
//       else {
//          std::sort (sarr.begin(), sarr.end(), 
//                    std::greater<SortElement>());
//       }
//       // Now swap original events into correct order
//       std::vector<Event> temp (sarr.size());
//       std::vector<Event>::iterator i = temp.begin();
//       for (sortarray::iterator j = sarr.begin(); j != sarr.end(); 
//           ++j, ++i) {
//           std::swap (*const_cast<Event*>(j->Get()), *i);
//       }
//       i = temp.begin();
//       for (Iterator j = sourceBegin; j != sourceEnd; ++j, ++i) {
// 	  std::swap (*j, *i); 
//       }
//    }

//______________________________________________________________________________
   int MakeHistogram (Histogram1& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& func,
                     const Condition& cond,
                     const TimeWindow& window)
   {
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // loop over events
      int n = 0;
      for (; i != end; ++i) {
         // evaluate conditition and function
         bool ret;
         Value val;
         double x;
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret && 
            func.Evaluate (const_cast<Window&>(*i), val) && val.Write (x)) {
            // found a event: add to histogram
            hist.Fill (x);
            ++n;
         }
      }
      return n;
   }

//______________________________________________________________________________
   int MakeHistogram (Histogram2& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& f1, 
                     const Function& f2, 
                     const Condition& cond,
                     const TimeWindow& window)
   {
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // loop over events
      int n = 0;
      for (; i != end; ++i) {
         // evaluate conditition and function
         bool ret;
         Value v1, v2;
         double x, y;
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret && 
            f1.Evaluate (const_cast<Window&>(*i), v1) && v1.Write (x) &&
            f2.Evaluate (const_cast<Window&>(*i), v2) && v2.Write (y)) {
            // found a event: add to histogram
            hist.Fill (x, y);
            ++n;
         }
      }
      return n;
   }

//______________________________________________________________________________
   int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     bool integrated,
                     const Condition& cond,
                     const TimeWindow& window)
   {
      if (sourceBegin == sourceEnd) {
         return 0;
      }
      // make sure we have a time series
      if ((ts.getStartTime() == Time(0,0)) ||
         (ts.getInterval() <= Interval (0.0)) ||
         (ts.getNSample() <= 0)) {
         ConstIterator last = sourceEnd;
         --last;
         Interval duration = last->GetTime() - sourceBegin->GetTime();
         int M = 1000;
         double dt = duration / double(M - 1);
         if (dt <= 0) {
            dt = 1.0;
            M = 1;
         }
         ts = TSeries (sourceBegin->GetTime(), Interval (dt), 1000);
      }
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // allocate data array
      int M = ts.getNSample();
      double* y = new double [M];
      memset (y, 0, M * sizeof (double));
      // setup loop variables
      Time t0 (ts.getStartTime());
      double dt = ts.getTStep();
      int num = 0;
      int indx = 0;
      bool ret;
      // loop through events
      for (; i != end; ++i) {
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret) {
            indx = (int) ((double)(i->GetTime() - t0) / dt + 0.5);
            if ((indx >= 0) && (indx < M)) {
               y[indx] += 1.0;
               ++num;
            }
         }
      }
      // integrate if requested
      if (integrated) {
         double integral = 0.0;
         for (int i = 0; i < M; ++i) {
            integral += y[i];
            y[i] = integral;
         }
      }
      // set data in time series
      ts.setData (t0, Interval (dt), y, M);
      delete [] y;
      return num;
   }

//______________________________________________________________________________
   int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& func,
                     const Condition& cond,
                     const TimeWindow& window)
   {
      if (sourceBegin == sourceEnd) {
         return 0;
      }
      // make sure we have a time series
      if ((ts.getStartTime() == Time(0,0)) ||
         (ts.getInterval() <= Interval (0.0)) ||
         (ts.getNSample() <= 0)) {
         ConstIterator last = sourceEnd;
         --last;
         Interval duration = last->GetTime() - sourceBegin->GetTime();
         int M = 1000;
         double dt = duration / double(M - 1);
         if (dt <= 0) {
            dt = 1.0;
            M = 1;
         }
         ts = TSeries (sourceBegin->GetTime(), Interval (dt), 1000);
      }
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // allocate data array
      int M = ts.getNSample();
      double* y = new double [M];
      int* w = new int [M];
      memset (y, 0, M * sizeof (double));
      memset (w, 0, M * sizeof (int));
      // setup loop variables
      Time t0 (ts.getStartTime());
      double dt = ts.getTStep();
      int num = 0;
      int indx;
      bool ret;
      Value val;
      double x;
      // loop through events
      for (; i != end; ++i) {
         // evaluate conditition and function
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret && 
            func.Evaluate (const_cast<Window&>(*i), val) && val.Write (x)) {
            indx = (int) ((double)(i->GetTime() - t0) / dt + 0.5);
            if ((indx >= 0) && (indx < M)) {
               if (w[indx]) {
                  y[indx] = ((double)w[indx] * y[indx] + x) / 
                     (double)(w[indx] + 1);
                  ++w[indx];
               }
               else {
                  y[indx] = x;
                  w[indx] = 1;
               }
               ++num;
            }
         }
      }
      // eliminate zeros
      double prev = 0;
      for (int i = 0; i < M; ++i) {
         if (w[i]) prev = y[i]; 
         else y[i] = prev;
      }
      // set data in time series
      ts.setData (t0, Interval (dt), y, M);
      delete [] y;
      delete [] w;
      return num;
   }

}
namespace std {
//______________________________________________________________________________
   ostream& operator<< (ostream& out, const events::List& list)
   {
      Write (out, list.Begin(), list.End());
      return out;
   }

//______________________________________________________________________________
   istream& operator>> (istream& inp, events::List& list)
   {
      events::Read (inp, back_inserter (list));
      return inp;
   }

//______________________________________________________________________________
   ostream& operator<< (ostream& out, const events::Chain& chain)
   {
      events::Write (out, chain.Begin(), chain.End());
      return out;
   }

//______________________________________________________________________________
   istream& operator>> (istream& inp, events::Chain& chain)
   {
      events::Read (inp, back_inserter (chain));
      return inp;
   }

//______________________________________________________________________________
   ostream& operator<< (ostream& out, const events::Set& set)
   {
      events::Write (out, set.Begin(), set.End());
      return out;
   }

//______________________________________________________________________________
   istream& operator>> (istream& inp, events::Set& set)
   {
      events::Read (inp, back_inserter (set));
      return inp;
   }
}
