/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGPAR_HH
#define TRIGPAR_HH
//
//    TrigPar  passes the parameters for a trigger filter or detector.
//
//    include templates
#include <map>
#include <functional>
#include <string>
#include <iosfwd>

class Param;

/**  A trigger parameter dictionary holds a list of parameters and their 
  *  values. Each parameter can be of type int, double, or string.
  *  \brief Trigger parameter dictionary class.
  *  \author John G Zweizig
  *  \version 1.2; Last Modified April 15, 1999.
  */
class TrigPar {
public:
    /**  Parameter dictionary default constructor.
      *  \brief Parameter dictionary default constructor.
      */
    TrigPar(void) {}

    /**  Make a dictionary and add parameters as described by the file.
      *  \brief Initializing constructor.
      *  \param FileName Parameter definition file path
      */
    TrigPar(const char *FileName);

    /**  Make a dictionary and add parameters as described by the file.
      *  \brief Initializing constructor.
      *  \param file Parameter definition stream
      */
    TrigPar(std::istream& file);

    /** Dictionary destructor.
      * \brief Dictionary destructor.
      */
  ~TrigPar(void);

    /**  Test whether the named parameter is defined.
      *  \brief Test whether the named parameter is defined.
      *  \param ParName Parameter name.
      */
    bool exists(const std::string& ParName) const;

    /**  The named parameter may be assigned a value (e.g. Dict["MyPar"] = 5;)
      *  or accessed (e.g. int = int(Dict["MyPar"]);) using the [] operator.
      *  If an undefined name is used on the left-hand-side of an assignment,
      *  the specified parameter is created with the type of the rhs of the
      *  assignment.
      *  \brief Reference to the specified parameter block.
      *  \param Name Parameter name
      *  \return Reference to a parameter descriptor.
      */
    Param& operator[](const std::string &Name);

    /**  The named parameter may be assigned a value (e.g. Dict["MyPar"] = 5;)
      *  or accessed (e.g. int = int(Dict["MyPar"]);) using the [] operator.
      *  Referencing an undefined parameter will cause an exception.
      *  \brief Constant reference to a specified parameter block.
      *  \param Name Parameter name
      *  \return Constant reference to the requested parameter object.
      */
    const Param& operator[](const std::string &Name) const;

    /**  The value of the specified parameter is converted to integer if 
      *  necessary and returned.
      *  \brief Get an integer parameter value.
      *  \param ParName Parameter name
      *  \return Integer parameter value.
      */
    int getInt(const std::string& ParName) const;

    /**  Get the parameter value (converted to double).
      *  \brief Get an double precision parameter value.
      *  \param ParName Parameter name
      *  \return Double precision parameter value.
      */
    double getDouble(const std::string& ParName) const;

    /**  Get the parameter value (converted to string).
      *  \brief Get a string parameter value.
      *  \param ParName Parameter name
      *  \return String parameter value.
      */
    std::string getString(const std::string& ParName) const;

    /**  Create an integer parameter and add it to the dictionary.
      *  \brief Add an integer parameter.
      *  \param ParName Parameter name
      *  \param ipar Integer parameter value
      */
    void addPar(const std::string& ParName, int ipar);

    /**  Add a double (float) parameter.
      *  \brief Add a double (float) parameter.
      *  \param ParName Parameter name
      *  \param dpar Double precision parameter value
      */
    void addPar(const std::string& ParName, double dpar);

    /**  Add a string parameter.
      *  \brief Add a string parameter.
      *  \param ParName Parameter name
      *  \param spar String parameter value
      */
    void addPar(const std::string& ParName, const std::string& spar);

    /**  Add a parameter of the specified type.
      *  \brief Add a parameter of the specified type.
      *  \param Name Parameter name
      *  \param Type Parameter data type
      *  \param Value String parameter value
      */
    void addPar(const std::string& Name, const char *Type, const char *Value);

    /**  Remove the specified parameter.
      *  \brief Remove the specified parameter.
      *  \param Name Parameter name
      */
    void remove(const std::string& Name);

    /**  Set parameter to an int
      *  \brief Set parameter to an int
      *  \param ParName Parameter name
      *  \param ipar Integer parameter value
      */
    void setPar(const std::string& ParName, int ipar);

    /**  Set parameter to a double
      *  \brief Set parameter to a double
      *  \param ParName Parameter name
      *  \param dpar Double precision parameter value
      */
    void setPar(const std::string& ParName, double dpar);

    /**  Set parameter to a string
      *  \brief Set parameter to a string
      *  \param ParName Parameter name
      *  \param spar String parameter value
      */
    void setPar(const std::string& ParName, const std::string& spar);

    /**  Fill parameter dictionary from the specified file.
      *  \brief Fill parameter dictionary from the specified file.
      *  \param istr   STL input stream containing parameter definitions.
      *  \param prefix Parameter name prefix.
      */
    void Read(std::istream& istr, const std::string& prefix="");

    /**  Write the parameters, their types and values to a stream.
      *  \brief Write the parameters, their types and values to a stream.
      *  \param ostr   STL output stream to write parameter definitions.
      *  \param prefix Parameter name prefix.
      *  \return Output stream reference.
      */
    std::ostream& Write(std::ostream& ostr, const std::string& prefix="") const;

  private:
    /**  Compare string names.
      */
    typedef std::less<std::string> scmp;
    typedef std::map<std::string, Param*, scmp> ParMap;
    typedef ParMap::iterator map_iter;
    typedef ParMap::const_iterator const_map_iter;
    ParMap  mParSet;
};

//--------------------------------------  Inline functions.
inline Param& 
TrigPar::operator[](const std::string &Name) {
    return *mParSet[Name];}

inline const Param& 
TrigPar::operator[](const std::string &Name) const {
    return *(mParSet.find(Name)->second);}


#endif  //  TRIGPAR_HH
