/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef WINDOW_API_HH
#define WINDOW_API_HH

#include "Pipe.hh"
#include <memory>
#include "gds_memory.hh"

class DVector;

/**  The %window_api class is an abstract base class that performs common 
  *  windowing calculations. The derived classes typically differ from one 
  *  another only in the window function and the constructors needed to 
  *  provide whatever variation is appropriate to the specific window type. 
  *  The %window_api class forces a normalization of \f$\Sigma W[i]^2 = 1.0\f$.
  *  This maintains the power of a windowed series approximately constant, 
  *  but it overrides any normalization built into the window function.
  *
  *  The window function calculation can save half of the computation 
  *  steps for windows that are symmetric about the center bin (\f$x = \pi\f$). 
  *  This symmetry is assumed by default. Derived classes representing 
  *  windows that don't exhibit this symmetry should define isPiSymmetric() 
  *  to return false.
  *  @memo Windowing base class.
  *  @version 1.1 ; Modified July 21, 1999
  *  @author John G. Zweizig
  */
class window_api : public Pipe {
public:
   using Pipe::apply;
   using Pipe::dataCheck;
   using Pipe::isDataValid;
  
  /// Element type of window array
  typedef double element_type;
   
  /**  Build an empty window function.
    *  @memo Default constructor.
    */
  window_api(void);

  /**  Copy constructor.
    *  @memo Copy constructor.
    *  @param w %window_api to be copied.
    */
  window_api (const window_api& w);
  
  /**  Destroy the %Window object and release the function storage.
    *  @memo Virtual destructor.
    */
  virtual ~window_api();

  /**  Copy the contents of the argument %window_api base class into this.
    *  @memo Assignment operator.
    *  @param w %Window to be copied.
    *  @return Reference to this %Window.
    */
  window_api& operator= (const window_api& w);
  
  /**  The argument time series is multiplied on an element-by-element basis
    *  by the windowing function. The resulting windowed series is returned.
    *  The argument TSeries is left unchanged. The Window normalization is
    *  set to \f$\Sigma_{i=0}^{N-1} W[i]^2 = 1.0\f$, as described in the class
    *  description.
    *  @memo Return a windowed TSeries.
    *  @param ts Data to which the window is to be applied.
    *  @return Windowed time series.
    */
  TSeries apply(const TSeries& ts);

  /**  The clone method returns a pointer to a new %Window object identical 
    *  to the current one.
    *  @memo Clone a %Window.
    *  @return Pointer to thenew identical %Window.
    */
  virtual window_api* clone(void) const = 0;

  /**  Check the data for validity. If the data are not applicable for 
    *  this window, an exception is thrown.
    *  @memo Test if data are valid for input to window.
    *  @param ts Time series data to test for Window compatibility.
    */
  void dataCheck(const TSeries& ts) const;

  /**  Check the data for validity. Performs the same data checks as 
    *  dataCheck() but returns a boolean status instead of throwing an
    *  exception.
    *  @memo Test if data are valid for input to window.
    *  @param ts Time series data to test for Window compatibility.
    *  @return True if data works for window.
    */
  bool isDataValid(const TSeries& ts) const;

  /**  Return the current window length.
    *  @memo Get the current window length.
    *  @return length of window vector in samples.
    */
  int getLength(void) const;

  /**  Return the window function RMS. This is calculated automatically in 
    *  the setWindow() method before the window is normalized.
    *  @memo Get window function RMS.
    *  @return RMS of the original window function.
    */
  double getRMS(void) const;

  /**  The Window table length is set to the specified value. A new function 
    *  array is allocated if insufficient storage is available. The function 
    *  array is then filled with the calculated window coefficients.
    *  @memo Set the length and calculate the Window function.
    *  @param N Number of entries in the current window function table.
    */
  void setWindow(int N);

  /**  The window function is calculated at the argument position 
    *  (\f$arg = 2 \pi n/(N-1)\f$, n is the entry number, N is the total 
    *  number of entries)
    *  @memo Calculate the Window function.
    *  @param arg The function parameter, x, defined in the range 
    *             \f$0 <= x < 2 \pi \f$.
    *  @return the window function at the specified argument value.
    */
  virtual double WinFc(double arg) = 0;

  /**  Return true if the Window function is symmetric around \f$arg = \pi\f$.
    *  Functions are by default symmetric, but the symmetry assumption 
    *  may be preempted by defining isPiSymmetric() for a derived class.
    *  @memo Test the Window function for symmetry around \f$ arg = \pi \f$.
    *  @return True if the window function is symmetric around 
    *          \f$ arg = \pi \f$.
    */
  virtual bool isPiSymmetric(void);

  /**  Get the start time of the current data epoch. The start time is 
    *  set to the time of the first filtered sample after either the
    *  filter is constructed or the filter is reset. If no data have 
    *  been windowed since the construction or reset, the start time 
    *  is given as 0.
    *  @memo Get start time.
    *  @return Current epoch start time or 0 if no data have been windowed.
    */
  Time getStartTime(void) const;

  /**  Get the current time of the current data epoch. The current time is 
    *  set with the time of the next expected sample to be filtered.
    *  @memo Get current time.
    *  @return GPS time currently being processed.
    */
  Time getCurrentTime(void) const;

  /**  Returns true if the filter is in use.
    *  @memo Test whether windo is in use.
    *  @return True if in use.
    */
  bool inUse(void) const;

  /**  Get a reference to the window data Vector.
    *  @memo Get the window function Vector.
    */
  const DVector& refDVect(void) const;

  /**  Prepare the window for a disjoint data epoch. This method is needed
    *  for the Pipe class.
    *  @memo Reset the window. 
    */
  void reset(void);

private:
  /**  Data vector containing the window coefficients.
    */
  std::unique_ptr<DVector> mWindow;

  /**  RMS value of window coefficients.
    */
  double mRMS;
};

//======================================  Inline functions
inline double 
window_api::getRMS(void) const {
    return mRMS;
}

inline Time 
window_api::getStartTime(void) const {
    return Time(0);
}

inline Time 
window_api::getCurrentTime(void) const {
    return Time(0);
}

inline bool 
window_api::inUse(void) const {
    return true;
}

inline void
window_api::reset(void) {
}

inline const DVector& 
window_api::refDVect(void) const {
    return *mWindow;
}

#endif  // !defined(WINDOW_API_HH)
