/*----------------------------------------------------------------------*/
/*                                                                      */
/* Module Name: monweb                                                  */
/*                                                                      */
/* Module Description: web server class for monitor data and provide    */
/* graphical representation. It requires the 'graph' utility of the     */
/* plotutils package.	                                             	*/
/*                                                                      */
/* Revision History:                                                    */
/* Rel   Date     Programmer  	Comments                                */
/* 0.1	4Jan02   D. Sigg    	First release                           */
/* 0.2  15July02  Sz. Marka   Graphics support                          */
/*                                                                      */
/* Documentation References:                                            */
/*	Man Pages: webmon.html                                          */
/*	References: none                                                */
/*                                                                      */
/* Author Information:                                                  */
/* Name          Telephone       Fax             e-mail                 */
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu    */
/* Szabi Marka   (626) 395-2005                  smarka@ligo.caltech.edu*/
/*                                                                      */
/*                                                                      */
/*                      -------------------                             */
/*                                                                      */
/*                             LIGO                                     */
/*                                                                      */
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.   	*/
/*                                                                      */
/*                     (C) The LIGO Project, 1999.                      */
/*                                                                   	*/
/*                                                                   	*/
/* Caltech				MIT                             */
/* LIGO Project MS 51-33		LIGO Project NW-17 161          */
/* Pasadena CA 91125			Cambridge MA 01239              */
/*                                                                      */
/* LIGO Hanford Observatory		LIGO Livingston Observatory     */
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.             */
/* Richland WA 99352			Livingston, LA 70754            */
/*                                                                      */
/*----------------------------------------------------------------------*/

#ifndef _LIGO_WEBMON_H
#define _LIGO_WEBMON_H

#include "web/webserver.hh"
#include "Time.hh"

   //const char LogFileName[]="/export/home/smarka/LOGS/WebViewAccess.log";

namespace web {

   const int DEFAULT_THREADS = 500;  // number of parallel web server threads
   const int DEFAULT_CACHE_SIZE = 16*1024*1024;// in bytes
   const int CACHING_TIME = 59;      // in sec

#ifndef __CINT__
   const char* const kSeriesTypenames[4] = 
   {"TSeries", "FSeries", "FSpectrum", "Histogram1"};
#endif


/** @name Web server class for displaying montitor data
    This web server displays data from the DMT monitors which 
    are exported by monitor API. Result pages are created 
    dynamically using the data monitor API and the SVG plotting
    tools.

    @memo Data monitor web server
    @author Written January 2002 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class monweb_server : public basic_server
   {
   public:
      /// Data object type
      enum SeriesType {
      /// Time series
      TimeSeries = 0,
      /// Frequency series
      FrequencySeries = 1,
      /// Spectrum
      Spectrum = 2,
      /// 1D histogram
      Histogram1D = 3
      };
   
      /// Supported data types
      enum data_type {
      /// Unsupported
      unsupported,
      /// plain XML
      xml,
      /// Text
      ascii,
      /// Binary
      binary,
      /// Scalable vector format
      svg,
      /// JPEG
      jpeg,
      /// GIF
      gif,
      /// Portable document format
      pdf,
      /// Postscript
      ps,
      /// Portable network graphics
      png
      };
   
      /// Data object
      class data_object {
      public:
         /// Name
         std::string 		fName;
         /// Type
         SeriesType 		fType;
         /// Comment
         std::string 		fOpt;
      };
      /// Object access history
      class object_access {
      public:
         object_access () : fLoaded (false) {
	 }
         /// Successfully loaded
	 bool			fLoaded;
	 /// Last update
	 Time			fUpdateTime;
      };
      /// Data object list
      typedef std::map <std::string, data_object> object_list;
      /// Listings for data objects
      typedef std::pair <object_access, object_list> object_listing;
      /// Monitor list
      typedef std::map <std::string, object_listing> monitor_list;
   
      /** Constructs a monitor web server.
          
          @memo Constructor
          To disable caching set the size to zero. A negative cache
          size indicates no limit.
          @param argn Number of cmd line arguments
          @param argv list of cmd line arguments
          @param cachesize Maximum size of cache (bytes)
       ******************************************************************/
      monweb_server (int argn, char** argv, 
                     int cachesize = DEFAULT_CACHE_SIZE);
      /** Process a GET request.
          Can be overriden by descendents.
          @memo Process GET request
          @param request requested page
          @param answer Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestGet (const request_t& request,
                        response_t& response);
   
      /** Updates the list of monitor names.
          @memo Update monitor names
          @return True if successful
       ******************************************************************/
      virtual bool UpdateNames();
      /** Updates a monitor.
          @memo Update monitor objects
          @return True if successful
       ******************************************************************/
      virtual bool UpdateObjects (monitor_list::iterator i);
      /** Get the update time.
          @memo Update time
          @return Time of last update
       ******************************************************************/
      Time GetUpdateTime() const;
      /** Is name part of the monitor list?
          @memo Is monitor
          @return True if name is a monitor
       ******************************************************************/
      virtual bool IsMonitor (const std::string& name);
      /** Is name part of the object/monitor list?
          @memo Is data object
          @return True if monitor/object exists
       ******************************************************************/
      virtual bool IsObject (const std::string& mon, 
                        const std::string& obj);
      /** Get the data object
          @memo Get data object
          @return True if data object can be loaded
       ******************************************************************/
      virtual bool GetData (const std::string& mon, 
                        const std::string& obj, char*& buf, int& len, 
                        data_type dtype);
   
      /** Get the data type from extension
          @memo Get data type
          @return data type
       ******************************************************************/
      static data_type fromExt (const std::string& ext);
      /** Get the mime type from the data type
          @memo Get mime type
          @return mime type
       ******************************************************************/
      static std::string mimeType (data_type dt);
      /** Get the file type from the data type
          @memo Get file type
          @return file type
       ******************************************************************/
      static std::string fileType (data_type dt);
   
   private:
      /// Monitor list
      monitor_list	fNames;
      /// Timestamp when monitor list was last read
      Time		fNamesTime;
   };

}
#endif // _LIGO_WEMON_H
