/* -*-  mode: c++; c-basic-offset: 4; -*- */
#include <time.h>
#include "DVecType.hh"
#include "Dacc.hh"
#include "TSeries.hh"
#include "TrendChan.hh"
#include "FrWriter.hh"
#include "ChanName.hh"
#include <iostream>
#include <iomanip>
#include <stdexcept>

using namespace std;

//======================================  Check channel name
const string::size_type max_trend_chan_name_len = 255;

bool 
TrendChan::validName(const std::string& chan, std::string& monid) {
    ChanName ch;
    if (chan.size() > max_trend_chan_name_len) {
        cerr << "Channel name > " << max_trend_chan_name_len << " characters." 
	     << endl;
	return false;
    }
    try {
        ch.setChannel(chan);
    } catch (std::exception& e) {
        cerr << "Error in channel name: " << e.what() << endl;
	return false;
    }
    if (ch.getSubSystem() != "DMT") {
        cerr << "Error in channel name: SubSystem is not DMT." << endl;
	return false;
    }
    if (ch.getLocale().size() != 4) {
        cerr << "Error in channel name: Program Id not 4 characters." << endl;
	return false;
    } else if (monid.empty()) {
        monid = ch.getLocale();
    } else if (monid != ch.getLocale()) {
        cerr << "Error in channel name: Program Id not " << monid << endl;
	return false;
    }
    return true;
}

//======================================  Default constructor
TrendChan::TrendChan(void) 
  : mStartFrame(0), mStartAcc(0), mAccIntvl(0.0)
{
}

//======================================  Copy Constructor
TrendChan::TrendChan(const TrendChan& x) {
    *this = x;
}

//======================================  Constructor
TrendChan::TrendChan(const char* Name, Interval nAvg) 
  : mName(Name), mStartFrame(0), mStartAcc(0), mAccIntvl(nAvg)
{
    mNSampTS.setName( (mName + ".n").c_str()    );
    mAvgTS.setName(   (mName + ".mean").c_str() );
    mRmsTS.setName(   (mName + ".rms").c_str()  );
    mMinTS.setName(   (mName + ".min").c_str()  );
    mMaxTS.setName(   (mName + ".max").c_str()  );
}

//======================================  Destructor
TrendChan::~TrendChan(void) {
}

//======================================  Copy a TrendChan
TrendChan&
TrendChan::operator=(const TrendChan& x) {
    mName       = x.mName;
    mAccum      = x.mAccum;
    mStartFrame = x.mStartFrame; 
    mStartAcc   = x.mStartAcc;
    mAccIntvl   = x.mAccIntvl;
    mNSampTS    = x.mNSampTS; 
    mAvgTS      = x.mAvgTS;
    mRmsTS      = x.mRmsTS;
    mMinTS      = x.mMinTS;
    mMaxTS      = x.mMaxTS;
    mNSampTS.setName( x.mNSampTS.getName()  );
    mAvgTS.setName(   x.mAvgTS.getName()    );
    mRmsTS.setName(   x.mRmsTS.getName()    );
    mMinTS.setName(   x.mMinTS.getName()    );
    mMaxTS.setName(   x.mMaxTS.getName()    );
    return *this;
}

//======================================  Addition operator
TrendChan&
TrendChan::operator+=(const TrendChan& x) {
    if (mStartFrame != x.mStartFrame) 
        throw runtime_error("Unequal start times");

    if (mAccIntvl  != x.mAccIntvl) throw runtime_error("Unequal intervals");

    if (mStartAcc != Time(0)) appendPoint(); 

    double* pAvg1 = reinterpret_cast<double*>(mAvgTS.refData());
    double* pRms1 = reinterpret_cast<double*>(mRmsTS.refData());
    float*  pMin1 = reinterpret_cast<float*>(mMinTS.refData());
    float*  pMax1 = reinterpret_cast<float*>(mMaxTS.refData());
    int*    pCnt1 = reinterpret_cast<int*>(mNSampTS.refData());

    const double* pAvg2 = reinterpret_cast<const double*>(x.mAvgTS.refData());
    const double* pRms2 = reinterpret_cast<const double*>(x.mRmsTS.refData());
    const float*  pMin2 = reinterpret_cast<const float*>(x.mMinTS.refData());
    const float*  pMax2 = reinterpret_cast<const float*>(x.mMaxTS.refData());
    const int*    pCnt2 = reinterpret_cast<const int*>(x.mNSampTS.refData());

    unsigned long N = mNSampTS.getNSample();
    if (N > x.mNSampTS.getNSample()) N = x.mNSampTS.getNSample();
    for (unsigned long i=0; i<N; ++i) {
       if (!pCnt1[i]) {
	   pAvg1[i] = pAvg2[i];
	   pRms1[i] = pRms2[i];
	   pMin1[i] = pMin2[i];
	   pMax1[i] = pMax2[i];
	   pCnt1[i] = pCnt2[i];
       } else if (pCnt2[i]) {
	   double n1 = pCnt1[i];
	   double n2 = pCnt2[i];
	   pAvg1[i]  = (pAvg1[i] * n1 + pAvg2[i] * n2)/(n1 + n2);
	   pRms1[i]  = sqrt((pow(pRms1[i],2)*n1+pow(pRms2[i],2)*n2)/(n1+n2));
       	   pCnt1[i] += pCnt2[i];
	   if (pMin2[i] < pMin1[i]) pMin1[i] = pMin2[i];
	   if (pMax2[i] > pMax1[i]) pMax1[i] = pMax2[i];
       }
    }
    return *this;
}

//=======================================  Add all data from a time series
void 
TrendChan::addData(const TSeries& ts) {
    if (ts.isEmpty()) return;
    Time       t = ts.getStartTime();
    Interval  dT = ts.getTStep();
    count_t nwds = ts.getNSample();
    math_t* data = new math_t[nwds];
    ts.getData(nwds, data);
    for (count_t i=0 ; i < nwds ; i++) {
        addData(t, data[i]);
	t += dT;
    }
    delete data;
}

//=======================================  Add a specified data value.
void 
TrendChan::addData(const Time& t, math_t point) {
    if (!mStartAcc) {
	startAcc(t);
    } if (t < mStartAcc || t >= mStartAcc+mAccIntvl) {
        trendPoint();
	startAcc(t);
    }
    mAccum.addData(point);
}

//======================================  Dump the channel state.
ostream&
TrendChan::dump(ostream& os) const {
    os << mName << " " << setprecision(12) << mStartFrame.totalS() << " " 
       << mStartAcc.totalS() << " " << setprecision(5) << mAccIntvl << endl;
    return os;
}

//======================================  Get the current end-time
Time
TrendChan::getEndTime(void) const {
    return mNSampTS.getEndTime();
}

long 
TrendChan::getNSample(const Time& tStart, const Time& tStop) const {
    int iStart = mNSampTS.getBin(tStart);
    int iStop  = mNSampTS.getBin(tStop);

    const int* pSample = reinterpret_cast<const int*>(mNSampTS.refData());
    long Sum = 0;
    for (int i=iStart ; i<iStop ; i++) Sum += pSample[i];
    return Sum;
}

//======================================  Set the trend data
void 
TrendChan::setData(const TSeries& Avg) {
    if (Avg.isEmpty()) return;
    TSeries Nsamp(Avg);
    Nsamp *= 0.0;
    Nsamp.Convert(DVecType<int>::getDataType());
    int N=Nsamp.getNSample();
    int* p = reinterpret_cast<int*>(Nsamp.refData());
    for (int i=0; i<N; ++i) *p++ = 1;
    setData(Avg, Avg, Avg, Avg, Nsamp);
}

//======================================  Set the trend data
void 
TrendChan::setData(const TSeries& Avg, const TSeries& Rms, const TSeries& Min, 
		   const TSeries& Max, const TSeries& Num) {
    Time startRepl = Avg.getStartTime();
    if (startRepl != Rms.getStartTime() || startRepl != Num.getStartTime() || 
	startRepl != Min.getStartTime() || startRepl != Max.getStartTime()) {
        throw runtime_error("Inconsistent replacement times");
    }
    mStartFrame = startRepl;

    mAccIntvl = Avg.getTStep();
    if (mAccIntvl != Rms.getTStep() || mAccIntvl != Num.getTStep() || 
	mAccIntvl != Min.getTStep() || mAccIntvl != Max.getTStep()) {
        throw runtime_error("Inconsistent replacement t-step");
    }

    mNSampTS = Num;
    mNSampTS.Convert(DVecType<int>::getDataType());
    mAvgTS   = Avg;
    mAvgTS.Convert(DVecType<double>::getDataType());
    mRmsTS   = Rms;
    mRmsTS.Convert(DVecType<double>::getDataType());
    mMinTS   = Min;
    mMinTS.Convert(DVecType<float>::getDataType());
    mMaxTS   = Max;
    mMaxTS.Convert(DVecType<float>::getDataType());
    mStartAcc = Time(0);
    mAccum.reset();
}

//======================================  Set the trend channel name
void 
TrendChan::setName(const char* Name) {
    mName = Name;
}

//======================================  Set the trend channel name
void 
TrendChan::setUnits(const string& units) {
    mUnits = units;
}

//======================================  Reset the accumulator
void 
TrendChan::startAcc(const Time& start) {
    long iBin = long((start - mStartFrame)/mAccIntvl);
    Time accStart = mStartFrame + double(iBin)*mAccIntvl;
    if (!mStartAcc || mStartAcc != accStart) {
        mStartAcc = accStart;
	mAccum.reset();
    }
}

//====================================== Start a trend at the specified time
void 
TrendChan::startFrame(const Time& Start) {
    if (!mNSampTS.isEmpty() && mNSampTS.getEndTime() > Start) {
        discard(Start);
    } else {
        float  fx(0.0);
	double dx(0.0);
	int    ix(0);
        mNSampTS.setData(Start, mAccIntvl, &ix, 1);
	mAvgTS.setData(Start, mAccIntvl, &dx, 1);
	mRmsTS.setData(Start, mAccIntvl, &dx, 1);
	mMinTS.setData(Start, mAccIntvl, &fx, 1);
	mMaxTS.setData(Start, mAccIntvl, &fx, 1);
    }
    mStartFrame = Start;
    startAcc(Start);
}

//======================================  Synchronize to the specified time
void 
TrendChan::synch(const Time& t) {
    if (mStartAcc != Time(0) && mStartAcc < t) trendPoint();
    if (t > getEndTime()) extend(t);
}

//======================================  Clear all data
void 
TrendChan::clear(void) {
    mStartFrame = Time(0);
    mNSampTS.Clear();
    mAvgTS.Clear();
    mRmsTS.Clear();
    mMinTS.Clear();
    mMaxTS.Clear();
    reset();
}

//======================================  Discard data from front of trend
void 
TrendChan::discard(const Time& Start) {
    Interval tErase = Start - mNSampTS.getStartTime();
    mNSampTS.eraseStart(tErase);
    mAvgTS.eraseStart(tErase);
    mRmsTS.eraseStart(tErase);
    mMinTS.eraseStart(tErase);
    mMaxTS.eraseStart(tErase);
}

//======================================  Extend series to the specified time
void 
TrendChan::extend(const Time& t) {
    mNSampTS.extend(t);
    mAvgTS.extend(t);
    mRmsTS.extend(t);
    mMinTS.extend(t);
    mMaxTS.extend(t);
}

//======================================  Add n accumulated point to the trend
void 
TrendChan::trendPoint(void) {
    Time StartTime = mNSampTS.getStartTime();
    Time SeriesEnd = mNSampTS.getEndTime();
    if (Almost(mStartAcc, SeriesEnd)) {
        appendPoint();
    } else if (mStartAcc >= SeriesEnd) {
        extend(mStartAcc);
	SeriesEnd =  mNSampTS.getEndTime();
	if (Almost(mStartAcc, SeriesEnd)) appendPoint();
	else {
	    cout << "Start Frame/StartAcc/SeriesEnd = " << mStartFrame 
		 << "/" << mStartAcc << "/" << SeriesEnd << endl;
	    throw runtime_error("Unable to append trend point");
	}
    } else if (mStartAcc < StartTime) {
        reset();
	cout << "Start Frame/StartAcc = " << mStartFrame 
	     << "/" << mStartAcc << endl;
	throw runtime_error("Attempt to replace trend point");
    } else if (mAccum.getNSample() != 0) {
        incrementPoint();
    }
}

//======================================  Append point to end of current trend
void 
TrendChan::appendPoint(void) {
    int  nSample = mAccum.getNSample();
    mNSampTS.Append(mStartAcc, mAccIntvl, &nSample, 1);
    double Avg = mAccum.getAverage();
    mAvgTS.Append(mStartAcc, mAccIntvl, &Avg, 1);
    double Rms = mAccum.getRMS();
    mRmsTS.Append(mStartAcc, mAccIntvl, &Rms, 1);
    float  Min = mAccum.getMinimum();
    mMinTS.Append(mStartAcc, mAccIntvl, &Min, 1);
    float  Max = mAccum.getMaximum();
    mMaxTS.Append(mStartAcc, mAccIntvl, &Max, 1);
    reset();
}

//======================================  Append point to end of current trend
void 
TrendChan::incrementPoint(void) {
    int  n1 = mAccum.getNSample();
    if (!n1) return;

    int i = mNSampTS.getBin(mStartAcc + 0.5*mAccIntvl);

    int*    nSample = reinterpret_cast<int*>(mNSampTS.refData());
    double* Avg     = reinterpret_cast<double*>(mAvgTS.refData());
    double* Rms     = reinterpret_cast<double*>(mRmsTS.refData());
    float*  Min     = reinterpret_cast<float*>(mMinTS.refData());
    float*  Max     = reinterpret_cast<float*>(mMaxTS.refData());

    int  n0 = nSample[i];
    if (n0) {
        nSample[i] += n1;
	Avg[i] = (Avg[i]*n0 + mAccum.getAverage()*n1)/double(n0+n1);
	Rms[i] = sqrt((Rms[i]*Rms[i]*n0 + mAccum.getSumSq())/double(n0+n1));
	if (mAccum.getMinimum() < Min[i]) Min[i] = mAccum.getMinimum();
	if (mAccum.getMaximum() > Max[i]) Max[i] = mAccum.getMaximum();
    } else {
        nSample[i] = n1;
	Avg[i] = mAccum.getAverage();
	Rms[i] = mAccum.getRMS();
	Min[i] = mAccum.getMinimum();
	Max[i] = mAccum.getMaximum();
    }
    reset();
}

void 
TrendChan::reset(void) {
    mStartAcc = Time(0);
    mAccum.reset();
}

//======================================  Get ready to read out the data.
void 
TrendChan::setReadout(Dacc& In) {
    In.addChannel(mNSampTS.getName());
    In.addChannel(  mAvgTS.getName());
    In.addChannel(  mRmsTS.getName());
    In.addChannel(  mMinTS.getName());
    In.addChannel(  mMaxTS.getName());
}

//======================================  Get ready to read out the data.
void 
TrendChan::setReadData(Dacc& In) {
    //cerr << "Dump of: " << mAvgTS.getName() << endl;
    //In.refData(  mAvgTS.getName()) -> Dump(cerr);
    mNSampTS = *In.refData(mNSampTS.getName());
    mAvgTS   = *In.refData(  mAvgTS.getName());
    mRmsTS   = *In.refData(  mRmsTS.getName());
    mMinTS   = *In.refData(  mMinTS.getName());
    mMaxTS   = *In.refData(  mMaxTS.getName());

    //----------------------------------  Reset the accumulator.
    reset();

    //----------------------------------  Set the fill Time.
    mStartFrame = In.getFillTime();
}

//======================================  Add channels to be written.
void
TrendChan::write(FrWriter& out) {
    if (mNSampTS.isEmpty()) return;
    mNSampTS.setUnits("ct");
    out.addRawSeries(mNSampTS.getName(), mNSampTS);
    mAvgTS.setUnits(mUnits);
    out.addRawSeries(mAvgTS.getName(), mAvgTS);
    mRmsTS.setUnits(mUnits);
    out.addRawSeries(mRmsTS.getName(), mRmsTS);
    mMinTS.setUnits(mUnits);
    out.addRawSeries(mMinTS.getName(), mMinTS);
    mMaxTS.setUnits(mUnits);
    out.addRawSeries(mMaxTS.getName(), mMaxTS);
}






