#ifndef TRENDACC_HH
#define TRENDACC_HH

#include "math.h"

/**  Accumulate trend data.
  *  @memo Trend data accumulator.
  *  @author John Zweizig.
  *  @version 1.1; Modified February 18, 2005.
  */
class TrendAcc {
public:
  /**  Data type for data counts.
    */
  typedef unsigned int count_t;

  /**  Arithmetic data type.
    */
  typedef double math_t;

  /**  Default constructor.
    *  \brief Default constructor.
    */
  TrendAcc(void);

  /**  Destructor
    *  \brief Destructor.
    */
  ~TrendAcc();

  /**  Return the current running average. The running average is an average 
    *  of all data points that have not been added into a trend frame.
    *  @memo Running average.
    *  @return Average of points not entered into a Trend frame TSeries.
    */
  math_t getAverage(void) const;

  /**  Return the current running Minimum. The running minimum is the
    *  smallest of all data points after the last trend calculation.
    *  @memo Running minimum.
    *  @return Minimum point not entered into a Trend frame TSeries.
    */
  math_t getMinimum(void) const;

  /**  Return the current running Maximum. The running minimum is the
    *  smallest of all data points after the last trend calculation.
    *  @memo Running maximum.
    *  @return Maximum point not entered into a Trend frame TSeries.
    */
  math_t getMaximum(void) const;

  /**  Return the current running average. The running average is calculated 
    *  from all data points after the last trend calculation.
    *  @memo Running average.
    *  @return Average of points not entered into a Trend frame TSeries.
    */
  count_t getNSample(void) const;

  /**  Return the current running RMS. The running RMS is calculated from
    *  all data points after the last point added to the trend series.
    *  @memo Running RMS.
    *  @return Average of points not entered into a Trend frame TSeries.
    */
  math_t getRMS(void) const;

  /**  Return the current running standard deviation. The running standard 
    *  deviation is calculated from all data points after the last point 
    *  added to the trend series.
    *  @memo Running standard deviation.
    *  @return Standard deviation of points not entered into a Trend TSeries.
    */
  math_t getSigma(void) const;

  /**  Return the current running sum of the values squared. The running
    *  sum squared is calculated from all data points after the last point 
    *  added to the trend series.
    *  @memo Running sum of values squared.
    *  @return Average of point
    */
  math_t getSumSq(void) const;

  /**  Add a data point to the trend running average.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void addData(math_t point);

  /**  Add each element in a vector of double data points to the trend 
    *  running average.
    *  \brief Add a vector of doubles.
    *  \param vector Constant pointer to the vector to be added.
    *  \param nData  Number of data points in vector.
    */
  void addData(const double* vector, count_t nData);

  /**  Add each element in a vector of float data points to the trend running 
    *  average.
    *  \brief Add a vector of floats.
    *  \param vector Constant pointer to the vector to be added.
    *  \param nData  Number of data points in vector.
    */
  void addData(const float* vector, count_t nData);

  /**  Add each element in a vector of int data points to the trend running 
    *  average.
    *  \brief Add a vector of ints.
    *  \param vector Constant pointer to the vector to be added.
    *  \param nData  Number of data points in vector.
    */
  void addData(const int* vector, count_t nData);

  /**  Add each element in a vector of short data points to the trend running 
    *  average.
    *  \brief Add a vector of shorts.
    *  \param vector Constant pointer to the vector to be added.
    *  \param nData  Number of data points in vector.
    */
  void addData(const short* vector, count_t nData);

  /**  Reset the running average.
    *  \brief Reset accumulator.
    */
  void reset(void);

private:
  /**  Number of samples collected in the current trend period.
    */
  count_t  mNSample;

  /**  Sum of samples in the current trend period.
    */
  math_t   mAvg;

  /**  Sum of squares of samples in the current trend period.
    */
  math_t   mSumSq;

  /**  Minimum sample in the current trend period.
    */
  math_t   mMinimum;

  /**  Maximum sample in the current trend period.
    */
  math_t   mMaximum;
};

//--------------------------------------  Inline accessors
inline TrendAcc::math_t 
TrendAcc::getAverage(void) const {
    if (mNSample == 0) return 0.0;
    return mAvg/mNSample;
}

inline TrendAcc::math_t 
TrendAcc::getMaximum(void) const {
    return mMaximum;
}

inline TrendAcc::math_t 
TrendAcc::getMinimum(void) const {
    return mMinimum;
}

inline TrendAcc::count_t 
TrendAcc::getNSample(void) const {
    return mNSample;
}

inline TrendAcc::math_t 
TrendAcc::getRMS(void) const {
    if (mNSample == 0) return 0.0;
    return sqrt(mSumSq/mNSample);
}

inline TrendAcc::math_t 
TrendAcc::getSigma(void) const {
    if (mNSample == 0) return 0.0;
    math_t Avg = mAvg/mNSample;
    math_t SigSq = mSumSq/math_t(mNSample) - Avg*Avg;
    if (SigSq > 0.0) return sqrt(SigSq);
    return 0.0;
}

inline TrendAcc::math_t 
TrendAcc::getSumSq(void) const {
    return mSumSq;
}

inline void 
TrendAcc::addData(math_t point) {
    if (!mNSample++) {
        mMinimum = point;
        mMaximum = point;
    } else if (point < mMinimum) {
        mMinimum = point;
    } else if (point > mMaximum) {
        mMaximum = point;
    }
    mAvg   += point;
    mSumSq += point*point;
}

#endif  // TRENDACC_HH
