/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "PConfig.h"
#include <sstream>
#include <cstdlib>
#include <unistd.h>
#include <cstdio>
#include <cstring>
#include <iostream>
#include <map>
#include "monapi/monaccess_dmtserver.hh"
#include "gmutex.hh"
#include "web/webclient.hh"
#include "xml/XsilTSeries.hh"
#include "xml/XsilFSpectrum.hh"
#include "xml/XsilHistogram.hh"

namespace monapi {
   using namespace std;
   using namespace web;

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Constants                                                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   const char* const kEnvWeb = "DMTWEBSERVER";
   const char* const kEnvLmsg = "DMTNAMESERVER";

   const int kDefaultPort = 0;
   const char* const kDefaultNameServers = 
   "marble.ligo-wa.caltech.edu:443/dmtview/LHO,"
   "llodmt.ligo-la.caltech.edu:443/dmtview/LLO";


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Globals                                                              //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   thread::readwritelock 	nameservermux;
   bool 			nameserverinit = false;
   monaccess_dmtserver::NameServerList monaccess_dmtserver::fList;



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// monaccess_dmtserver                                                  //
//                                                                      //
// monitor access by DMT server                                         //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   monaccess_dmtserver::monaccess_dmtserver (const char* server) 
   : monaccess (server)
   {
      if (nameserverinit) {
         return ;
      }
   
      nameservermux.writelock();
      if (!nameserverinit) {
         const char* nameserver = ::getenv (kEnvWeb);
         if (!nameserver) {
            nameserver = ::getenv (kEnvLmsg);
         }
         if (!nameserver) {
            nameserver = kDefaultNameServers;
         }
            // parse name server string
         if (nameserver) {
            int indx = 0;
            string n (nameserver);
            while (!n.empty()) {
               // find first name server
	       string m = n;
	       string::size_type pos = n.find (',');
               if (pos != string::npos) {
                  m = n.substr(0, pos);
                  n.erase(0, pos + 1);
               }
               else {
                  n.clear();
               }
               // cout << "n = " << n << endl;
               // cout << "m = " << m << endl;
               // find nick name
               NameServer nserv;
	       //-----------------------  Get the nick-name
	       pos = m.rfind ('/');
               if (pos != string::npos) {
                  nserv.fNickName = string (m, pos + 1);
                  m.erase (pos);
               }
               else {
                  ostringstream nick;
                  nick << indx + 1;
                  nserv.fNickName = nick.str();
               }

	       //-----------------------  Look for a base-path
	       pos = m.find ('/');
               if (pos != string::npos) {
                  nserv.fBasePath = string (m, pos + 1);
                  m.erase (pos);
               }

               // cout << "m = " << m << endl;
               // find port number
	       pos = m.rfind (':');
               if (pos != string::npos) {
                  nserv.fPort = strtoul(m.c_str() + pos + 1, 0, 0);
                  m.erase(pos, string::npos);
               }
               else {
                  nserv.fPort = kDefaultPort;
               }

               // cout << "m = " << m << endl;
               // rest is address
               nserv.fAddr = m;
               if (!nserv.fAddr.empty()) {
                  fList.push_back (nserv);
                  ++indx;
#if 0
                  cout << "fAddr     = " << nserv.fAddr << endl;
                  cout << "fNickName = " << nserv.fNickName << endl;
                  cout << "fBasePath = " << nserv.fBasePath << endl;
                  cout << "fPort     = " << nserv.fPort << endl;
#endif
               }
            }
         }
         nameserverinit = true;
      }
      nameservermux.unlock();
   }

//______________________________________________________________________________
   monaccess_dmtserver::~monaccess_dmtserver(void) {
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::getIndex (std::string& list)
   {
      nameservermux.writelock();
      list = "";
      for (NameServerList::iterator i = fList.begin();
          i != fList.end(); ++i) {
      	 // make HTTP request
         http_request req;
	 string server_addr = i->fAddr;
	 if (! i->fBasePath.empty()) {
	    server_addr +=  "/";
	    server_addr += i->fBasePath;
	 }
         if (!req.request ("monitors.txt", server_addr.c_str(), i->fPort)) {
	    cerr << "monitors request to: " << i->fAddr.c_str() << ":"
		 << i->fPort;
	    if (!(i->fBasePath.empty())) cerr << "/" << i->fBasePath;
	    cerr << " failed" << endl;
            continue;
         }
      
         // parse reponse
         if ((req.size() == 0) || !req.data()) {
	    cerr << "invalid monitors response" << endl;
            continue;
         }
         string n (req.data());
         string m;
         while (!n.empty()) {
            // find first name server
	    string::size_type pos = n.find ("\r\n");
            if (pos != string::npos) {
               m = string (n, 0, pos);
               n.erase (0, pos + 2);
            }
            else {
               m = n;
               n = "";
            }
            //cout << "mm = " << m << endl;
            if (!m.empty()) {
               i->fServers.push_back (m);
               if (!list.empty()) list += ";";
               list += m;
               // add nickname
               if (fList.size() > 1) {
                  list += "@";
                  list += i->fNickName;
               }
            }
         }
         i->fServersInit = true;
      }
      nameservermux.unlock();
   
      return true;
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::setServer (const char* server)
   {
      fServerEntries.clear();
      return monaccess::setServer (server);
   }

//______________________________________________________________________________
   std::string
   getServerURL(const std::string& server, const std::string& path) {
      string url;
      if (!path.empty()) url += path + "/";
      string::size_type pos = server.rfind("@");
      if (pos == string::npos) {
	 url += http_request::mangle (server);
      }
      else {
	 url += http_request::mangle (server.substr (0, pos));
      }
      return url;
   }

//______________________________________________________________________________
   bool 
   monaccess_dmtserver::getServerEntry(int n, std::string& name, 
				       std::string& type, std::string& comm)
   {
      if (fServer.empty() || (n < 0)) {
         return false;
      }
   
      if ((n == 0) || fServerEntries.empty()) {
         fServerEntries.clear();
         string addr, path;
         int port;
         if (findNameServer (fServer.c_str(), addr, port, path) < 0) {
            return false;
         }
      	 // make HTTP request
         http_request req;
         string url = getServerURL(fServer, path);
         url += "/objects.txt";
         if (!req.request (url.c_str(), addr.c_str(), port)) {
            return false;
         }
         // parse reponse
         if ((req.size() > 0) && req.data()) {
            string nn (req.data());
            string mm;
            string::size_type pos;
            while (!nn.empty()) {
               // find first name server
               if ((pos = nn.find ("\r\n")) != string::npos) {
                  mm = string (nn, 0, pos);
                  nn.erase (0, pos + 2);
               }
               else {
                  mm = nn;
                  nn = "";
               }
               //cout << "mm = " << mm << endl;
               if (!mm.empty()) {
                  fServerEntries.push_back (mm);
               }
            }
         }
      }
   
      // too far?
      if (n >= (int)fServerEntries.size()) {
         return false;
      }
   
      // parse line
      string m = fServerEntries[n];
      string::size_type pos = m.find ('\t');
      if (pos == string::npos) {
         return false;
      }
      name = string (m, 1, pos - 2);
      m.erase (0, pos + 1);
      pos = m.find ('\t');
      if (pos == string::npos) {
         return false;
      }
      type = string (m, 0, pos);
      m.erase (0, pos + 1);
      if (m == "-") {
         comm = "";
      }
      else {
         comm = string (m, 1, m.size() - 2);
      }
      return true;
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::getData (const char* name, TSeries* ts)
   {
      string s;
      if (!getdata (name, s)) {
	 cerr << "getData (\"" << name << "\", TSeries*) failed" << endl;
         return false;
      }
   
      std::vector<TSeries> objects;
      xml::xsilHandlerQueryTSeries query (objects);
      xml::xsilParser parser;
      parser.AddHandler (query);
      parser.Parse (s.data(), s.size());
   
      if (objects.empty()) {
	 cerr << "Unable to parse " << name << " TSeries xml" << endl;
         return false;
      }
      *ts = objects[0];
      //ts->Dump(cout);
   
      return true;
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::getData (const char* name, FSeries* fs)
   {
      string s;
      if (!getdata (name, s)) {
	 cerr << "getData (\"" << name << "\", FSeries*) failed" << endl;
         return false;
      }
   
      std::vector<FSeries> objects;
      xml::xsilHandlerQueryFSeries query (objects);
      xml::xsilParser parser;
      parser.AddHandler (query);
      parser.Parse (s.data(), s.size());
   
      if (objects.empty()) {
	 cerr << "Unable to parse " << name << " FSeries xml" << endl;
         return false;
      }
      *fs = objects[0];
   
      return true;
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::getData (const char* name, FSpectrum* fs)
   {
      string s;
      if (!getdata (name, s)) {
	 cerr << "getData (\"" << name << "\", FSpectrum*) failed" << endl;
         return false;
      }
   
      std::vector<FSpectrum> objects;
      xml::xsilHandlerQueryFSpectrum query (objects);
      xml::xsilParser parser;
      parser.AddHandler (query);
      parser.Parse (s.data(), s.size());
   
      if (objects.empty()) {
	 cerr << "Unable to parse " << name << " FSpectrum xml" << endl;
         return false;
      }
      *fs = objects[0];
   
      return true;
   }

//______________________________________________________________________________
   bool monaccess_dmtserver::getData (const char* name, Histogram1* hg)
   {
      string s;
      if (!getdata (name, s)) {
         return false;
      }
   
      std::vector<Histogram1> objects;
      xml::xsilHandlerQueryHistogram query (objects);
      xml::xsilParser parser;
      parser.AddHandler (query);
      parser.Parse (s.data(), s.size());
   
      if (objects.empty()) {
         return false;
      }
      *hg = objects[0];
   
      return true;
   }

//______________________________________________________________________________
   bool 
   monaccess_dmtserver::getdata (const char* name, string& data)
   {
      if (!name || !*name) {
         return false;
      }
      string addr, path;
      int port;
      //cout << "Server = " << fServer << endl;
      if (findNameServer (fServer.c_str(), addr, port, path) < 0) {
	 cerr << "failed to find name server" << fServer << endl;
         return false;
      }
      // make HTTP request
      http_request req;
      string url = getServerURL(fServer, path);
      url += "/";
      url += http_request::mangle (name);
      url += "/data.xml";
      if (!req.request (url.c_str(),addr.c_str(), port)) {
	 cerr << "Request for addr: " << addr << ":" << port << " url: " 
	      << url << " failed" << endl;
         return false;
      }
      // parse response
      if ((req.size() <= 0) || !req.data()) {
	 cerr << "Requested data not returned." << endl;
         return false;
      }
      if (strncasecmp (req.data(), "<?xml", 5) == 0) {
         data = req.data();
      }
      else {
         ostringstream os;
         os << xml::xsilHeader() << endl;
         os << req.data() << endl;
         os << xml::xsilTrailer() << endl;
         data = os.str();
      }
      return true;
   }

//______________________________________________________________________________
   int 
   monaccess_dmtserver::findNameServer (const char* server, std::string& addr, 
					int& port, std::string& path)
   {
      thread::semlock 	lockit (nameservermux);
      if (!server || fList.empty()) {
         return -1;
      }
      // if we have one name server that must be it
      if (fList.size() == 1) {
         addr = fList[0].fAddr;
         port = fList[0].fPort;
	 path = fList[0].fBasePath;
         return 0;
      }
      // first look for nick name
      string s (server);
      string::size_type pos = s.rfind ("@");
      if (pos != string::npos) {
         string nick = string (s, pos + 1);
         for (NameServerList::iterator i=fList.begin(); i != fList.end(); ++i) {
            if (nick == i->fNickName) {
               addr = i->fAddr;
               port = i->fPort;
	       path = i->fBasePath;
               return i - fList.begin();
            }
         }
         s.erase (pos);
      }
      // check if index was loaded
      bool init = true;
      for (NameServerList::iterator i = fList.begin();
          i != fList.end(); ++i) {
         if (!i->fServersInit) {
            init = false;
         }
      }
      if (!init) {
         nameservermux.unlock();
         string list;
         getIndex (list);
         nameservermux.readlock();
      }
      // now look for server name in list
      for (NameServerList::iterator i = fList.begin();
          i != fList.end(); ++i) {
         for (int j = 0; j < (int)i->fServers.size(); ++j) {
            if (i->fServers[j] == s) {
               addr = i->fAddr;
               port = i->fPort;
 	       path = i->fBasePath;
              return i - fList.begin();
            }
         }
      }
   
      return -1;
   }

}

//______________________________________________________________________________
extern "C" {
   monapi::monaccess* getMonAccessDMTServer (void)
   {
      return new monapi::monaccess_dmtserver();
   }
}


