/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Multiple streams of data access implementation
//
#include "MultiDacc.hh"
#include <stdio.h>
#include <iostream>

using namespace std;

//  dTcontig is the maximum acceptable discontinuity in nsec.
static const int dTcontig = 1;

//======================================  Default (Null) constructor.
MultiDacc::MultiDacc(void)
{
}

//======================================  Construct and connect to source.
MultiDacc::MultiDacc(const std::string& Source, const Time& STime)
{
    addMulti(Source);
}

//======================================  Destructor.
MultiDacc::~MultiDacc(void) {
    close();
    mChanList.clear();
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	delete mIn[i];
    }
}

//======================================  Add multiple frame sources.
void
MultiDacc::addMulti(const string& name) {
    size_t n0 = mList.size();
    mList.addMulti(name.c_str());
    for(size_t i=n0; i<mList.size(); i++) {
	Dacc* newDacc = new Dacc();
	newDacc->addPathList(mList.getList(i).c_str());
	mIn.push_back(newDacc);
    }
}

//======================================  Add a single frame source.
void
MultiDacc::addSingle(const string& name) {
    mList.addSingle(name.c_str());
    Dacc* newDacc = new Dacc();
    if (name[0] == '=') newDacc->addPath(name.substr(1));
    else                newDacc->addPathList(name);
    mIn.push_back(newDacc);
}
//======================================  Close all frame readers
void 
MultiDacc::channel_id(const string& chanid, int& daccid, string& chan) {
    size_t inx = chanid.find_first_of('/');
    if (inx == string::npos) {
	chan = chanid;
	daccid = getDaccIndex(chan);
    } else if (chanid[0] >= '0' && chanid[0] <= '9') {
	daccid = strtoul(chanid.c_str(), 0, 0);
	chan = chanid.substr(inx+1);
    } else {
	daccid = frame_type(chanid.substr(0, inx));
	chan = chanid.substr(inx+1);
    }
}

//======================================  Close all frame readers
void
MultiDacc::close(void) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->close();
    }
}

//======================================  Open a frame file on all streams
int
MultiDacc::open() {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	int j=mIn[i]->open();
	if(j!=0) return j;
    }
    return 0;
}

//======================================  Set the buffer count on all streams
void
MultiDacc::setBuffer(int nBuf) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setBuffer(nBuf);
    }
}

//======================================  Set the debug level on all streams
void
MultiDacc::setDebug(int nDbg) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setDebug(nDbg);
    }
}

//======================================  Set the ignore flag on all streams
void
MultiDacc::setIgnoreMissingChannel(bool yn) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setIgnoreMissingChannel(yn);
    }
}

//======================================  Set the nowait flag on all streams
void
MultiDacc::setNoWait(bool now) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setNoWait(now);
    }
}

//======================================  Set the stride on all streams
void
MultiDacc::setStride(Interval Dt) {
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	mIn[i]->setStride(Dt);
    }
}

//======================================  Get debug level
int
MultiDacc::getDebug(void) {
    if (mIn.empty()) return 0;
    return mIn[0]->getDebug();
}

//======================================  Get number of frames read
long
MultiDacc::getTotalFrames(void) const {
    if (mIn.empty()) return 0;
    return mIn[0]->getTotalFrames();
}

//======================================  Get current time (or Time(0))
Time
MultiDacc::getCurrentTime(void) const {
    if (mIn.empty()) return Time(0);
    return mIn[0]->getCurrentTime();
}

//======================================  Get fill time (or Time(0))
Interval
MultiDacc::getFillStride(void) const {
    if (mIn.empty()) return 0.0;
    return mIn[0]->getFillStride();
}

//======================================  Get fill time (or Time(0))
Time
MultiDacc::getFillTime(void) const {
    if (mIn.empty()) return Time(0);
    return mIn[0]->getFillTime();
}

//======================================  Get stride (or 0)
Interval
MultiDacc::getStride(void) const {
    if(mIn.empty()) return 0.0;
    return mIn[0]->getStride();
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addChannel(const string& Name, int id, int Decimate,
		      TSeries **TSptr) {
    int daccid;
    string channame;
    channel_id(Name, daccid, channame);

    chan_iter i=findChannel(channame);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    
    if (id < 0) id = daccid;
    mChanList.push_back(ChannelIndex(channame, id));
    if (id < 0) {
	cerr << "MultiDacc: failed to add channel " << Name << endl;
    } else {
	mIn[id]->addChannel(channame, Decimate, TSptr);
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addFSeries(const string& Name, int id, FSeries **FSptr) {
    int daccid;
    string channame;
    channel_id(Name, daccid, channame);

    chan_iter i=findChannel(channame);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    
    if (id < 0) id = daccid;
    mChanList.push_back(ChannelIndex(channame, id));
    if (id != -1) {
	mIn[id]->addFSeries(channame, FSptr);
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addRaw(const string& Name, int id, int Decimate, TSeries **TSptr) {
    int daccid;
    string channame;
    channel_id(Name, daccid, channame);

    chan_iter i=findChannel(channame);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    
    if (id < 0) id = daccid;
    mChanList.push_back(ChannelIndex(channame, id));
    if (id != -1) {
	mIn[id]->addRaw(channame, Decimate, TSptr);
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addProcessed(const string& Name, int id, int Decimate,
			TSeries **TSptr) {
    int daccid;
    string channame;
    channel_id(Name, daccid, channame);

    chan_iter i=findChannel(channame);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    
    if (id < 0) id = daccid;
    mChanList.push_back(ChannelIndex(channame, id));
    if (id != -1) {
	mIn[id]->addProcessed(channame, Decimate, TSptr);
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addSimulated(const string& Name, int id, int Decimate,
			TSeries **TSptr) {
    int daccid;
    string channame;
    channel_id(Name, daccid, channame);

    chan_iter i=findChannel(channame);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    
    if (id < 0) id = daccid;
    mChanList.push_back(ChannelIndex(channame, id));
    if (id != -1) {
	mIn[id]->addSimulated(channame, Decimate, TSptr);
    }
}

//======================================  See if channel was requested.
bool 
MultiDacc::isChannelRead(const string& Name) const {
    return findChannel(Name) != mChanList.end();
}

//======================================  List requested channels
ostream&
MultiDacc::list(ostream& out) const {
    for (size_t id=0; id<mIn.size(); id++) {
	out << "Data stream " << id << ": frame_type: " << frame_name(id)
	    << endl;
	mIn[id]->list(out);
    }
    return out;
}

//======================================  Frame type of the specified stream
std::string
MultiDacc::frame_name(int id) const {
    if (id < 0 || id >= int(mIn.size())) return "";
    string framepath = mIn[id]->refList().first();
    string::size_type inx = framepath.rfind("/");
    if (inx != string::npos) framepath.erase(0, inx+1);
    inx = framepath.find("-");
    if (inx != string::npos) {
	string::size_type inx2 = framepath.find("-", inx+1);
	if (inx2 != string::npos) inx = inx2;
    }
    if (inx != string::npos) framepath.erase(inx);
    return framepath;
}

//======================================  Stream with the specified frame type
int
MultiDacc::frame_type(const std::string& frameType) const {

   //--------------------------------  Find an input stream ID.
   size_t N = mIn.size();
   for (size_t i=0; i < N; ++i) {
       if (frame_name(i) == frameType) return i;
   }
   return -1;
}

//======================================  Find channel in request list.
MultiDacc::const_chan_iter 
MultiDacc::findChannel(const std::string& name) const {
    size_t inx = name.find_first_of('/');
    string raw = (inx == string::npos) ? name : name.substr(inx+1);
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	if (i->EqName(raw)) return i;
    }
    return mChanList.end();
}

MultiDacc::chan_iter 
MultiDacc::findChannel(const string& name) {
    size_t inx = name.find_first_of('/');
    string raw = (inx == string::npos) ? name : name.substr(inx+1);
    for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	if (i->EqName(raw)) return i;
    }
    return mChanList.end();
}

//======================================  Remove channel from request list.
void 
MultiDacc::rmChannel(const string& Name) {
    int index=getDaccIndex(Name);
    if (index != -1) {
	mIn[index]->rmChannel(Name);
    }
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	mChanList.erase(i);
    }
}

//======================================  Synchronize the data streams
int 
MultiDacc::synch(void) {

    //----------------------------------  See how they stack up
    Time tEarly(0), tLate(0);
    if (getDebug()) cerr << "Start Synch:";
    size_type N = mIn.size();
    for(size_type i=0; i<N; ++i) {
	int rc = mIn[i]->synch();
	if (rc != 0) return rc;
	Time ti = mIn[i]->getCurrentTime();
	if (getDebug()) cerr << " t[" << i << "]=" << ti;
	if (ti > tLate) tLate = ti;
	if (!i || ti < tEarly) tEarly =ti;
    }
    if (getDebug()) cerr << endl;


    //----------------------------------  Seek the latest.
    while (! Almost(tEarly, tLate) ) {
	Time tSeek = tLate;
	if (getDebug()) cerr << "Synch to: " << tSeek << endl;
	for(size_type i=0; i<N; ++i) {
	    int rc = mIn[i]->seek(tSeek);
	    if (rc != 0) return rc;
	    Time ti = mIn[i]->getCurrentTime();
	    if (getDebug()) cerr << " t[" << i << "]=" << ti;
	    if (ti > tLate) tLate = ti;
	    if (!i || ti < tEarly) tEarly =ti;
	}
	if (getDebug()) cerr << endl;
    }
    return 0;
}

//======================================  Fill requested series containers.
//    Error codes:
//      0  Successful completion
//     -1  Frame start not contiguous to previous data
//     -2  Sample rate isn't compatible with previous data.
//     -3  Requested data not found in current frame 
//     -4  Error reading frame
//     -5  Frame data are not self-consistent.
//     -6  TSeries is not allocated.
//     -7  Unsupported data type
//     -8  Interrupter by signal.
int 
MultiDacc::fillData(Interval Stride, bool start) {
    int rc = synch();
    if (rc) return rc;
    size_type N = mIn.size();
    for (size_type i=0; i<N; i++) {
	int rci = mIn[i]->fillData(Stride, start);
	if (rci != 0) rc = rci;
    }
    return rc;
}

//======================================  Fill data from a single frame
void
MultiDacc::zeroChans(Interval dT) {
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	mIn[i]->zeroChans(dT);
    }
}

//======================================  Fill data from a single frame
int
MultiDacc::fillChans(Interval Offset, Interval dT) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i<N; ++i) {
	rc = mIn[i]->fillChans(Offset, dT);
    }
    return rc;
}

int
MultiDacc::getDaccIndex(const string& name) {
    chan_iter i=findChannel(name); 
    if (i != mChanList.end()) {
	int index=i->getIndex();
	if(index==-1)
	    //--------------------------  Take the easy way out for now
	    return 0;
	else
	    return index;
    }
    return -1;
}

//======================================  Flush the specified data
int 
MultiDacc::flush(Interval Stride) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	rc = mIn[i]->flush(Stride);
    }
    return rc;
}

//======================================  Open a file for input.
int
MultiDacc::seek(Time STime) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	rc = mIn[i]->seek(STime);
    }
    return rc;
}

//======================================  Get a reference to the named Channel
TSeries* 
MultiDacc::refData(const string& name) {
    chan_iter chaninx = findChannel(name);
    if (chaninx != mChanList.end()) {
	int id = chaninx->getIndex();
	if (id != -1) return mIn[id]->refData(chaninx->getName());
    }
    return (TSeries*) 0;
}

FSeries* 
MultiDacc::refFData(const string& name) {
    chan_iter chaninx = findChannel(name);
    if (chaninx != mChanList.end()) {
	int id = chaninx->getIndex();
	if (id != -1) return mIn[id]->refFData(chaninx->getName());
    }
    return (FSeries*) 0;
}
