/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIG_SEG_ACCOUNTANT_HH
#define TRIG_SEG_ACCOUNTANT_HH

#include "Time.hh"
#include "Interval.hh"
#include <string>
#include <vector>
#include <iostream>

class TrigClient;

namespace trig {

    class Segment;

    /**  The SegAccountant class performs segment accounting for a single
      *  segment generation process. Specifically, it:
      *  <ul>
      *    <li>Allows user to define multiple segments.</li>
      *    <li>Tracks segment status and whether it is defined.</li>
      *    <li>Floshes segment according to one of three different modes</li>
      *    <li> Allows user to define multiple segments.</li>>
      *  </ul>
      *
      *  Usage:
      *
      *  Initialization:
      *
      *  \verbatim
      *  SegAccountant sa(tc); // Construct
      *
      *  sa.addSegment("SegName", 1, kAligned, 16.0);
      *  \endverbatim
      *
      *  Stride processing;
      *
      *  \verbatim
      *  sa.start_stride(ts.getStartTime());
      *  ...
      *  
      *  ...
      *  sa.update(ts.getEndTime());
      *  \endverbatim
      *
      *  \version 1.0; Last modified Jan 5, 2009
      *  \author  John G. zweizig
      */
    class SegAccountant {
    public:
	/** Enumerate the segment flush modes.
	 */
	enum flush_mode {
	    kNone,    ///< Allow segments to accumulate until the segment 
	              ///< changes state or until the accountant is deleted.
	    kMaxTime, ///< impose a maximum time period on all segments.
	    kAligned, ///< Flush at regular boundaries.
	    kWriteThrough ///< write each segment immediately.
	};

	///  Index data type definition.
	typedef unsigned long index_type;

	/** Segment identifier.
	  */
	class seg_id {
	public:
	    /**  Construct and empty segment id.
	      *  \brief Default constructor.
	      */
	    seg_id(void);

	    /**  Construct a segment information entry for a named segment.
	      *  Online segments should have version 1.
	      *  \brief Construct a segment information entry.
	      *  \param name Segment name
	      *  \param version Segment version number
	      */
	    seg_id(const std::string& name, int version=1);

	    /** Destructor
	     */
	    virtual ~seg_id(void);

	    /**  Get the full segment name (\e i.e. \c ifo:name).
	      *  \brief Full segment name. 
	      *  \return String containing the full segment name string.
	     */
	    std::string full_name(void) const;

	    /**  Get the ifo name.
	      *  \brief Segment ifo name. 
	      *  \return Pointer to the segment ifo name string.
	      */
	    const char* ifo(void) const;

	    /**  Get the segment name.
	      *  \brief Segment name. 
	      *  \return Pointer to the segment name string.
	     */
	    const char* name(void) const;

	    /**  Set the ifo.
	      *  \brief Set the sgment ifo name.
	      *  \param ifo Segment IFO string.
	      */
	    void setIfo(const std::string& ifo);

	    /**  Get the segment version number.
	      *  \brief Segment version. 
	      *  \return Version number.
	      */
	    int version(void) const;

	    /**  Test for equal segment id
	      *  \brief Compare segment IDs
	      *  \param si Id to be compared to this instance.
	      *  \return True if this instance is equal
	     */
	    bool operator==(const seg_id& si) const;

	    /**  Test for equal segment name
	      *  \brief Compare segment names
	      *  \param si Name string to be compared to this instance.
	      *  \return True if this name is equal
	     */
	    bool operator==(const std::string& si) const;

	    /**  Test for unequal segment id
	      *  \brief Compare segment IDs
	      *  \param si Id to be compared to this instance.
	      *  \return True if IDs are not equal.
	     */
	    bool operator!=(const seg_id& si) const;

	    /**  Test for unequal segment name
	      *  \brief Compare segment names
	      *  \param si Segment name to be compared to this instance.
	      *  \return True if names are unequal.
	     */
	    bool operator!=(const std::string& si) const;

	    /**  Test for segid less.
	      *  \brief Compare segment IDs
	      *  \param si Id to be compared to this instance.
	      *  \return True if this instance is less
	      */
	    bool operator<(const seg_id& si) const;

	    /**  Return a constant reference to the Ifo name.
	      *  \brief Reference segment ifo name.
	      *  \return Constant reference to the segment ifo string.
	     */
	    const std::string& refIfo(void) const;

	    /**  Return a constant reference to the segment name;
	      *  \brief Reference segment name.
	      *  \return Constant reference to the segment name string
	      */
	    const std::string& refName(void) const;

	private:
	    std::string _segname;
	    std::string _ifo;
	    int         _version;
	};

	/**  Segment information structure.
	  */
	class seg_info : public seg_id {
	public:
	    /**  Construct a segment information entry for a named segment.
	      *  Online segments should have version 1.
	      *  @memo construct a segment information entry.
	      *  \param sid Segment ID structure.
	      *  \param comment Segmenet description string
	      *  \param maxt    Maximum segment length.
	      */
	    seg_info(const seg_id& sid, const std::string& comment, 
		     Interval maxt=0.0);

	    /**  Destroy a segment information entry.
	      *  Online segments should have version 1.
	      */
	    ~seg_info(void);

	    /**  Get the last update time.
	      *  \brief Last update time
	      *  \return Time of last update.
	      */
	    const Time& last(void) const;

	    /**  Write the current segment to the trigger client. If the 
	      *  end parameter is not null (Time(0)), the segment up to 
	      *  the specified end time is writeen and that portion is
	      *  deleted from the current segment.
	      *  \brief write the current segment
	      *  \param tc Trigger client to write the segment to.
	      *  \param end Last time to write a trigger.
	      */
	    void seg_write(TrigClient* tc, const Time& end=Time(0));

	    /**  Set the state of this segment between the specified start
	      *  and end times If the segment changes state the start segment
	      *  state is written to the specified trigger client.
	      *  \brief Set a segment
	      *  \param tc Trigger cliento to which the segment is to be 
	      *            written if its state changes.
	      *  \param start Segment interval start time.
	      *  \param end   Segment interval end time.
	      *  \param onoff Segment State during the specified interval.
	      */
	    void set_segment(TrigClient* tc, const Time& start, 
			     const Time& end, bool onoff);

	    /**  Get the segment start time.
	      *  \brief Segment start time.
	      *  \return Segment start time.
	      */
	    const Time& start(void) const;

	    /**  Test the current state of the segment.
	      *  \brief Segment state.
	      *  \return True if segment is ON.
	      */
	    bool test(void) const;

	    /**  Write out the current segment.
	      *  \brief Update the spciefied segment.
	      *  \param tc Trigger client.
	      *  \param end Update time.
	      */
	    void update(TrigClient* tc, const Time& end=Time(0));

	private:
	    //--------------------------  Segment parameters
	    std::string _comment;
	    Interval    _maxtime;

	    //--------------------------  Segment status.
	    Time        _start;
	    Time        _last;
	    bool        _state;
	};

    public:
	/**  Construct an unattached segment accountant.
	  *  \brief Default constructor.
	  */
	SegAccountant(void);

	/**  Construct a segment accountant and attach it to the specified 
	  *  trigger client.
	  *  \brief Default constructor.
	  *  \param tc Trigger client
	  */
	SegAccountant(TrigClient& tc);

	/**  Destroy a segment accountant.
	  *  \brief Default constructor.
	  */
	~SegAccountant(void);

	/**  Define a segment type to be accounted. Then name, version, 
	  *  comment string and maximum segment length are specified.
	  *  \brief Add a segment type.
	  *  \param name    Segment type name.
	  *  \param version Segment Type version number.
	  *  \param comment Segment comment string.
	  *  \param max_time Maximum interval for the segment type.
	  */
	void addSegment(const std::string& name, int version=1, 
			const std::string& comment="", Interval max_time=0.0);

	/**  Define a segment type to be accounted. Then segment identifier, 
	  *  comment string and maximum segment length are specified.
	  *  \brief Add a segment type.
	  *  \param name    Segment seg_id.
	  *  \param comment Segment comment string.
	  *  \param max_time Maximum interval for the segment type.
	  */
	void addSegment(const seg_id& name, const std::string& comment="", 
			Interval max_time=0.0);

	/**  Close and delete the attached trigger client.
	  *  \brief Close the segment accountant.
	  */
	void close(void);

	/**  Dump out the contents of the segment list to the specified 
	  *  output stream.
	  *  \brief Dump %Seg Accountant contents
	  *  \param out Output stream reference.
	  *  \return Output straem reference.
	  */
	std::ostream& dump(std::ostream& out) const;

	/**  Find the specified segment, return a reference. If no segment
	  *  is found matching the specified ID, an exception is thrown.
	  *  \brief Find information for a segment.
	  *  \param sid Identified for the segment type to be found.
	  *  \return seg_info reference.
	  */
	seg_info& find(const seg_id& sid);

	/**  Return a string describing the mode.
	  *  \brief Convert flush mode to string.
	  *  \param fm Mode flag to be converted.
	  *  \return Pointer to flush mode name string.
	  */
	static const char* flush_mode_to_char(flush_mode fm);

	/**  Return a mode name by the string.
	  *  \brief Convert flush mode string to enumerated value.
	  *  \param fc Mode string to be converted.
	  *  \return Flush mode enumerated value.
	  */
	static flush_mode flush_char_to_mode(const std::string& fc);

	/**  Find the specified segment, return an index into the list of
	  *  the specified segment or the index at which the segment would
	  *  be inserted.
	  *  \brief Locate segment information.
	  *  \param sid Segment identified to be located
	  *  \return Index of requested segment.
	  */
	index_type locate(const seg_id& sid);

	/**  Set the segment flush mode and timing.
	  *  \brief Set Flush mode.
	  *  \param m Flush mode.
	  *  \param t Flush interval.
	  */
	void set_mode(flush_mode m, Interval t=0);

	/**  Set segment state in specified time interval.
	  *  \brief Set segment value.
	  *  \param si Name of segment type to be set.
	  *  \param start Segment start time.
	  *  \param end   Segment end time.
	  *  \param onoff Segment value.
	  */
	void set_segment(const std::string& si, const Time& start, 
			 const Time& end, bool onoff);

	/**  Set segment state in specified time interval.
	  *  \brief Set segment value.
	  *  \param si    Segment type identifier.
	  *  \param start Segment start time.
	  *  \param end   Segment end time.
	  *  \param onoff Segment value.
	  */
	void set_segment(const seg_id& si, const Time& start, 
			 const Time& end, bool onoff);

	/**  Set up at start of analysis segment.
	  *  \brief Start a stride
	  *  \param ts Stride start time.
	  */
	void start_stride(const Time& ts);

	/**  Get the time for the next file to be written
	  *  \brief Get next file time.
	  *  \param tup Update time
	  *  \return %Time at which the next file is to be written.
	  */
	Time test_write(const Time& tup) const;

	/**  Update to the specified time.
	  *  \brief Update to time \a tup.
	  *  \param tup Update time.
	  */
	void update(const Time& tup);

    private:
	typedef std::vector<seg_info>    seg_list;
	typedef seg_list::iterator       seg_iter;
	typedef seg_list::const_iterator const_seg_iter;

    private:
	seg_list    mList;
	TrigClient* mTC;
	flush_mode  mFlush;
	Interval    mFlushInterval;
	Time        mStart;
	Time        mLastUpdate;
	Time        mLatest;
    };

    //==================================  SegAccountant::seg_info inlines
    inline const char*
    SegAccountant::seg_id::ifo(void) const {
	return _ifo.c_str();
    }

    inline const char*
    SegAccountant::seg_id::name(void) const {
	return _segname.c_str();
    }

    inline const std::string& 
    SegAccountant::seg_id::refIfo(void) const {
	return _ifo;
    }

    inline const std::string& 
    SegAccountant::seg_id::refName(void) const {
	return _segname;
    }

    inline int 
    SegAccountant::seg_id::version(void) const {
	return _version;
    }

    inline bool 
    SegAccountant::seg_id::operator==(const seg_id& si) const {
	return _segname == si._segname && _version == si._version;
    }

    inline bool 
    SegAccountant::seg_id::operator!=(const seg_id& si) const {
	return ! operator==(si);
    }

    inline bool 
    SegAccountant::seg_id::operator==(const std::string& si) const {
	return full_name() == si;
    }
 
    inline bool 
    SegAccountant::seg_id::operator!=(const std::string& si) const {
	return ! operator==(si);
    }

    inline std::string
    SegAccountant::seg_id::full_name(void) const {
	if (_ifo.empty()) return _segname;
	return _ifo + ":" + _segname;
    }

    inline bool 
    SegAccountant::seg_id::operator<(const seg_id& si) const {
	if (_ifo  < si._ifo) return true;
	if (_ifo != si._ifo) return false;
	if (_segname  < si._segname) return true;
	if (_segname != si._segname) return false;
	if (_version  < si._version) return true;
	return false;
    }

    inline std::ostream&
    operator<<(std::ostream& out, const SegAccountant::seg_id& si) {
	return out << si.refIfo() << ":" << si.name() << ":" << si.version();
    }

    inline const Time&
    SegAccountant::seg_info::last(void) const {
	return _last;
    }

    inline const Time& 
    SegAccountant::seg_info::start(void) const {
	return _start;
    }

    inline bool 
    SegAccountant::seg_info::test(void) const {
	return _state;
    }

    //======================================  write segment
    inline std::ostream&
    operator<<(std::ostream& out, const SegAccountant::seg_info& s) {
	out << "segment: " << dynamic_cast<const SegAccountant::seg_id&>(s)
	    << " " << s.start().getS() << "-" << s.last().getS();
	return out;
    }
}

#endif  // !defined(TRIG_SEG_ACCOUNTANT_HH)
