/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "PConfig.h"
#include "matchInterface.hh"
#include "sockutil.h"
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <cstring>
#include <unistd.h>

// #define MIF_DEBUG 1
#ifdef MIF_DEBUG
#include <iostream>
#include <iomanip>
using namespace std;
#endif

namespace framexmit {

   //===================================  Get a list of interfaces.
   bool 
   getInterfaces(int sock, interfaceList& iList)  {
      // clear list
      iList.clear();

      // get list of interfaces
      ifconf 	ifc;
      char 	buf[2048];
      ifc.ifc_len = sizeof (buf);
      ifc.ifc_buf = buf;
      if (ioctl (sock, SIOCGIFCONF, (char*) &ifc) < 0) {
         return false;
      }
   
      // obtaining interface flags
      ifreq*	ifr = ifc.ifc_req;
      for (int n = ifc.ifc_len / sizeof (ifreq); --n >=0; ifr++) {
         // inet interfaces only
         if (ifr->ifr_addr.sa_family != AF_INET) {
            continue;
         }
      	 // get flags
         if (ioctl (sock, SIOCGIFFLAGS, (char*) ifr) < 0) {
            return -1;
         }
      	 // skip unintersting cases
         if (((ifr->ifr_flags & IFF_UP) == 0) ||
	     ((ifr->ifr_flags & IFF_LOOPBACK) != 0) ||
	     ((ifr->ifr_flags & (IFF_BROADCAST | IFF_POINTOPOINT)) == 0)) {
            continue;
         }
         interface_t iface;
         strcpy (iface.name, ifr->ifr_name);
         iface.addr = ((sockaddr_in*)(&ifr->ifr_addr))->sin_addr.s_addr;
         iList.push_back (iface);
      } 

      return true;
   }

   //===================================  Match interface to network.
   bool 
   matchInterface (int sock, const char* net, in_addr& i_addr)
   {
      //--------------------------------  Get net address in host ordering
      in_addr_t net_ip(~0);
      if (net)  net_ip = ntohl(inet_addr(net));

      //--------------------------------  If net address not specified
      if ( net_ip == in_addr_t(~0) ) {
         // use default host address
         char		myname[256];	// local host name
         if (gethostname(myname, sizeof(myname)) || nslookup(myname, &i_addr)){
            return false;
         }
      }

      //--------------------------------  Search for specified address.
      else {
         // get a list of all interface addresses
         interfaceList iList;
         if (!getInterfaces (sock, iList)) {
            return false;
         }

         // go through interface list and take closest match
         i_addr.s_addr = 0;
         unsigned int diff = ~0;
         for (iflist_iter i = iList.begin(); i != iList.end(); i++) {
#ifdef MIF_DEBUG
	    cerr << "Compare net addresses " << hex << ntohl(i->addr) << " " 
		 << net_ip << endl;
#endif
	    uint32_t diff_i =  ntohl(i->addr) - net_ip;
            if ((ntohl(i->addr) > net_ip) && (diff_i < diff)) {
               i_addr.s_addr = i->addr;
               diff          = diff_i;
            }
         }
#ifdef MIF_DEBUG
	 cerr << "Selected net addresses " << ntohl(i_addr.s_addr) << dec
	      << endl;
#endif
         if (!i_addr.s_addr) return false;
      }
      return true;
   }

}  // namespace framexmit
