#ifndef ALARMDATA_HH
#define ALARMDATA_HH

#include "Time.hh"
#include "Interval.hh"
#include <string>
#include <stdexcept>
//#include <sstream>

namespace lmsg {
    class TransOutput;
    class TransInput;
}

//======================================  Alarm handle

/**  The AlarmHandle class defines an identifier used by the Alarm Manager
  *  to refer to a specific active alarm.
  *  @memo Active alarm identifier.
  *  @author J. Zweizig
  *  @version $Revision: 6273 $; Modified $Date: 2010-08-01 00:47:51 -0700 (Sun, 01 Aug 2010) $
  */
class AlarmHandle {
public:
    /**  Construct a handle. THe handle is initialized to an invalid
      *  (not defined) state.
      *  @memo construct a handle.
      */
    AlarmHandle(void);

    /**  Copy an alarm handle.
      *  @memo Copy constructor.
      *  @param x Alarm handle to be copied.
      */
    AlarmHandle(const AlarmHandle& x);

    /**  Hate to do this, but it's really hard to get around it.
     */
    explicit AlarmHandle(long x) : mHandle(x) {}

    /**  Destroy an alarm handle.
      *  @memo Destructor.
      */
    ~AlarmHandle(void);

    /**  Alarm handle assignment operator.
      *  @memo Assignment operator.
      *  @param x reference to a Handle to be copied.
      *  @return Reference to the lhs object.
      */
    AlarmHandle& operator= (const AlarmHandle& x);

    /**  Comparison operator (less than).
      *  @memo Compare alarm handles
      *  @param x Reference to alarm handle to compare to this
      *  @return True if this is less than the argument handle.
      */
    bool operator< (const AlarmHandle& x) const;

    /**  Comparison operator (equal to)
      *  @memo Compare for equality.
      *  @param x Alarm handle to compare to this.
      *  @return Trus if handles are equal.
      */
    bool operator==(const AlarmHandle& x) const;

    /**  Test for invalid state.
      *  @memo Test for invalid state.
      *  @return true if handle is not defined or invalid.
      */
    bool operator!(void) const;

    /**  Generate a unique handle.
      *  @memo generate a unique handle
      *  @return Unique handle.
      */
    static AlarmHandle getUnique(void);

    /**  Get a unique integer equicvalent ot the handle.
      *  @memo Get integer version of handle.
      *  @return integer with handle encoded.
      */
    long getInt(void) const;

    friend lmsg::TransInput& operator>>(lmsg::TransInput&, AlarmHandle&) throw(std::runtime_error);

private:
    long mHandle;
    static long mNext;
};

/**  Data translation object used by message handler.
 */
lmsg::TransOutput& operator<<(lmsg::TransOutput&, const AlarmHandle&);


//======================================  AlarmHandle inline functions
#ifndef __CINT__

inline 
AlarmHandle::AlarmHandle(void) 
  : mHandle(0) 
{}

inline 
AlarmHandle::AlarmHandle(const AlarmHandle& x) 
  : mHandle(x.mHandle) 
{}

inline 
AlarmHandle::~AlarmHandle(void) {}

inline AlarmHandle& 
AlarmHandle::operator= (const AlarmHandle& x) {
    mHandle = x.mHandle;
    return *this;
}

inline bool 
AlarmHandle::operator< (const AlarmHandle& x) const {
    return mHandle < x.mHandle;
}

inline bool 
AlarmHandle::operator! (void) const {
    return !mHandle;
}

inline bool 
AlarmHandle::operator==(const AlarmHandle& x) const {
    return mHandle == x.mHandle;
}

inline long
AlarmHandle::getInt(void) const {
    return mHandle;
}

#endif // !defined(__CINT__)

//======================================  Alarm ID
/**  The AlarmID class identified an alarm type. The ID contains the name
  *  of the monitor that produces the alarm type and an type name. The 
  *  AlarmID may contain a limited regular expression in one or both of 
  *  the fields.
  *  @memo Alarm Type Identifier
  *  @author J. Zweizig
  *  @version $Revision: 6273 $; Modified $Date: 2010-08-01 00:47:51 -0700 (Sun, 01 Aug 2010) $
  */
class AlarmID {
public:
    /**  Construct an empty ID.
      *  @memo Default constructor.
      */
    AlarmID(void) {}

    /**  Construct an alarm ID with the specified ID and Monitor name.
      *  @memo Data constructor.
      *  @param ID      Alarm name.
      *  @param Monitor Monitor name.
      */
    AlarmID(const std::string& ID, const std::string& Monitor);

    /**  Construct an identical AlarmID.
      *  @memo Copy constructor
      *  @param x AlarmID to be copied.
      */
    AlarmID(const AlarmID& x) {operator=(x);}

    /** Destructor.
      */
    virtual ~AlarmID(void);

    /**  Assignment operator.
      *  @memo Assignment operator.
      *  @param x AlarmID to be copied.
      *  @return Referene to this AlarmID
      */
    AlarmID& operator=(const AlarmID& x);

    /**  Compare an AlarmID to another ID. The comparison is true if both
      *  the monitor name and alarm name are identical.
      *  @memo Comparison operator.
      *  @return True if alarmIDs are identical.
      *  @param x AlarmID to which the current ID is compared.
      */
    virtual bool operator==(const AlarmID& x) const;


    /**  Compare an AlarmID to a limited regular expression. The two fields
      *  are compared individually and the result is true if both fields
      *  match tehir respective regular expressions.
      *  @memo RegExp comparison.
      *  @return True if alarmID matches the regular expression.
      *  @param regexp Regular expression to which the current ID is compared.
      */
    bool match(const AlarmID& regexp) const;

    /**  Get a constant pointer to the alarm name.
      *  @memo Get alarm name.
      *  @return Constant pointer to the alarm name.
      */
    const char* getAlarm(void) const;

    /**  Get a constant pointer to the monitor name.
      *  @memo Get alarm name.
      *  @return Constant pointer to the monitor name.
      */
    const char* getMonitor(void) const;

    /**  Set the alarm name.
      *  @memo Set alarm name.
      *  @param id Constant string containing the new alarm name.
      */
    void        setAlarm(const std::string& id);

    /**  Set the monitor name.
      *  @memo Set monitor name.
      *  @param mon Constant string containing the new monitor name.
      */
    void        setMonitor(const std::string& mon);
private:
    std::string mMonitor;
    std::string mAlarmID;
};

/**  Data translation functions used by message handler.
 */
lmsg::TransOutput& operator<<(lmsg::TransOutput&, const AlarmID&);
lmsg::TransInput&  operator>>(lmsg::TransInput&, AlarmID&) throw(std::runtime_error);

//======================================  Inline functions
#ifndef __CINT__
inline bool
AlarmID::operator==(const AlarmID& x) const {
    return mMonitor == x.mMonitor && mAlarmID == x.mAlarmID;
}

inline const char*
AlarmID::getAlarm(void) const {
    return mAlarmID.c_str();
}

inline const char*
AlarmID::getMonitor(void) const {
    return mMonitor.c_str();
}
#endif  // !defined(__CINT__)

/**  The alarm data class contains the description of the Alarm and 
  *  all associated parameters. The AlarmData is used to define an
  *  alarm type, to set a specific alarm, and to return the current
  *  alarm status. The Alarm description contains the following fields:
  *  <ul>
  *    <li> \b AlarmID
  *       The alarm descriptor is based on an AlarmID. This gives the
  *       alarm name and group name.
  *    </li>
  *    <li> <b> Timeout period</b>
  *       The timeout period specifies the delay before an alarm expires. 
  *       When defining an alarm, this field contains the default interval,
  *       when issuing an alarm this field contains the interval to be 
  *       used (or zero for the default) and when testing the alarm status,
  *       this field contains the remaining time before the alarm expires.
  *    </li>
  *    <li> \b Severity
  *       This field contains the alarm severity.
  *    </li>
  *    <li> \b Description
  *       This string contains a description of the alarm or the file name
  *       or URL where the description may be found. File names are prefixed
  *       with "file:" and URLs are in the standard form starting with 
  *       "http:".
  *    </li>
  *    <li> <b> Modification Parameters </b>
  *       This string contains a list of parameter names and values that
  *       are to be substituted for the escaped parameter names in the 
  *       description string.
  *    </li>
  *    <li> <b> Start Time</b>
  *       The start time is recorded by the alarm manager and returned 
  *       with a status request. It is ignored in Alarm definition and 
  *       issuance requests.
  *    </li>
  *  </ul>
  *  @memo Alarm description.
  *  @author John Zweizig
  *  @version $Revision: 6273 $; Modified $Date: 2010-08-01 00:47:51 -0700 (Sun, 01 Aug 2010) $
  */
class AlarmData : public AlarmID {
public:
    /** Data type used for the Flag word.
      */
    typedef unsigned long flag_t;

    /**  Alarm status word bit assignments.
     */
    enum StatusFlags {
        kNone=0,              ///< Idle
        kAcknowledged=1,      ///< Alarm has been acknowledged
	kNeedsAcknowledge=2,  ///< Alarm requires acknowledgment
	kReTrigger=4,         ///< Only one alarm of this type
	kDisable=8            ///< Ignore alarms of this type
    };

public:
    /**  Construct an empty alarm decription object.
      *  @memo Default constructor.
      */
    AlarmData(void);

    /**  Construct an alarm descriptor identical to and existing object.
      *  @memo Copy constructor.
      *  @param x Alarm data to be copied.
      */
    AlarmData(const AlarmData& x);

    /**  Construct an alarm description object with specified parameters.
      *  @memo Data constructor.
      *  @param Monitor Group name.
      *  @param ID      Alarm name.
      *  @param timeout Alarm timeout delay.
      *  @param severe  Alarm severity.
      *  @param desc    Description string, file name or URL.
      *  @param params  Alarm instance parameter values.
      */
    AlarmData(const std::string& Monitor, const std::string& ID, 
	      Interval timeout=0.0, int severe=0, const std::string& desc="",
	      const std::string& params="");

    /**  Destroy an alarm descriptor.
      *  @memo Alarm destructor.
      */
    virtual ~AlarmData(void) {}

    /**  Alarm descriptor assignment operator.
      *  @memo Assignment operator.
      *  @param x Alarm data to be copied.
      */
    AlarmData& operator=(const AlarmData& x);

    /**  Alarm descriptor comparator.
      *  @memo Comparison operator.
      *  @param x Data to be compared.
      */
    bool operator==(const AlarmData& x);

    /**  Alarm descriptor comparator.
      *  @memo Comparison operator.
      *  @param id Data to be compared.
      */
    bool operator==(const AlarmID& id) const {
       return this->AlarmID::operator==(id); }

    /**  Append a parameter to the parameter string. A blank space is used 
      *  to seperate the current sting and the new paramter. If the specified
      *  parameter contains spaces, the parmaeter is enclosed in quotes.
      *  @memo Add a parameter to the parameter string.
      *  @param param Parameter string.
      */
    void addParameter(const std::string& param);

    /**  Clear one or more modifier flags.
      *  @memo Clear a modiifier flag.
      *  @param flags Flag to be cleared.
      */
    void clearFlags(flag_t flags);

    /**  Clear the parameter string.
      *  @memo Clear the parameter string.
      */
    void clearParameters(void);

    /**  Get the short description of the alarm. The description may contain
      *  parameter identifiers (specified as '$n') which will be substituted
      *  by the parameters when the alarm is specified.
      *  @memo Get the alarm description.
      */
    const char* getDescription(void) const;

    /**  Get the alarm expiration time.
      *  @memo Get the alarm expiration time.
      */
    Time getExpire(void) const;

    /**  Get the alarm flag field.
      *  @memo Get the alarm flags.
      */
    flag_t getFlags(void) const;

    /**  Get the multimedia file name.
      *  @memo Get the alarm multimedia file name.
      */
    const char* getMultiMedia(void) const;

    /**  Get the alarm parameter specifier.
      *  @memo Get the alarm parameter string.
      */
    const char* getParameters(void) const;

    /**  Get the alarm Web accessible help-file specifier. The syntax is 
      *  described in.
      *  @memo Get the web based help file name.
      */
    const char* getWebFile(void) const;

    /**  Get the alarm severity.
      *  @memo Get the alarm severity.
      *  @return The alarm severity.
      */
    int getSeverity(void) const;

    /**  Get the alarm start time.
      *  @memo Get the alarm time.
      */
    Time getStartTime(void) const;

    /**  Get the alarm Timeout period or remaining time.
      *  @memo Get the alarm timeout .
      */
    Interval getTimeout(void) const;

    /**  Test the alarm acknowledged flag.
     */
    bool isAcknowledged(void) const;

    /**  Or specified flags into alarm flag field.
      *  @memo Set specified flags.
      *  @param flags Flag mask to be set.
      */
    void jamFlags(flag_t flags);

    /**  Set the alarm severity to the minimum of the current and the
      *  specified severity.
      *  @memo Lower the severity.
      *  @param severe Minimum severity
      */
    void minSeverity(int severe);

    /**  Add the specified duration to the alarm timeout.
      *  @memo Prolong the alarm.
      *  @param dT Prolongation time.
      */
    void prolong(Interval dT);

    /**  Set/Clear the acknowledge bit.
      *  @memo Set the start timeout.
      *  @param onoff New ackknowlege bit state.
      */
    void setAcknowledge(bool onoff=true);

    /**  Set the alarm description file.
      *  @memo Set the alarm description file.
      *  @param desc Alarem description file path.
      */
    void setDescription(const std::string& desc);

    /**  Set the multimedia file name.
      *  @memo Set the multimedia file name.
      *  @param name Multimedia file path.
      */
    void setMultiMedia(const std::string& name);

    /**  Set the alarm parameter string.
      *  @memo Set the parameter string.
      *  @param param Parameter string.
      */
    void setParameters(const std::string& param);

    /**  Set the start Time.
      *  @memo Set the start timeout.
      *  @param t Alarm start time.
      */
    void setStartTime(const Time& t);

    /**  Set the Timeout interval.
      *  @memo Set the timeout interval.
      *  @param dt Alarm timeout duration.
      */
    void setTimeout(Interval dt);

    /**  Set the web-based help file name for the alarm.
      *  @memo Set web file name.
      *  @param name Web file path
      */
    void setWebFile(const std::string& name);

private:
    Interval    mTimeout;
    Time        mStart;
    int         mSeverity;
    flag_t      mFlags;
    std::string mDescribe;
    std::string mParams;
    std::string mWebFile;
    std::string mMultiMedia;
};

/**  Data translation functions used by message handler.
 */
lmsg::TransOutput& operator<<(lmsg::TransOutput&, const AlarmData&);
lmsg::TransInput&  operator>>(lmsg::TransInput&, AlarmData&) throw(std::runtime_error);

//======================================  Inline methods
#ifndef __CINT__

inline const char*
AlarmData::getDescription(void) const {
    return mDescribe.c_str();
}

inline Time
AlarmData::getExpire(void) const {
    return mStart + mTimeout;
}

inline AlarmData::flag_t
AlarmData::getFlags(void) const {
    return mFlags;
}

inline const char*
AlarmData::getMultiMedia(void) const {
    return mMultiMedia.c_str();
}

inline const char*
AlarmData::getParameters(void) const {
    return mParams.c_str();
}

inline int
AlarmData::getSeverity(void) const {
    return mSeverity;
}

inline Time
AlarmData::getStartTime(void) const {
    return mStart;
}

inline Interval
AlarmData::getTimeout(void) const {
    return mTimeout;
}

inline const char*
AlarmData::getWebFile(void) const {
    return mWebFile.c_str();
}

inline bool
AlarmData::isAcknowledged(void) const {
    return (mFlags & kAcknowledged) != 0;
}

#endif // !defined(__CINT__)

#endif // !defined(ALARMDATA_HH)
