#ifndef EQMON_HH
#define EQMON_HH
#define EQMONVERSION "1.2 (Last modified Sat Dec 8, 2001)"

//I'm only including headers necessary for the .hh file here; rest are in .cc
#include <vector>
#include <fstream>
#include "DatEnv.hh"            // Standard DMT libraries
#include "TrigClient.hh"
#include "TSeries.hh"
#include "TrigPar.hh"

/// A vector of floats
typedef std::vector<float> Fvector;
/// An iterator to a Fvector
typedef Fvector::iterator FvIt;
/// A constant iterator to a Fvector
typedef Fvector::const_iterator FvCIt;
/// A class which contains all the data for a channel
class ChannelStatus;

/// changes an integer into a string
template<class T> std::string tostring(const T& n);
/// averages a subsample of the data (every 'step' datapoints)
float getAvg(const int, FvCIt, const int);
/// computes the standard deviation for a subsample of the data
float getSDev(const int, FvCIt, const float, const int);

/** This monitor checks the seismometers and tiltmeters for unusually long periods 
  * of noisy data.  These are considered "earthquakes", and interferometer data 
  * obtained during this time should not be used.
  * @memo An earthquake monitor.
  * @author Masahiro Ito and Rauha Rahkola
  * @version 1.2 (Last modified Sat Dec 8, 2001)
  */
class eqMon : public DatEnv, TrigClient
{
private:
  TSeries **ts;

  /// Config file parameters, for the most part
  int NChannel,count,thNStations,maxNeqStation,maxCount,sample;
  /// final sample rate of each channel (after averaging, power of 2)
  const int sampRate;
  /// number of samples in recent and total data, resp.
  int numRecent, numAveraged;
  /// internal parameters for calculations
  float threshold,threshAlt,norm,decay;
  /** eqON is a 3-state flag, tdataMax= size of the history vector, others are 
    * config params
    */
  int eqON,tdataMax, waitTime,durationTime;
  /// internal flags for I/O control
  bool triggerOn,channelsOn,fullData,verbose, quiet;
  /// 'eqStartTime' refers to the start of a noisy period, not necessarily an event
  Time baseTime,eqStartTime,eqEndTime;
  /// Parameter Dictionary for config Parameters
  TrigPar mDict;
  /// Config file test variable
  std::string ConfigName;
  /// Channel vector test variable
  std::vector<std::string> mChans;
  ///  The output filenames
  std::string mDataName;
  std::string mEventName;
  std::ofstream dataFile;
  std::ofstream eventFile;
  ///  Array of channel info
  ChannelStatus *chStat;

  /// Actually performs the calculations for sending triggers
  void eventCheck();
  /** Checks whether the next word on the command line is an option or an 
    * argument for an option
    */
  bool chkNxtArg(int i, const char* argv[], const int argc);
  /// Prints the usage message
  int Usage(const char* progname);
  /// Prints the help info
  void Help();

public:
  /** Reads the configuration file, initializes variables and sets up the 
    * channels from Dacc.\\
    * {\bf Usage}: #eqMon [-conf <file>] [-verbose | -quiet] [-h | --help] #\\
    *
    * eqMon detects an earthquake roughly ~1min after it starts.
    * Various output verbosities are available.  Choose:
    * \begin{tabular}{ll}
    *   no options              & for updated mean, std dev for each channel 
    *                             (default)\\
    *   #-quiet#                & for no data (still sends triggers to EPICS)\\
    *   #-verbose#              & for monitor status\\
    *   #-debug [1]#            & for preventing system calls and status 
    *                             to stdout\\
    *   #-verbose -debug [1]#   & for all of the above output and basic 
    *                             channel stats\\
    *   #-debug {>1}#           & for timing and memory status\\
    *   #-verbose -debug {>1}#  & for all of the above output and full
    *                              channel stats\\
    * \end{tabular}
    * Please note that event data is always generated.
    *
    * {\large {\bf Configuration File}}
    * 
    * Example configuration file:
    * \begin{verbatim}
Parameter EventPrefix ./data/test_event
Parameter DataPrefix ./data/test_data
Parameter MaxThresh 4.5
Parameter MinThresh 2.0
Parameter TriggerOn 1
Parameter ChanThresh 7
H0:PEM-EX_SEISX
H0:PEM-EX_SEISY
H0:PEM-EX_SEISZ
...
\end{verbatim}
    *
    * For an explanation of the Configuration file, please see the ReadConfig()
    * command.
    * @memo Constructor.
    */
  eqMon(int argc,const char *argv[]);

  /** Deletes the history, resets the EPICS earthquake alarm, and (optionally) 
    * prints out the final statistics.
    * @memo Destructor.
    */
  ~eqMon(void);

  /** eqMon first averages the data into 8Hz, then gathers 10 minutes of data 
    * before it is able to send triggers.  Each minute later, eqMon checks each
    * channel for noise (data has deviated from the mean by #<threshold># * 
    * std dev within the last #<wait_sec>#).  If a sufficient number of 
    * channels (>= #<num_chans>#) is noisy for a sufficient length of time 
    * (#<wait_sec> + <hold_sec>#), a trigger is sent (if #<triggerOn># is true)
    * -- an earthquake has started.  Once there is no longer a sufficient
    * number of noisy channels, the alarm is reset, and the earthquake has 
    * stopped.  When there is not an earthquake, the mean and std dev for 
    * the past 10 min. are updated each minute.
    * @memo Processes one second of data.
    */
  void ProcessData(void);
  /** Reads and parses the configuration file.  Lines in the file are separated
    * into two categories-- Parameters and Channels.
    *
    * {\bf Parameters}-- Parameter lines take the form: #Parameter <param> #
    * #<value>#, where #param# is one of the following:
    * \begin{itemize}
    *   \item #DataPrefix  -# Data filename prefix (start date is appended)
    *   \item #EventPrefix -# Event filename prefix (start date is appended)
    *   \item #MaxThresh   -# threshold to trigger (units of sigma)
    *   \item #MinThresh   -# threshold to release (units of sigma)
    *   \item #ChanThresh  -# minimum \# of channels needed to trigger an event
    *   \item #TriggerOn   -# trigger option (boolean)
    *   \item #HoldTime    -# \# of seconds to wait before reseting a channel's
    *                         event flag
    *   \item #WaitTime    -# \# of consecutive seconds before an event 
    *                         gets processed
    *   \item #SubSample   -# the speed (integer) at which to process avg and 
    *                         std dev of each channel (1=slowest,4=fastest)
    * \end{itemize}
    *
    * {\bf Channels}-- each desired channel should be entered on its own line.
    *@memo Reads and parses configuration file
    **/
  void ReadConfig(void);
};

/** Contains all the information needed for each channel (including history)
  * @author Masahiro Ito and Rauha Rahkola
  * @version 1.2 (Last modified Sat Dec 8, 2001)
  */
class ChannelStatus {
  friend class eqMon;
private:
  std::string name;
  float lastnorm;
  float sdev,mean,max,min;
  int signalON;
  Time endTime;
  Fvector tdata;
  FvIt place;
  FvIt tail;
  void reset(void){
    lastnorm=0;
    sdev=0;
    signalON=0;
  };
public:
  /// Basic constructor
  ChannelStatus(void) {reset();}
  /// Basic destructor
  ~ChannelStatus(void) {reset();}
};

#endif
