/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "dq-module.hh"
#include "DVecType.hh"
#include "Bits.hh"
#include "checksum_crc32.hh"
#include <sstream>
#include <fstream>
#include <iostream>
#include <stdexcept>
#include <algorithm>
#include <json/reader.h>
#include <json/value.h>
#include <dlfcn.h>

using namespace std;

EXECDAT(DQ_Module)

DQ_Module::DQ_Module(int argc, const char* argv[])
 : MultiStream(argc, argv), stride(4.0), flag_step(0.0625),
   parity_bit(true), invert(false), config_version("Not Specified"), 
   config_crc32(0), frame_path("$HOME/frames/X-DQ_MODULE-%05r", "X-DQ_MODULE"),
   compression("uncompressed"), output_start(0), current(0), frames_out(0)
{
   try {
      config_file = frame_name::resolve_env(argv[argc - 1]);
      config_file = frame_name::make_absolute(config_file);
      config(config_file);
      if (Debug()) put_config(cout);
   } catch (std::exception& e) {
      cerr << "Caught exception reading configuration file: " 
	   << e.what() << endl;
      finish();
      return;
   }
   getDacc().setStride(stride);
   setStrideAlignment(stride, 0.0);
   book_channels();
   fr_writer.setCompress(FrVectRef::get_compression_mode(compression));
}

DQ_Module::~DQ_Module(void) {
}

//======================================  convert bool to char*
const char*
boolstr(bool torf) {
   return torf ? "true" : "false";
}

//======================================  Print configuration info

void 
DQ_Module::put_config(ostream& out) const {
   out << "Monitor configuration" << endl;
   out << "config-file:    " << config_file << endl;
   out << "config-version: " << config_version << endl;
   out << "config-crc32:   " << config_crc32 << endl;
   out << "stride:         " << stride << endl;
   out << "parity-bit      " << boolstr(parity_bit) << endl;
   out << "invert          " << boolstr(invert) << endl;
   out << "flag_step:      " << flag_step << endl;
   out << "Frame_name:     " << frame_path.file_path(Time(0)) << endl;
   out << "compression:    " << compression << endl;
   out << "DQ configuration" << endl;
   size_t nFlags = flag_vect.size();
   for (size_t i=0; i<nFlags; i++) {
      flag_vect[i]->put_config(out);
      out << endl;
   }
   out << "DQ copy channels" << endl;
   size_t nChans = copy_channels.size();
   for (size_t i=0; i<nChans; i++) {
      out << "                " << copy_channels[i] << endl;
   }
   out << "DQ copy history" << endl;
   size_t nHist = copy_history.size();
   for (size_t i=0; i<nHist; i++) {
      out << "                " << copy_history[i] << endl;
   }
}

//=======================================  Json string element parse
bool
parse_json(const Json::Value& doc, const std::string& name, std::string& val) {
   if (!doc.isMember(name)) return false;
   val = doc[name].asString();
   return true;
}

//=======================================  Json double element parse
bool
parse_json(const Json::Value& doc, const std::string& name, double& val) {
   if (!doc.isMember(name) || !doc[name].isNumeric()) return false;
   val = doc[name].asDouble();
   return true;
}

//=======================================  Json bool element parse
bool
parse_json(const Json::Value& doc, const std::string& name, bool& val) {
   if (!doc.isMember(name) || !doc[name].isBool()) return false;
   val = doc[name].asBool();
   return true;
}

//=======================================  Json string vector parse
bool
parse_json(const Json::Value& doc, const std::string& name,
	   std::vector<std::string>& val) {
   if (!doc.isMember(name)) return false;
   Json::Value vec = doc[name];
   Json::ArrayIndex N = vec.size();
   val.clear();
   val.reserve(N);
   for (Json::ArrayIndex i=0; i < N; i++) {
      val.push_back(vec[i].asString());
   }
   return true;
}

//=======================================  Json double vector parse
bool
parse_json(const Json::Value& doc, const std::string& name,
	   std::vector<double>& val) {
   if (!doc.isMember(name)) return false;
   Json::Value vec = doc[name];
   Json::ArrayIndex N = vec.size();
   val.clear();
   val.reserve(N);
   for (Json::ArrayIndex i=0; i < N; i++) {
      if (!vec[i].isNumeric()) return false;
      val.push_back(vec[i].asDouble());
   }
   return true;
}


//=======================================  Parse json configuration
void 
DQ_Module::config(const std::string& cfg_file) {
   //------------------------------------  Read the configuration file.
   ifstream in(cfg_file.c_str());
   if (!in.is_open()) {
      throw runtime_error(string("unable to open file: ") + cfg_file);
   }
   string conf_str;
   getline(in, conf_str, (char)EOF);
   if (conf_str.empty()) {
      throw runtime_error(string("Configuration file: ") 
	                  + cfg_file + " is empty");
   }

   //------------------------------------  Immortalize the file crc32 checksum
   checksum_crc32 crc;
   crc.add(conf_str);
   config_crc32 = crc.result();

   //------------------------------------  Parse out the file.
   Json::Value doc;
   Json::Reader rdr;
   rdr.parse(conf_str.data(), conf_str.data() + conf_str.size(), doc);
   bool syntax = false;

   parse_json(doc, "config_version", stride);
   parse_json(doc, "stride", stride);
   parse_json(doc, "flag_step", flag_step);
   parse_json(doc, "parity-bit", parity_bit);
   parse_json(doc, "invert", invert);
   string write_path;
   if (parse_json(doc, "write_path", write_path)) frame_path.split(write_path);
   if (!parse_json(doc, "copy_channels", copy_channels)) {
      cout << "No copied channels" << endl;
   }
   parse_json(doc, "compression", compression);

   parse_json(doc, "copy_history", copy_history);

   //---------------------------------- Parse the dq_flags structure
   if (doc.isMember("dq_flags")) {
      Json::Value obj = doc["dq_flags"];
      Json::ArrayIndex nEnt = obj.size();
      for (Json::ArrayIndex i=0; i < nEnt; i++) {
	 const Json::Value& flag_def = obj[i];
	 string name, chan;
	 parse_json(flag_def, "name", name);

	 if (!parse_json(flag_def, "channel", chan)) {
	    cerr << "DQ_Module::config Flag " << name
		 << ", channel is not specified" << endl;
	    syntax = true;
	 }

	 int bit = 0;
	 if (flag_def.isMember("bit")) {
	    bit = flag_def["bit"].asInt();
	 } else {
	    cerr << "DQ_Module::config Flag " << name
		 << ", bit number is not specified" << endl;
	    syntax = true;
	 }
      
	 if (name.empty()) {
	    ostringstream pseudo;
	    pseudo << chan << "[" << bit << "]";
	    name = pseudo.str();
	    cout << "Unnamed flag referred to as " << name << endl; 
	 }

	 string plugin;
	 if (!parse_json(flag_def, "class", plugin)) {
	    cerr << "DQ_Module::config: Flag " << name
		 << ", class name is not specified" << endl;
	    syntax = true;
	 }

	 string library = "/usr/lib/libdq-module.so";
	 if (!parse_json(flag_def, "library", library)) {
	    cerr << "DQ_Module::config: Flag " << name
		 << ", library is not specified." << library << " is assumed."
		 << endl;
	    syntax = true;
	 }

	 //-----------------------------  Add channel list
	 DQ_bit::chanlist_type chan_list;
	 if (!parse_json(flag_def, "inputs", chan_list)) {
	    cerr << "No input channels specified for DQ Flag " << name << endl;
	    syntax = true;
	 }
	 
	 //-----------------------------  Make executable symbols global
	 void* handle = dlopen("",  RTLD_NOW | RTLD_GLOBAL);
	 if (handle == 0) {
	    perror("system error");
	    cerr << "Unable to load executable." << endl;
	    cerr << dlerror() << endl;
	    syntax = true;
	    continue;
	 }

	 //-----------------------------  Load the plugin
         handle = dlopen (library.c_str(), RTLD_NOW | RTLD_GLOBAL);
	 if (handle == 0) {
	    perror("system error");
	    cerr << "Unable to load " << library << endl;
	    cerr << dlerror() << endl;
	    syntax = true;
	    continue;
	 }

	 typedef DQ_bit* (*func_t) (void);
	 string func = plugin + "_init";
	 func_t plug_init=reinterpret_cast<func_t>(dlsym(handle, func.c_str()));
	 if (plug_init == 0) {
	    cerr << "dlsym failed for " << func << endl;
	    syntax = true;
	    continue;
	 }
	 
	 //----------------------------  Construct the channel
	 DQ_bit* dq_obj = plug_init();
	 dq_obj->dq_class(plugin);
	 dq_obj->dq_name(name);
	 dq_obj->bit_channel(chan);
	 dq_obj->bit_number(bit);
	 dq_obj->channel_list(chan_list);

	 //-----------------------------  Add parameter lists
	 if (flag_def.isMember("parameters")) {
	    const Json::Value& par_list = flag_def["parameters"];
	    Json::Value::Members pars = par_list.getMemberNames();
	    size_t nPar = pars.size();
	    for (size_t j=0; j < nPar; j++) {
	       string par_name = pars[j];
	       const Json::Value& par_val = par_list[par_name];
	       if (par_val.isNumeric()) {
		  dq_obj->numeric_param(par_name, par_val.asDouble());
	       }
	       else if (par_val.isBool()) {
		  double val = par_val.asBool() ? 1.0 : 0.0;
		  dq_obj->numeric_param(par_name, val);
	       }
	       else {
		  dq_obj->string_param(par_name, par_val.asString());
	       }
	    }
	 }
	 flag_vect.push_back(dq_obj);
      }

   }
   //-----------------------------------  No flags defined
   else {
      cerr << "No flags defined in document: " << cfg_file << endl;
      syntax = true;
   }
}

//======================================  Request all input channels.
void
DQ_Module::book_channels(void) {
   if (Debug()) cout << "DQ_Module::book_channels: Booking channels" << endl;

   //--------------------------------- combine all channels
   DQ_bit::chanlist_type combined;
   size_t nFlags = flag_vect.size();
   if (nFlags) {
      combined = flag_vect[0]->channel_list();
      for (size_t i=1; i<nFlags; i++) {
	 const DQ_bit::chanlist_type& chanlist = flag_vect[i]->channel_list();
	 combined.insert(combined.end(), chanlist.begin(), chanlist.end());
      }
   }

   //---------------------------------  add in copied channels
   if (!copy_channels.empty()) {
      combined.insert(combined.end(), copy_channels.begin(),
		      copy_channels.end());
   }
   if (combined.empty()) return;
   
   //---------------------------------  Remove stream definitions.
   size_t nCopy = copy_channels.size();
   for (size_t i=0; i<nCopy; i++) {
      size_t inx = copy_channels[i].find_last_of('/');
      if (inx != string::npos) copy_channels[i].erase(0, inx+1);
   }

   //---------------------------------- make a unique list.
   std::sort(combined.begin(), combined.end());
   size_t nChans = combined.size();
   size_t j = 0;
   for (size_t i=1; i<nChans; i++) {
      if (combined[i] != combined[j]) {
	 j++;
	 if (i != j) combined[j] = combined[i];
      }
   }
   j++;
   combined.erase(combined.begin() + j, combined.end());

   //-----------------------------------  Book all unique channels
   nChans = combined.size();
   for (size_t i=0; i<nChans; i++) {
      getDacc().addChannel(combined[i]);
      if (Debug()) cout << "DQ_Module::book_channels: Added channel "
			<< combined[i] << endl;
   }
}

//======================================  
void
DQ_Module::set_flag(DQ_bit* dq_def, const Time& t0, Interval dT) {
   DQ_bit::tser_vect data;
   const DQ_bit::chanlist_type& chanlist = dq_def->channel_list();
   size_t nChan = chanlist.size();
   data.reserve(nChan);
   for (size_t i=0; i<nChan; i++) {
      const TSeries* tsi = getDacc().refData(chanlist[i]);
      if (tsi) {
	 data.push_back(tsi->extract(t0, dT));
      } 
      else if (flag_map.find(chanlist[i]) != flag_map.end()) {
	 data.push_back(flag_map[chanlist[i]].extract(t0,dT));
      }
      else {
	 ostringstream msg;
	 msg << "DQ_Module: Unable to find channel " << chanlist[i] 
             << " to evaluate " << dq_def->dq_name() << endl;
	 throw std::runtime_error(msg.str());
      }
   }

   bool bit = dq_def->bit_value(data);
   if (bit) {
      flag_map_iter out_chan = flag_map.find(dq_def->bit_channel());
      TSeries& tsout = out_chan->second;
      size_t inx = tsout.getBin(t0);
      DVectU* dv = dynamic_cast<DVectU*>(tsout.refDVect());
      if (!dv) {
	 cerr << "Unable to find data vector for " << dq_def->bit_channel()
	      << endl;
      } else {
	 (*dv)[inx] |= 1 << dq_def->bit_number();
      }
   }
}

void
DQ_Module::init_all(void) {
   size_t nFlags = flag_vect.size();
   for (size_t i=0; i<nFlags; i++) {
      flag_vect[i]->init();
   }
}

//======================================  Process a stride.
void 
DQ_Module::ProcessData(void) {
   uint32_t data=0;

   if (!current || current < getDacc().getFillTime()) {
      init_all();
      current = getDacc().getFillTime();
   }
   Interval dT = getDacc().getFillStride();
   
   //-----------------------------------  Initialize template/
   Interval fstep(flag_step);
   int nstep = int(double(dT) / fstep);
   TSeries temp(current, fstep, 1, &data);
   temp.extend(current + dT);

   //-----------------------------------  Initialize output bit masks.
   size_t nFlags = flag_vect.size();
   for (size_t iflag=0; iflag < nFlags; ++iflag) {
      string chan = flag_vect[iflag]->bit_channel();
      flag_map_iter itr = flag_map.find(chan);
      if (itr == flag_map.end()) {
	 flag_map.insert(flag_map_type::value_type(chan, temp));
      } else {
	 itr->second = temp;
      }
   }
   
   //-----------------------------------  Initialize step over times
   for (int istep=0; istep < nstep; istep++) {
      Time t0 = current + fstep * double(istep);
      for (size_t iflag=0; iflag<nFlags; iflag++) {	 
	 set_flag(flag_vect[iflag], t0, Interval(flag_step));
      }
   }

   //-----------------------------------  Set the parity bit and inver data
   if (parity_bit || invert) {
      const uint32_t parity_bit = (1 << 31);
      for (flag_map_iter out_chan=flag_map.begin(); 
	   out_chan != flag_map.end();
	   out_chan++) { 
         TSeries& tsout = out_chan->second;
         DVectU* dv = dynamic_cast<DVectU*>(tsout.refDVect());
	 size_t N=tsout.getNSample();
	 for (size_t i=0; i<N; i++) {
	    uint32_t dvi = (*dv)[i];
	    if (parity_bit) {
	       dvi &= ~parity_bit;
	       if (!odd_parity(dvi)) dvi |= parity_bit;
            }
	    if (invert) dvi = ~dvi;
	    (*dv)[i] = dvi;
	 }
      } // loop over DQ vectors
   }

   //-----------------------------------  Write a frame
   write_frame(current, dT);
   current += getDacc().getFillStride();
}

//======================================  Write one frame
void
DQ_Module::write_frame(const Time& t0, Interval dT) {
   fr_writer.buildFrame(t0, dT);

   //-----------------------------------  Copy requested history records.
   size_t nHist = copy_history.size();
   for (size_t i=0; i<nHist; i++) {
      string frametype = copy_history[i];
      int inx = getDacc().frame_type(frametype);
      if (inx < 0) {
	 if (frametype[0] >= '0' && frametype[0] <= '9') {
	    inx = strtoul(frametype.c_str(), 0, 0);
	 } else {
	    continue;
         }
      }
      fr_writer.copyHistory(*(getDacc(inx)->getFrame()));
   }

   //-----------------------------------  Add a history record
   ostringstream config;
   config << "Configuration file: " <<  config_file << ", version: "
	  << config_version << ", crc32: " << config_crc32;
   fr_writer.addHistory("dq-module", Now(), config.str());

   //-----------------------------------  Copy requested channels.
   size_t nCopy = copy_channels.size();
   for (size_t i=0; i<nCopy; i++) {
      string tsname = copy_channels[i];
      TSeries* tsi = getDacc().refData(tsname);
      if (!tsi) {
	 cerr << "Error: Copied channel: " << tsname << "is not available." 
	      << endl;
      } else {
	 fr_writer.addProcSeries(tsname, *tsi);
      }
   }

   //-----------------------------------  Write DQ flag series.
   for (const_flag_map_iter i=flag_map.begin(); i != flag_map.end(); i++) {
      fr_writer.addProcSeries(i->first, i->second);
   }

   //-----------------------------------  Open a writer to a frame file
   string outdir(frame_path.dir_name(t0));
   if (!frame_path.online()) {
      frame_path.make_dir(outdir);
      string file_path = frame_path.file_path(t0, dT);
      if (Debug()) cout << "  opening file: " << file_path << endl;
      if (fr_writer.open(file_path.c_str(), true)) {
	 fr_writer.erase();
	 cerr << "DQ_Module: Failed to open frame " << file_path
	      << " for output." << endl;
	 return;
      }
   }
   
   //----------------------------------  Open a writer to shared memory.
   else {
      if (fr_writer.open(outdir.c_str(), true)) {
	 fr_writer.erase();
	 cerr << "DQ_Module: Failed to open partition " << outdir
	      << " for output." << endl;
	 return;
      }
   }
 
   //-----------------------------------  Write the frame
   try {
      if (fr_writer.writeFrame()) {
	 cerr << "DQ_Module: Error writing frame with GPS " << t0.getS()
	      << " to output Directory " << outdir << endl;
      } else {
	 frames_out++;
      }
   } catch (std::exception& e) {
      cerr << "Caught exception in writeFrame: " << e.what() << endl;
   }
   
   //-----------------------------------  Clean up
   fr_writer.close();
   fr_writer.erase();
}
