#include <iostream>
#include <sstream>
#include <iomanip>
#include <string>
#include <vector>
#include <utility>

#include "Time.hh"
#include "xml/XsilTSeries.hh"
#include "xml/Xsil.hh"
#include "html/writer.hh"
#include "html/align.hh"  
#include "html/document.hh" 
#include "html/font.hh" 
#include "html/hline.hh" 
#include "html/table.hh"
#include "html/text.hh"
#include "html/color.hh" 

#include "dewarMon.hh"

   using namespace std;
   using namespace xml;
   typedef std::pair<string, string> ConfPair;
   typedef std::vector< ConfPair > ConfElements;
   typedef std::vector<int> config_numbers;
   typedef std::map<std::string, config_numbers> ChanConfig_map; 


	//===========================================================
   bool TagCompare ( const std::string& key, const char* tag ) {
      if ( strcasecmp ( key.c_str(), tag ) != 0 ) {
         return false;
      }
      return true;
   }

	//===========================================================
   bool GetBool (const std::string& s ) {
      bool ret = false;
      if ( TagCompare ( s , "ON")  ) {
         ret = true;
      }
   
      return ret;
   }

	//===========================================================
   bool GetElements (const std::string& value_org, ConfElements& elements ) {
      std::string value = value_org + " ";
      // split string into elements
      std::string::size_type pos = 0, prev_pos = 0;
      std::vector<std::string> contents;
      std::string word;
      while( ( pos = value.find_first_of(' ', pos)) != string::npos ) {
         if ( value[prev_pos] == '{' ) {
	   int count = 0;
	   bool cont = true;
	   while (cont) {
	     while ((value[pos] != '{') && (value[pos] != '}')) {
	       pos++;
	     }
	     if (value[pos] == '{') {
	       count++;
	       pos++;
	     }
	     else {
	       if ((value[pos] == '}') && (count >0)) {
		 count = count-1;
		 pos++;
	       }
	       else if ((value[pos] == '}') && (count == 0)) {
		 if ( pos == string::npos ) { 
		   return false;
		 }
		 word = value.substr( prev_pos, pos - prev_pos + 1);

		 cont = false;
	       } 
	     }
	   }
	 }
         else if ( value[prev_pos] == '"' ) {
            std::string::size_type qpos = ++prev_pos;
            while ( ( pos = value.find_first_of ('"', qpos) ) != string::npos &&
            value[ pos - 1 ] == '\\' ) {
               qpos = ++pos;
            }
            if ( pos == std::string::npos ) {
               cerr << "\" mismatch. " << value_org << endl;
               return false;
            }
            word = value.substr( prev_pos, pos - prev_pos );
         }
         else word = value.substr( prev_pos, pos - prev_pos);
         if ( word.size() ) contents.push_back(word);
         prev_pos = ++pos;
      }
   
      // create std::map of tag and value
      std::vector<string>::iterator itr = contents.begin();
      while ( itr != contents.end() ) {
         std::string key = *itr;
         std::string val;
         ++itr;
         if ( itr != contents.end() ) {
            val = *itr;
         }
         else 
            return false;
      
         // elements[key] = val;
         elements.push_back( ConfPair(key, val) );
         ++itr;
      }
   
      return true;
   }

	//===========================================================
   bool parse ( ifstream& file, std::string& tag, std::string& value, 
   std::string& remainder ) {
      bool done = false;
      int depth = 0;
      bool find_beg = false;
      bool find_end = false;
   
      std::string line = remainder;
      std::string newline;
      while ( !file.eof() && !done ) {
         // read one line from config file 
         getline( file, newline );
         // remove extra spaces
         while (!newline.empty() && isspace (newline[0])) {
            newline.erase (0, 1);
         }
         // process comment line
         if ( newline.empty() ) 
            continue;
         if ( newline[0] == '#' ) 
            continue;
      
         int newlen = newline.length()-1;
         if (!newline.empty() && newline[newlen] == '\n') 
            newline.erase(newlen,1);
      
         line += newline;
         while (!line.empty() && isspace (line[0])) {
            line.erase (0, 1);
         }
         line += " ";
      
         string::size_type pos = 0;
         while ( (pos = newline.find_first_of ("{}" , pos )) != string::npos
		 && !(find_beg && find_end) ) {
            if ( newline[pos] == '{'  ) {
               if ( !find_beg ) find_beg = true;
               ++depth;
            }
            else if ( newline[pos] == '}' ) {
               --depth;
               if ( !find_end && depth == 0 ) find_end = true;
            }
            if ( depth < 0 ) {
               cerr << "{} mismatch error." << endl;
               return false;
            }
         
            ++pos;
         }
      
         if ( find_beg && find_end ) done = true;
      }
      if ( done ) {
         std::string::size_type beg = line.find_first_of ('{');
         std::string::size_type end = line.find_last_of ('}');
         tag = line.substr ( 0, beg );
         while ( !tag.empty() && isspace (tag[tag.size() - 1]) ) {
            tag.erase(tag.size() - 1, 1);
         }
         value = line.substr ( beg + 1, end - beg - 1);
         remainder = line.substr(end + 1);
         return true;
      }
      else {
         return false;
      }
   }

//===========================================================
//
//  dewarMon
//
//===========================================================
   EXECDAT(dewarMon)
   
    // constructor
   dewarMon::dewarMon(int argc, const char *argv[]) :  
     DatEnv(argc,argv), fMonStartTime(0), fFirstEvent(true)
   {
      fFList[ "Default" ] = FilterInfo();
      fTList[ "Default" ] = ThresholdInfo();
      fCPList[ "Default" ] = Parameter();
      fHList[ "Default" ] = HistogramInfo();

   
      const char* filename = 0;
      if (argc > 1) {
         filename = argv[1];
      }
      else {
         cerr << "Error : No Filename." << endl;
         finish ();
         return;
      }
   
   	//open config file
      ifstream configfile (filename, ios::in);
      if (!configfile.is_open()){
	cerr << "Configuration File Not Found." << endl;
         finish ();
         return;
      }
      else {
      //read configuration file
         if ( !SetConfig ( configfile ) ) {
            finish();
            return;
         }
      }
      configfile.close();
      RegisterChannels();
      fMonServ.setServerName ( fMonConf.fName.c_str() );
      if ( !Reset() ) {
         finish();
         return;
      }
      ConfigLog ();
      //MakeHtmlLog ();
   }

	//===========================================================
	// destructor
	//===========================================================
   dewarMon::~dewarMon(void)
   {
 
      OutputEvents ();
      CloseFiles ();
   
      ofstream tsout ( kTSBackUpName, ios::out | ios::trunc );
      if ( tsout.is_open() ) {
         tsout << xsilHeader() << endl;

	 for (SetList::iterator sitr = fSList.begin();
	      sitr != fSList.end(); ++sitr ) {
	   sitr->TSSave( tsout );
	 }

         // for ( ChannelList::iterator itr = fChList.begin();
//          itr != fChList.end(); ++itr ) {
//             itr->TSSave ( tsout );
//          }
         tsout << xsilTrailer() << endl;
      }
   }


	//===========================================================
   void dewarMon::Attention() {

     for (SetList::iterator sitr = fSList.begin();
	  sitr != fSList.end(); ++sitr ) {
       sitr -> Attention ();
     }

     // for ( ChannelList::iterator itr = fChList.begin();
// 	   itr != fChList.end(); ++itr ) {
//        itr->Attention ();
//      }
     
   
     fMonServ.Attention();
   }

	//===========================================================
	// main loop
	//===========================================================
   void dewarMon::ProcessData(void)
   {
     long b = getDacc().getCurrentTime().getS();
     if (!fMonStartTime || (b % 16)==0 ) {
       MakeHtmlLog();
       UpdateTimeStamps();
       //cout <<"Time now " << b <<endl;
     }		
     
     fLock.Check ( getDacc() );
     for (SetList::iterator sitr = fSList.begin();
	  sitr != fSList.end(); ++sitr) {
       if (!(sitr->IsInit())) {
	 sitr->Init(getDacc().getCurrentTime());
       }
     }
     for (ChannelList::iterator itr = fChList.begin(); 
	  itr != fChList.end(); ++itr) {
       itr->ProcessChannel (getDacc(), fLock );
     }
     for (SetList::iterator sitr = fSList.begin();
	  sitr != fSList.end(); ++sitr) {
       sitr->findCoincidentEvents( fEList, einfo_log, fChList);
     }
     for (ChannelList::iterator chitr = fChList.begin();
	  chitr != fChList.end(); ++chitr) {
       chitr->EraseCoincEvents_ch();
     }
     OutputEvents ();
     ProcessLog ();
     
     if ( fMonConf.fDuration.GetS() && fMonConf.fDuration < getDacc().getCurrentTime() - fMonStartTime ) {
       
       finish ();
     }
   }


	//===========================================================
   bool dewarMon::OpenFiles () {
      int t = Now().getS();
      int n = Now().getN();

      if ( fMonConf.fDataPath.size() ) {
         ostringstream str;
         str << t << "_" << n ;
         std::string data = fMonConf.fDataPath;
         data += fMonConf.fName + "_";
         data += str.str();
         data += ".xml";
         fDataFile.clear();
	 fDataFile.open ( data.c_str(), ios::out|ios::ate );
         if ( !fDataFile.is_open() ) {
            cerr << "Error opening Data File." << endl;
            return false;
         }
         fDataFile << xsilHeader() << endl;
         fDataFile << xsilTableBegin (fMonConf.fName.c_str(), "dewarMon") << endl;
         fDataFile << xsilComment ("dewarMon events") << endl;
         fDataFile << "    <Column Name=\"Name\" Type=\"string\"/>" << endl;
         fDataFile << "    <Column Name=\"Ifo\" Type=\"string\"/>" << endl;
         fDataFile << "    <Column Name=\"Start_Time\" Type=\"int\"/>" << endl;
         fDataFile << "    <Column Name=\"Start_Time_NS\" Type=\"int\"/>" << endl;
         fDataFile << "    <Column Name=\"Duration\" Type=\"double\"/>" << endl;
         fDataFile << "    <Column Name=\"Peak\" Type=\"double\"/>" << endl;
         fDataFile << "    <Column Name=\"NSigma\" Type=\"double\"/>" << endl;
         fDataFile << "    <Column Name=\"Sigma\" Type=\"double\"/>" << endl;
         fDataFile << "    <Column Name=\"Mean\" Type=\"double\"/>" << endl;
         fDataFile << "    <Column Name=\"NPoints\" Type=\"int\"/>" << endl;
         fDataFile << "    <Column Name=\"Comment\" Type=\"string\"/>" << endl;
         fDataFile << xsilTableDataBegin ("Events") << endl;
      
         fFirstEvent = true;
      }
   
      if ( fMonConf.fDoLog ) {
         const char* htmldir = getenv("DMTHTMLOUT");
         std::string log;
         if (htmldir) log = string ( htmldir ) + "/";
         //ostringstream str;
         //str << t;
         log += fMonConf.fName + "_";
         //log += str.str();
         log += "log.html";
         fLogFile.clear();
	 fLogFile.open ( log.c_str(),  ios::out | ios::app);
	 fLogFile.close();
	 fLogFile.open(log.c_str(), ios::in | ios::out | ios::ate);
         if ( !fLogFile.is_open() ) {
            cerr << "Error opening Log File." << endl;
            return false;
         }
      }

      return true;
   }

	//===========================================================
   void dewarMon::CloseFiles () {
      if ( fDataFile.is_open() ) {
         fDataFile << endl;
         fDataFile << xsilTableDataEnd () << endl;
         fDataFile << xsilTableEnd () << endl;
         fDataFile << xsilTrailer() << endl;
         fDataFile.close();
      }
      if ( fLogFile.is_open() ) {
         fLogFile.close();
      }
   }

	//===========================================================
   void dewarMon::UpdateFiles () {
      CloseFiles ();
      OpenFiles ();
   }

	//===========================================================
   void dewarMon::OutputEvents () {
      if ( fDataFile.is_open() ) {
         for ( EventList::iterator itr = fEList.begin();
         itr != fEList.end(); ++itr ) {
            if ( !fFirstEvent ) {
               fDataFile << "," << endl;
            }
            else {
               fFirstEvent = false;
            }
            itr->Dump ( fDataFile );
         }
      }
   
      if ( fMonConf.fDoTrigger ) {
         for ( EventList::iterator itr = fEList.begin();
         itr != fEList.end(); ++itr ) {
         
            trig::TrigRslt trigData( fMonConf.fName.c_str(), itr->fComment.c_str() );
            if ( itr->fIfo.size() ) trigData.setIfos( itr->fIfo.c_str() );
            trigData.setTime ( itr->fStartTime );
            trigData.setDuration( itr->GetDuration() );
            trigData.setIntensity( itr->fMaxData );
            trigData.setSignificance ( itr->fMaxSDev );
            trigData.setFrequency ( itr->GetPeakTime().GetSecs() );
            trigData.setBandwidth ( (double)itr->fEventNPoint );
            trigData.setPriority(trig::p_info);
            trigData.setDisposition(trig::d_metaDB);
            sendTrigger( trigData );
         }
      }
   
      if ( !fEList.empty() ) {
	MakeHtmlLog();
	UpdateTimeStamps();
	fEList.clear();
	einfo_log.clear();
      }
   }

	//===========================================================
   void dewarMon::ProcessLog () {
   
      Time currenttime = getDacc().getCurrentTime();
   
      if ( fLogStartTime == Time(0,0) ) {
         fMonStartTime = currenttime;
         fLogStartTime = currenttime;
      }

      for (SetList::iterator sitr = fSList.begin();
	   sitr != fSList.end(); ++sitr) {
	sitr->UpdateHistogram( currenttime, fTrend );
      }

   
      // for ( ChannelList::iterator itr = fChList.begin();
// 	    itr != fChList.end(); ++itr ) {
// 	itr->UpdateHistogram( currenttime, fTrend );
//       }
   
      fTrend.Update ( currenttime );
   
   // update log file every fLogInterval
      if ( currenttime - fLogStartTime >= fMonConf.fLogInterval ){
      
         UpdateFiles ();
      
         fLogStartTime = currenttime;
      
      }
   
      return;
   }

	//===========================================================
void dewarMon::MakeHtmlLog () {
  Time time = Now();
  if ( !fLogFile.is_open() ) return;
  fLogFile.seekp(0, ios_base::end);
  int length = fLogFile.tellp();
  char buf[128];
  if (length < 1) {
    std::string site;
    if (fMonConf.fSite == MonitorConf::kLHO) {
      site = "LHO";
    }
    else if (fMonConf.fSite == MonitorConf::kLLO) {
      site = "LLO";
    }
    std::string str = "<html><head><title>dewarMon Log - ";
    str += site;
    str += "</title><meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\"></head> \n";
    str += "<body><center>dewarMon Log - ";
    str += site;
    str += "</center> \n";
    str +="<hr width=100%><left>Log Start (UTC): ";
    str += TimeStr ( getDacc().getCurrentTime(), buf, "%M %d, %Y at %H:%N:%S");
    str += "</left> \n";
    str +="<hr width=100%><center>This page was last updated : ";
    str += TimeStr ( time, buf, "%M %d, %Y at %H:%N:%S");
    str += " (UTC) | ";
    str +=  LocalStr ( time, buf, "%M %d, %Y at %H:%N:%S");
    str += " (Local) | ";
    str +=  TimeStr ( time, buf, "%s (GPS)" );
    str += "</center> \n";
    
    str += "<blockquote><center><table border=1> \n";
    str += "<tr><th>GLITCH TIME (UTC)</th><th>GLITCH TIME (LOCAL)</th><th>DEWAR LOCATION</th><th>CHANNELS USED</th><th>FILTERS USED</th><th>MAX NSIGMA</th></tr> \n </table></center></blockquote></html>";
    fLogFile << str;
  }

  fLogFile_update_loc[0] = 291;
  fLogFile_update_loc[1] = 327;
  fLogFile_update_loc[2] = 365;

  char t;
  char T[8]= {'0', '0', '0', '0', '0', '0', '0', '0'};
  char Table[8] = {'<', '/', 't', 'a', 'b', 'l', 'e','>'};
  bool cont = true;
  fLogFile.seekg(-1,ios::cur);
  int iteration = 0;
  while((cont)&&(iteration<=100)) {
    if (iteration==100) {
      cerr<<"MakeHtmlLog Error: Cannot find end of table"<<endl;
    }
    iteration++;
    fLogFile.get(t);
    fLogFile.seekg(-2,ios::cur);
    for (int i = 0; i<=6; i++) {
      T[7-i] = T[6-i];
    }
    T[0] = t;
    cont = false;
    for (int i = 0; i<=7; i++) {
      if (T[i]!=Table[i]) {
	cont = true;
      }
    }
  }
  fLogFile.seekg(1, ios::cur);
  int current = 0;
  for (EventList::iterator itr = fEList.begin();
       itr != fEList.end(); ++itr) {
    int time_utc = int(itr->fStartTime.getS());
    char* time_local =  LocalStr ( itr->fStartTime, buf, "%M %d, %Y at %H:%N:%S");
    std::string rem_info = einfo_log.at(current);
    fLogFile << "<tr><td>" << time_utc << "</td><td>" 
	     << time_local << "</td>" 
	     << rem_info << "</tr> \n";
    current++;
  }
  fLogFile << "</table></center></blockquote></html>";
  
}



	//===========================================================
   void dewarMon::ConfigLog () {
     ofstream conflog;
      Time t = Now();
      if ( fMonConf.fDoLog ) {
         const char* htmldir = getenv("DMTHTMLOUT");
         std::string log;
         if (htmldir) log = string ( htmldir ) + "/";
         ostringstream time;
         time << t;
         log += fMonConf.fName + "_";
         log += time.str();
         log += "_conf.html";
	 conflog_name = log;
         conflog.clear();
         conflog.open ( log.c_str(), ios::out );
         if ( !conflog.is_open() ) {
            cerr << "Error opening configuration summary File." << endl;
            return;
         }
      }
   
      html::document doc ( "dewarMon configuration summary" );
      html::text title ( "dewarMon configuration summary" );
   
      html::block titleblk ( "center" );
      titleblk.addObject ( title );
      doc.addObject ( titleblk );
   
      doc.addObject ( html::hline() );

      char buffer[128];
      html::block updateblk ("center"); 
      html::text lasttime ("This page was last updated : "); 
      lasttime << TimeStr ( t, buffer, "%M %d, %Y at %H:%N:%S");
      lasttime << " (UTC) | ";
      lasttime << LocalStr ( t, buffer, "%M %d, %Y at %H:%N:%S");
      lasttime << " (Local) | ";
      lasttime << TimeStr ( t, buffer, "%s (GPS)");
      updateblk.addObject (lasttime);	
      doc.addObject(updateblk); 
      doc.addObject ( html::hline() );

      conflog_update_loc[0] = 293;
      conflog_update_loc[1] = 330;
      conflog_update_loc[2] = 369;


      html::text timeinfo ("Monitor Start (UTC) : ");
      char buf[128];
      timeinfo << TimeStr ( t, buf, "%M %d, %Y at %H:%N:%S");
      doc.addObject ( timeinfo );

      doc.addObject ( html::hline() );
      
      html::table monconf;
      monconf.setBorder ( true );
      monconf.addColumn ( "Monitor Name" );
      monconf.addColumn ( "Site" );
      monconf.addColumn ( "Monitor Duration (min)" );
      monconf.addColumn ( "Data Location" );
      monconf.addColumn ( "Log" );
      monconf.addColumn ( "Log Interval (min)" );
      monconf.addColumn ( "Trigger" );
      monconf.addColumn ( "Trend" );
      monconf.addColumn ( "Time Series Length (min)" );
   
      monconf.addRow();
      monconf.insertData ( 0, 0, fMonConf.fName );
      std::string confstr = fMonConf.GetSiteName();
      monconf.insertData ( 0, 1, confstr );
      monconf.insertData ( 0, 2, fMonConf.fDuration.GetS()/60 );
      monconf.insertData ( 0, 3, fMonConf.fDataPath );
      confstr = (fMonConf.fDoLog) ? "ON" : "OFF";
      monconf.insertData ( 0, 4, confstr );
      monconf.insertData ( 0, 5, fMonConf.fLogInterval.GetS()/60 );
      confstr = (fMonConf.fDoTrigger) ? "ON" : "OFF";
      monconf.insertData ( 0, 6, confstr );
      confstr = (fMonConf.fDoTrend) ? "ON" : "OFF";
      monconf.insertData ( 0, 7, confstr );
      monconf.insertData ( 0, 8, (long)fMonConf.fTSInterval );
      doc.addObject ( monconf );
   
      doc.addObject ( html::hline() );
      html::table results;
      results.setBorder (true);
      results.addColumn ( "Dewar Location" );
      results.addColumn ( "Configuration" );
      results.addColumn ( "Channel Name" );
      results.addColumn ( "Check Lock" );
      results.addColumn ( "Threshold Type" );
      results.addColumn ( "Threshold Low" );
      results.addColumn ( "Threshold High" );
      results.addColumn ( "Filter Formula" );
      results.addColumn ( "Minimum Separation" );
      results.addColumn ( "Minimum Duration" );
      results.addColumn ( "Maximum Duration" );
      results.addColumn ( "Minimum Density" );
      results.addColumn ( "Comment" ); 


      for ( SetList::iterator itr = fSList.begin();
	    itr != fSList.end(); ++itr ) {
	itr->ConfigLog ( results , fChList );
      }



      doc.addObject ( results );
   
      doc.addObject ( html::hline() );
      html::block totalblk ("LEFT");

      html::text nset ("Total Number of Dewar Locations: ");
      nset << fSList.size();

      html::text nchannel ("Total Number of Channels: ");
      nchannel << fChList.size();

      totalblk.addObject (nset);
      totalblk.lineBreak();
      totalblk.addObject (nchannel);
      totalblk.lineBreak();
      html::text nconf ("Total Number of Configurations: ");
      int nc = 0;
      for (SetList::iterator sitr = fSList.begin();
	   sitr != fSList.end(); ++sitr) {
	nc += sitr->GetNConf();
      }
      nconf << nc;
      //nconf << results.getNRow ();
      totalblk.addObject (nconf);
      doc.addObject ( totalblk );
   
      doc.addObject (html::hline());
      html::block authorblk ("center"); 
      html::text author ("This monitor was written by Emelie Harstad");  
      authorblk.addObject (author); 
      doc.addObject (authorblk); 
      doc.addObject (html::hline());
   
      html::writer htmlout ( conflog );
      doc.write (htmlout);
   
      conflog.close();
   }

	//===========================================================
   bool dewarMon::Reset () {
   
      if ( !OpenFiles() ) {
         return false;
      }
   
      getDacc().setStride (1.0);
      getDacc().setIgnoreMissingChannel(true);
   
      vector<TSeries> tlist;
      xsilHandlerQueryTSeries tsQ (tlist);
      xsilParser parser;
      parser.AddHandler (tsQ);
      parser.ParseFile ( kTSBackUpName );
   
      if ( fMonConf.fDoTrend ) {
         fTrend.setName ( fMonConf.fName.c_str() );
         fTrend.setType (Trend::kMinute);
         fTrend.setAutoUpdate ( false );
      }

      for ( ChannelList::iterator itr = fChList.begin();
	    itr != fChList.end(); ++itr ) {
         itr->ResetStat();
      }
   
      for (SetList::iterator sitr = fSList.begin();
	   sitr != fSList.end(); ++sitr) {
	sitr->ResetStat(fMonServ, fTrend, tlist, fMonConf.fTSInterval);
      }
      
      if ( fMonConf.fDoTrend ) fTrend.writeIndex();
      return true;
   }

        //===========================================================
void dewarMon::UpdateTimeStamps() {
  Time t = Now();
  char buf[128];
  std::string UTC_time = TimeStr ( t, buf, "%M %d, %Y at %H:%N:%S");
  std::string LOCAL_time = LocalStr ( t, buf, "%M %d, %Y at %H:%N:%S");
  std::string GPS_time = TimeStr ( t, buf, "%s (GPS)");
  if (fLogFile.is_open()) {
    fLogFile.seekp(fLogFile_update_loc[0], ios_base::beg);
    fLogFile<<UTC_time;
    fLogFile.seekp(fLogFile_update_loc[1], ios_base::beg);
    fLogFile<<LOCAL_time;
    fLogFile.seekp(fLogFile_update_loc[2], ios_base::beg);
    fLogFile<<GPS_time;
  }
  else {
    cerr<<"Log file not open.  Cannot update time stamp."<<endl;
  }

  fstream configlog;
  configlog.open(conflog_name.c_str(), ios::in | ios::out);
  if (configlog.is_open()) {
    configlog.seekp(conflog_update_loc[0], ios_base::beg); 
    configlog<<UTC_time;
    configlog.seekp(conflog_update_loc[1], ios_base::beg);
    configlog<<LOCAL_time;
    configlog.seekp(conflog_update_loc[2], ios_base::beg);
    configlog<<GPS_time;
    configlog.close();
  }
  else {
    cerr<<"Cannot open config file for time stamp update."<<endl;
  }
 
}



	//===========================================================
   void dewarMon::RegisterChannels () {
   
      fLock.Register ( getDacc(), fMonConf.fSite );
   // channels to be monitored
      for ( ChannelList::iterator itr = fChList.begin();
	    itr != fChList.end(); ++itr) {
	itr->Register ( getDacc() );
      }
   }

	//===========================================================
	//
	// setup configuration
	//
	//===========================================================
   bool dewarMon::SetConfig( ifstream& configfile )
   {
     std::string tag;
      std::string value;
      std::string remainder;
      // std::vector<string> ch_values;
//       std::vector<string> th_values;
//       std::vector<string> flt_values;
//       std::vector<string> prm_values;
//       std::vector<string> hst_values;
//       flt_values.push_back("NAME bpx1 Formula \"ellip('BandPass',4,1,40,5,25)\"");
//       flt_values.push_back("NAME bpy1 Formula \"ellip('BandPass',4,1,40,5,25)\"");
//       flt_values.push_back("NAME bpz1 Formula \"ellip('BandPass',4,1,40,5,25)\"");
//       flt_values.push_back("NAME bpx2 Formula \"ellip('BandPass',4,1,40,12,20)\"");
//       flt_values.push_back("NAME bpy2 Formula \"ellip('BandPass',4,1,40,12,20)\"");
//       flt_values.push_back("NAME bpz2 Formula \"ellip('BandPass',4,1,40,12,20)\"");
//       th_values.push_back("NAME thx1 TYPE Relative LOW 4.0 HIGH 4.0");
//       th_values.push_back("NAME thy1 TYPE Relative LOW 4.0 HIGH 4.0");
//       th_values.push_back("NAME thz1 TYPE Relative LOW 4.0 HIGH 4.0");
//       th_values.push_back("NAME thx2 TYPE Relative LOW 4.0 HIGH 4.0");
//       th_values.push_back("NAME thy2 TYPE Relative LOW 4.0 HIGH 4.0");
//       th_values.push_back("NAME thz2 TYPE Relative LOW 4.0 HIGH 4.0");
//       prm_values.push_back("NAME param1 MINSEPARATION 2.0 MINDURATION 0.1 MAXDURATION 5.0 MINDENSITY 0.1");
//       hst_values.push_back("NAME hist100 BIN 200 LOW 0.0 HIGH 100.0");
//       ch_values.push_back("NAME H0:PEM-LVEA_SEISX CHECKLOCK OFF CONFIG { THRESHOLD thx1 FILTER bpx1 PARAMETER param1 HISTOGRAM hist100 } CONFIG { THRESHOLD thx2 FILTER bpx2 PARAMETER param1 HISTOGRAM hist100 }");
//       ch_values.push_back("NAME H0:PEM-LVEA_SEISY CHECKLOCK OFF CONFIG { THRESHOLD thy1 FILTER bpy1 PARAMETER param1 HISTOGRAM hist100 } CONFIG { THRESHOLD thy2 FILTER bpy2 PARAMETER param1 HISTOGRAM hist100 }");
//       ch_values.push_back("NAME H0:PEM-LVEA_SEISZ CHECKLOCK OFF CONFIG { THRESHOLD thz1 FILTER bpz1 PARAMETER param1 HISTOGRAM hist100 } CONFIG { THRESHOLD thz2 FILTER bpz2 PARAMETER param1 HISTOGRAM hist100 }");
//       size_type i;
//       if (!flt_values.empty()) {
// 	for (i=0; i<flt_values.size(); i++) {
// 	  AddFilterInfo (flt_values.at(i));
// 	}
//       }
//       if (!th_values.empty()) {
// 	for (i=0; i<th_values.size(); i++) {
// 	  AddThresholdInfo (th_values.at(i));
// 	}
//       }
//       if (!prm_values.empty()) {
// 	for (i=0; i<prm_values.size(); i++) {
// 	  AddParameter (prm_values.at(i));
// 	}
//       }
//       if (!hst_values.empty()) {
// 	for (i=0; i<hst_values.size(); i++) {
// 	  AddHistogramInfo (hst_values.at(i));
// 	}
//       }
//       if (!ch_values.empty()) {
// 	for (i=0; i<ch_values.size(); i++) {
// 	  AddChannelInfo (ch_values.at(i));
// 	}
//       }

      bool valid = true;
      while (!configfile.eof() && valid) {
         if ( parse (configfile, tag, value, remainder) ) {
            if ( TagCompare ( tag, "MONITOR" )  ) {
               valid = SetMonitorParam ( value );
            }
	    //	 }
	    //      }
            else if ( TagCompare( tag, "FILTER" )  ) {
               valid = AddFilterInfo ( value );
            }
            else if ( TagCompare( tag, "THRESHOLD" )  ) {
               valid = AddThresholdInfo ( value );
            }
            else if ( TagCompare( tag, "PARAMETER" )  ) {
               valid = AddParameter ( value );
            }
            else if ( TagCompare( tag, "HISTOGRAM" )  ) {
               valid = AddHistogramInfo ( value );
            }
	    // else if ( TagCompare( tag, "CHANNEL" )  ) {
            //   valid = AddChannelInfo ( value );
            //}
            else if ( TagCompare( tag, "SET" ) ) {
	      valid = AddSetInfo (value);
	    }
	    else {
               cerr << "Unknown Tag Error." << endl;
               valid = false;
            }
         
            if ( !valid ) {
               cerr << "Cofiguration File Error" << endl;
               return false;
            }
         }
      }
   
      return true;
   }

	//===========================================================
   bool dewarMon::SetMonitorParam ( const std::string& value ) {
      ConfElements elements;
      if ( !GetElements ( value, elements ) ) 
         return false;
   
      ConfElements::iterator itr = elements.begin();
      ConfElements::iterator end = elements.end();
      for ( ; itr != end; ++itr ) {
         if ( TagCompare (itr->first, "NAME")  ) {
            fMonConf.fName = itr->second;
         }
         else if ( TagCompare (itr->first, "SITE")  ) {
            if ( TagCompare (itr->second, "LHO")  ) {
               fMonConf.fSite = MonitorConf::kLHO;
            }
            else if ( TagCompare (itr->second, "LLO")  ) {
               fMonConf.fSite = MonitorConf::kLLO;
            }
            else {
               cerr << "Invalid Site Name -> " << itr->second << endl;
               return false;
            }
         }
         else if ( TagCompare (itr->first, "DATAPATH")  ) {
            fMonConf.fDataPath = itr->second;
            if ( fMonConf.fDataPath[fMonConf.fDataPath.size() - 1] != '/' ) {
               fMonConf.fDataPath += '/';
            }
         }
         else if ( TagCompare (itr->first, "LOG")  ) {
            fMonConf.fDoLog = GetBool ( itr->second );
         }
         else if ( TagCompare (itr->first, "DURATION")  ) {
            fMonConf.fDuration =  Interval ( atoi( itr->second.c_str() ) * 60, 0);
         }
         else if ( TagCompare (itr->first, "LOGINTERVAL")  ) {
            fMonConf.fLogInterval =  Interval ( atoi( itr->second.c_str() ) * 60, 0 );
         }
         else if ( TagCompare (itr->first, "TRIGGER")  ) {
            fMonConf.fDoTrigger = GetBool ( itr->second );
         }
         else if ( TagCompare (itr->first, "TREND")  ) {
            fMonConf.fDoTrend = GetBool ( itr->second );
         }
         else if ( TagCompare (itr->first, "TSINTERVAL")  ) {
            int min = atoi( itr->second.c_str() );
         	// min has to be within 4 - 24 hrs
            if ( min > 60 * 24 ) min = 60 * 24;
            else if ( min < 240 ) min = 240;
         
            fMonConf.fTSInterval = min;
         }
         else {
            cerr << "Monitor Parameter : Unknown Tag -> " << itr->first << endl;
            return false;
         }
      }
   
      return true;
   }

	//===========================================================
   bool dewarMon::AddFilterInfo ( const std::string& value ) {
      ConfElements elements; 
      if ( !GetElements ( value, elements ) ) 
         return false;
   
      FilterInfo f;
      std::string name;
   
      for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
         if ( TagCompare( itr->first, "NAME")  ) {
            name = itr->second;
            f.fName = itr->second;
         } 
         else if ( TagCompare( itr->first, "FORMULA") ) {
            f.fFormula = itr->second;
            std::string::size_type pos = 0;
            while ( ( pos = f.fFormula.find_first_of('\\', pos)) != std::string::npos ) {
               f.fFormula.erase( pos, 1 );
            }
         }
         else if ( TagCompare( itr->first, "LPEFLENGTH") ) {
            f.fLPEFLength = atoi( itr->second.c_str() );
         }
         else if ( TagCompare( itr->first, "LPEFTRAINPERIOD") ) {
            f.fLPEFTrainPeriod = atoi( itr->second.c_str() );
         }
         else if ( TagCompare( itr->first, "LPEFTRAINLENGTH") ) {
            f.fLPEFTrainLength = atoi( itr->second.c_str() );
         }
         else {
            cerr << "Filter : Unknown Tag -> " << itr->first << endl;
            return false;
         }
      }
   
      if ( name.empty() || fFList.find( name.c_str() ) != fFList.end() ) {
         cerr << "AddFilterInfo error." << endl;
         return false;
      }
   
      fFList[ name.c_str() ] = f;
   
      return true;
   
   }

	//===========================================================
   bool dewarMon::AddThresholdInfo ( const std::string& value ) {
      ConfElements elements;
      if ( !GetElements ( value, elements ) ) 
         return false;
   
      ThresholdInfo t;
      std::string name;
   
      for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
         if ( TagCompare( itr->first, "NAME")  ) {
            name = itr->second;
         }
         else if  (TagCompare( itr->first, "TYPE")  ) {
            if ( TagCompare( itr->second, "Absolute")  ) {
               t.fType = ThresholdInfo::kAbs;
            }
            else {
               t.fType = ThresholdInfo::kRel;
            }
         }
         else if ( TagCompare( itr->first, "LOW")  ) {
            t.fLow = atof ( itr->second.c_str() );
         }
         else if ( TagCompare( itr->first, "HIGH")  ) {
            t.fHigh = atof ( itr->second.c_str() );
         }
         else {
            cerr << "Threshold : Unknown Tag -> " << itr->first << endl;
            return false;
         }
      }
   
      if ( name.empty() || fTList.find ( name.c_str() ) != fTList.end() ) {
         cerr << "AddThresholdInfo error." << endl;
         return false;
      }
      fTList[ name.c_str() ] = t;
   
      return true;
   }

	//===========================================================
   bool dewarMon::AddParameter ( const std::string& value ) {
      ConfElements elements;
      if ( !GetElements ( value, elements ) ) 
         return false;
   
      Parameter p;
      std::string name;
   
      for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
         if ( TagCompare( itr->first, "NAME")  ) {
            name = itr->second;
         }
         else if ( TagCompare( itr->first, "MINSEPARATION")  ) {
            p.fMinSeparation = Interval ( atof ( itr->second.c_str() ) );
         }
         else if ( TagCompare( itr->first, "MINDURATION")  ) {
            p.fMinDuration = Interval ( atof ( itr->second.c_str() ) );
         }
         else if ( TagCompare( itr->first, "MAXDURATION")  ) {
            p.fMaxDuration = Interval ( atof ( itr->second.c_str() ) );
         }
         else if ( TagCompare( itr->first, "MINDENSITY")  ) {
            p.fMinDensity = atof ( itr->second.c_str() );
         }
         else {
            cerr << "Parameter : Unknown Tag -> " << itr->first << endl;
            return false;
         }
      }
   
      if ( name.empty() || fCPList.find( name.c_str() ) != fCPList.end() ) {
         cerr << "AddParameter error." << endl;
         return false;
      }
      fCPList[ name.c_str() ] = p;
   
      return true;
   }

	//===========================================================
   bool dewarMon::AddHistogramInfo ( const std::string& value ) {
      ConfElements elements;
   
      if ( !GetElements ( value, elements ) ) {
         return false;
      }
   
      HistogramInfo h;
      std::string name;
   
      for ( ConfElements::iterator itr = elements.begin();
      itr != elements.end(); ++itr ) {
         if ( TagCompare( itr->first, "NAME")  ) {
            name = itr->second;
         }
         else if ( TagCompare( itr->first, "BIN") ) {
            h.fNBins = atoi ( itr->second.c_str() );
         }
         else if ( TagCompare( itr->first, "LOW") ) {
            h.fLow = atof ( itr->second.c_str() );
         }
         else if ( TagCompare( itr->first, "HIGH") ) {
            h.fHigh = atof ( itr->second.c_str() );
         }
         else {
            cerr << "HistogramInfo : Unknown Tag -> " << itr->first << endl;
            return false;
         }
      }
   
      if ( name.empty() || fHList.find( name.c_str() ) != fHList.end() ) {
         cerr << "AddHistogramInfo error." << endl;
         return false;
      }
      fHList[ name.c_str() ] = h;
   
      return true;
   }

        //==========================================================
   bool dewarMon::AddSetInfo (const std::string& value ) {
     bool added_trigger = false;
     ConfElements elements;
     if ( !GetElements (value, elements) ) {
       return false;
     }
     SetInfo Set;
     std::string chname;
     int chlock = LockStatus::kInvalid;
     std::string second_new;
     ConfigList fConfList;

     for ( ConfElements::iterator itr = elements.begin();
	   itr != elements.end(); ++itr ) {
       if ( TagCompare( itr->first, "NAME") ) {
	 Set.SetName( itr->second );
       }
       else if (TagCompare( itr->first, "HISTOGRAM") ) {
	 HistogramList::iterator hitr = fHList.find( itr->second );
	 if ( hitr != fHList.end() ) {
	   Set.fHist = &(hitr->second);
	 }
       }
       else if ( TagCompare( itr->first, "CHANNEL")) {
	 fConfList.clear();
	 int pos1 = (itr->second).find_first_of('{',0);
	 int pos2 = (itr->second).find_last_of('}');
	 int diff = pos2-pos1-1;
	 if (diff>0) {
	   second_new = (itr->second).substr(pos1+1,diff);
	 } 
	 else {
	   second_new = itr->second;
	 }
	 
	 ConfElements celements;
	 GetElements(second_new, celements);
	 for ( ConfElements::iterator citr = celements.begin();
	       citr != celements.end(); ++citr ) {
	   if ( TagCompare( citr->first, "NAME")  ) {
	     chname = citr->second.c_str();
	     if (!added_trigger) {
	       Set.SetTrigger(citr->second);
	       added_trigger = true;
	     }
	     config_numbers c_numbers;
	     ChanConfig_map CCmap;
	   }
	   else if ( TagCompare( citr->first, "CHECKLOCK")  ) {
	     if ( TagCompare( citr->second, "H1")  ) {
	       chlock = LockStatus::kH1;
	     }
	     else if ( TagCompare( citr->second, "H2")  ) {
               chlock = LockStatus::kH2;
	     }
	     else if ( TagCompare( citr->second, "L1")  ) {
               chlock = LockStatus::kL1;
	     }
	     else if ( TagCompare( citr->second, "H1MC")  ) {
               chlock = LockStatus::kH1MC;
	     }
	     else if ( TagCompare( citr->second, "H2MC")  ) {
               chlock = LockStatus::kH2MC;
	     }
	     else if ( TagCompare( citr->second, "L1MC")  ) {
               chlock = LockStatus::kL1MC;
	     }
	     else {
               chlock = LockStatus::kInvalid;
	     }
	   }
	   else if ( TagCompare( citr->first, "CONFIG") ) {
	     ConfigInfo cinfo;
	     if ( GetConfigInfo ( citr->second, cinfo ) ) {
	       fConfList.push_back(cinfo);
	     }
	     else {
               cerr << value << endl;
               return false;
	     }
	   }
	 }

	 AddChannelInfo ( chname, chlock, fConfList, Set );
       }
       else {
	 cerr << "SetInfo : Unknown Tag -> " << itr->first << endl;
         return false;
       }
     }
     
     for ( SetList::iterator itr = fSList.begin(); 
      itr != fSList.end(); ++itr) {
        if ( TagCompare( itr->GetName(), (Set.GetName()).c_str() ) ) {
           cerr << "Set Name Redundancy." << endl;
           return false;
        }
     }
     fSList.push_back( Set );
     return true;
   }



	//===========================================================
    bool dewarMon::AddChannelInfo ( std::string& name, const int Chlock, 
				    ConfigList& ConfList, SetInfo& Set) {
      ChannelInfo chinfo;
      bool newchannel = true;
      int old_ch_number = 0;
      config_numbers c_num;
      c_num.clear();
      int current_ch = 0;
      for (ChannelList::iterator itr = fChList.begin();
	   itr != fChList.end(); ++itr) {
	if (TagCompare(itr->GetName(),name.c_str())) {
	  newchannel = false;
	  old_ch_number = current_ch;
	}
	current_ch++;
      }
      if (newchannel) {
	chinfo.SetName(name.c_str());
	chinfo.SetCheckLock(Chlock);
	int current_conf = 0;
	for (ConfigList::iterator citr = ConfList.begin();
	     citr != ConfList.end(); ++citr) {
	  chinfo.AddConfig(*citr);
	  c_num.push_back(current_conf);
	  current_conf++;
	}
	fChList.push_back(chinfo);
	for (config_numbers::iterator cnumitr = c_num.begin(); cnumitr != c_num.end(); ++cnumitr) {
	}
	Set.AddPair(name, c_num);
      }
      else {
	int current_conf = (fChList.at(old_ch_number)).getNConf();
	for (ConfigList::iterator citr = ConfList.begin();
	     citr != ConfList.end(); ++citr) {
	  bool same_conf = false;
	  for (int i=0; i<current_conf; ++i) {
	    ConfigInfo Cinfo;
	    (fChList.at(old_ch_number)).getCinfo(i, Cinfo);

	    if ((*(Cinfo.fThreshold) == *((*citr).fThreshold))&&(*(Cinfo.fFInfo) == *((*citr).fFInfo))&&(*(Cinfo.fParam) == *((*citr).fParam))) {
	      c_num.push_back(i);
	      same_conf = true;
	    }
	  }
	  if (same_conf == false) {
	    c_num.push_back(current_conf);
	    (fChList.at(old_ch_number)).AddConfig(*citr);
	  }
	}
	for (config_numbers::iterator cnumitr = c_num.begin(); cnumitr != c_num.end(); ++cnumitr) {
	}
	Set.AddPair(name, c_num);
      }

      return true;
    }


	//===========================================================
   bool dewarMon::GetConfigInfo ( const std::string& value, ConfigInfo& cinfo ) {
   
      std::string::size_type brkt_beg = value.find_first_of ('{');
      std::string::size_type brkt_end = value.find_last_of ('}');
      if ( brkt_beg >= brkt_end || brkt_beg == std::string::npos || 
      brkt_end == std::string::npos ) {
         cerr << "Config info {} mismatch." << endl;
         return false;
      }
   
      ConfElements elements;
      if ( !GetElements ( value.substr ( brkt_beg + 1, brkt_end - brkt_beg - 1), elements ) ) 
         return false;
   
      bool nothreshold = true;
      bool nofilter = true;
      bool noparameter = true;
      // bool nohistogram = true;
   
      bool ret = true;
      for ( ConfElements::iterator itr = elements.begin();
      itr != elements.end(); ++itr ) {
         if ( itr->first == "THRESHOLD" ) {
            ThresholdList::iterator titr = fTList.find( itr->second );
            if ( titr != fTList.end() ) {
               cinfo.fThreshold = &(titr->second);
               nothreshold= false;
            }
            else {
               cerr << "Can't find THRESHOLD " << itr->second << endl;
               ret = false;
            }
         }
         else if ( itr->first == "FILTER" ) {
            FilterList::iterator fitr = fFList.find( itr->second );
            if ( fitr != fFList.end() ) {
               cinfo.fFInfo = &(fitr->second);
               nofilter = false;
            }
            else {
               cerr << "Can't find FILTER " << itr->second << endl;
               ret = false;
            }
         
         }
         else if ( itr->first == "PARAMETER" ) {
            ParameterList::iterator pitr = fCPList.find( itr->second );
            if ( pitr != fCPList.end() ) {
               cinfo.fParam = &(pitr->second);
               noparameter = false;
            }
            else {
               cerr << "Can't find PARAMETER " << itr->second << endl;
               ret = false;
            }
         }
        //  else if ( itr->first == "HISTOGRAM" ) {
//             HistogramList::iterator hitr = fHList.find( itr->second );
//             if ( hitr != fHList.end() ) {
//                cinfo.fHist = &(hitr->second);
//                nohistogram = false;
//             }
//             else {
//                cerr << "Can't find HISTOGRAM " << itr->second << endl;
//                ret = false;
//             }
//          }
         else if ( itr->first == "COMMENT" ) {
            cinfo.fComment = itr->second;
         }
         else {
            cerr << "Configuration : Unknown Tag -> " << itr->first << endl;
            ret = false;
         }
      
      }
   
      if ( nothreshold ) {
         ThresholdList::iterator titr = fTList.find( "Default" );
         cinfo.fThreshold = &(titr->second);
      }
      if ( nofilter ) {
         FilterList::iterator fitr = fFList.find( "Default" );
         cinfo.fFInfo = &(fitr->second);
      }
      if ( noparameter ) {
         ParameterList::iterator pitr = fCPList.find( "Default" );
         cinfo.fParam = &(pitr->second);
      }
      // if ( nohistogram ) {
//          HistogramList::iterator hitr = fHList.find( "Default" );
//          cinfo.fHist = &(hitr->second);
//       }
   
      if ( !ret ) {
         cerr << "ConfigInfo error." << endl;
         return false;
      }
   
      return true;
   
   }

