/*  File burstMon.cc      
    author: Sergey Klimenko, Andrey Sazonov, University of Florida
*/

#ifndef _USE_DMT
#define _USE_DMT
#endif

#ifndef __CINT__

//  The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/burstMon/burstMon.cc 6385 2010-12-10 23:00:42Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Burst Sensitivity Monitor"
#include "ProcIdent.hh"


#include <fstream>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>

// next two lines are needed to check output directory access rights
#include <unistd.h>
#include <errno.h>

#include "TSeries.hh"
#include "Dacc.hh"
#include "Interval.hh"
#include "Time.hh"
#include "DVector.hh"

#include "burstMon.hh"
#include "wseries.hh"
#include "cluster.hh"
#include "Biorthogonal.hh"
#include "Symlet.hh"

#include <time.h>	// for CPU usage timing

//#include "xsil/Xwriter.hh"
//#include "xsil/xsil.hh"

EXECDAT(BurstMon)        // Generate the main routine.

#endif               // !def(__CINT__)

static void tmark(const char* m)
{ // print out CPU time usage
   double e_time;
   e_time = double(clock())/double(CLOCKS_PER_SEC);
   printf("CPU time: %7.3lf sec, %s\n", e_time, m);
   return;
}

// definitions of constants
#define ATTO_SCALE 1.e-18
#define ZEPTO_SCALE 1.e-21
#define H50_SCALE ZEPTO_SCALE


using namespace std;

//--- BurstMon constructor -------------
//    double   timeGate;		// coincidence time gate
//    double   pFraction;		// pixel fraction selected for cluster analysis
//    double   timeStride;	// time stride for input data
//    double   samplingRate;	// data sampling rate for analysis
//    double   resolution;	// frequency resolution for cluster analysis 
//    double   injectionWindow;   // injection window
//    double   wResolution;       // frequency resolution for data conditioning
//    double   liveTime;		// processed data length
//    double   totalTime;		// run time of monitor
//    double   offsetTime;	// length of data padding
//    double   startTime;		// first data start time
//    int      minClusterSize;	// parameter for cluster analysis
//    bool     includeHalo;	// parameter for cluster analysis
//    int      historyLength;   	// lengths of data for DMTviewer
//    int      verbose;		// stdout verbosity level 0..10
//    int      waveOrder;		// Symlet order for cluster analysis
//    bool     firstCall;		// true at first data input
//    bool     offLine;		// true if monitor works offline
//    int      whiteLevel;	// DWT transform level for whitening
//    int      analyLevel;	// DWT transform level for cluster analysis

BurstMon::BurstMon(int argc, const char *argv[]) 
  : DatEnv(argc, argv), timeGate(0.1), pFraction(1.), ifoLength(4000.),
    timeStride(60.), Threshold(0.), targetFAR(0.), samplingRate(8192.), 
    resolution(64.), injectionWindow(15.), wResolution(32.), 
    liveTime(0.), deadTime(0.), totalTime(0.), offsetTime(2.), startTime(0.),
    minClusterSize(1), includeHalo(0), historyLength(720), verbose(0), waveOrder(32),
    firstCall(true), offLine(false), ezCal(false), whiteLevel(9), analyLevel(7),
    approxLevel(true), mCalibrate(0), mOSC(getDacc()), masterTS(NULL), Injections(300)
{ 
   FILE* fp;
   char dir[256];

   char *w;
   char offlineFile[256];
   char monitorName[256];
   char frameFileName[256];
   char htmlDir[256];
   char trendDir[256];

   bool configure   = false;
   wavearray<double> ww;

   sprintf(masterChannelName,"%s", "" );
   sprintf(configFile,       "%s", "" );
   sprintf(summaryFile,      "%s", "" );
   sprintf(indexFile,        "%s", "" );
   sprintf(responseFile,     "%s", "" );
   sprintf(htmlDir,          "%s", "" );
   sprintf(trendDir,         "%s", "" );
   sprintf(lockCondition,    "%s", "" );
   sprintf(suffix,           "%s", "" );

/***************************************************************
 * The BurstMon parameters can be set either from command line *
 * or from configuration file (option -i fileName)             *
 ***************************************************************/

// command line input of BurstMon parameters

   if (argc < 2) {
     cout <<"\n -> BurstMon error: too few command line arguments\n"<<endl;
     dumpHelp();
   }
   else {
      for(int ac = 1; ac < argc; ++ac) {

	 if ( strcmp(argv[ac],"-i") == 0 ){
	   sprintf(configFile,"%s",argv[++ac]);
	   configure = true;
	 }
	 
	 else if ( strcmp(argv[ac],"-c") == 0 ){
	   sprintf(masterChannelName,"%s",argv[++ac]);
	   cout <<"master channel             = "<<masterChannelName<<endl;
	 }

         else if ( strcmp(argv[ac],"-inject") == 0 ){
	   w = new char[256];
	   sprintf(w,"%s",argv[++ac]);
           if (strlen(w) == 0) 
           {
              cout <<"\n -> BurstMon error:  null waveform file name\n"<<endl;
              dumpHelp();
           }
	   waveformFileName.push_back(w);
	   cout <<"waveform file              = "<<w<<endl;
	 }

	 else if ( strcmp(argv[ac],"-d") == 0 ){
	   sprintf(summaryFile,"%s",argv[++ac]);
	 }

	 else if ( strcmp(argv[ac],"-lock") == 0 ){
	   sprintf(lockCondition,"%s",argv[++ac]);
	 }

	 else if ( strcmp(argv[ac],"-suffix") == 0 ){
	   sprintf(suffix,"%s",argv[++ac]);
	 }

	 else if ( strcmp(argv[ac],"-infile") == 0 ){
	   sprintf(offlineFile,"%s",argv[++ac]);
	   offLine = true;
	 }

	 else if ( strcmp(argv[ac],"-inlist") == 0 ){
	   sprintf(offlineFile,"%s",argv[++ac]);
	   if ( (fp=fopen(offlineFile,"r")) == NULL ){ 
             cout << "\n -> BurstMon error: cannot open list of input files \""
                 << offlineFile <<"\"\n"<<endl;
	     dumpHelp(); 
	   }
	   else fclose(fp);
	 }

	 else if ( strcmp(argv[ac],"-f") == 0 )
	   sscanf(argv[++ac],"%lf",&resolution);

	 else if ( strcmp(argv[ac],"-a") == 0 )
	   approxLevel = false;

         else if ( strcmp(argv[ac],"-white") == 0 )
	   sscanf(argv[++ac],"%lf",&wResolution);

	 else if ( strcmp(argv[ac],"-v") == 0 )
	   sscanf(argv[++ac],"%d",&verbose);

	 else if ( strcmp(argv[ac],"-wavelet") == 0 )
	   sscanf(argv[++ac],"%d",&waveOrder);

	 else if ( strcmp(argv[ac],"-gate") == 0 )
	   sscanf(argv[++ac],"%lf",&timeGate);

	 else if ( strcmp(argv[ac],"-far") == 0 )
	   sscanf(argv[++ac],"%lf",&targetFAR);

	 else if ( strcmp(argv[ac],"-bpp") == 0 )
	   sscanf(argv[++ac],"%lf",&pFraction);

	 else if ( strcmp(argv[ac],"-rate") == 0 )
	   sscanf(argv[++ac],"%lf",&samplingRate);

	 else if ( strcmp(argv[ac],"-stride") == 0 )
	   sscanf(argv[++ac],"%lf",&timeStride);

         else if ( strcmp(argv[ac],"-window") == 0 )
	   sscanf(argv[++ac],"%lf",&injectionWindow);

         else if ( strcmp(argv[ac],"-size") == 0 )
	   sscanf(argv[++ac],"%d",&minClusterSize);

         else if ( strcmp(argv[ac],"-history") == 0 )
            sscanf(argv[++ac],"%d",&historyLength);

         else if ( strcmp(argv[ac],"-offset") == 0 )
            sscanf(argv[++ac],"%lf",&offsetTime);

         else if ( strcmp(argv[ac],"-events") == 0 )
            sscanf(argv[++ac],"%d",&Injections);

         else if ( strcmp(argv[ac],"-scut") == 0 )
            sscanf(argv[++ac],"%lf",&Threshold);

         else if ( strcmp(argv[ac],"-resp") == 0 ) {
            sprintf(responseFile,"%s",argv[++ac]);
	    ezCal = true;
	 }

         else if ( strcmp(argv[ac],"-gps") == 0 ) {
	   sscanf(argv[++ac],"%lf",&startTime);
	 }

         else cout <<"unknown argument: "<<argv[ac]<<endl;
      }
   }

   if (configure) readConfig(configFile);

// check parameters

   timeStride = double(int(timeStride+0.5));
   offsetTime = double(int(offsetTime+0.5));
   if (timeStride<=0.) timeStride = 60.;
   if (offsetTime<=2.) offsetTime = 2.;

   if (resolution<=0.) resolution = 64.;
   if (wResolution<=0.) wResolution = 16.;
   if (samplingRate<=0.) samplingRate = 8192.;

// frequency resolutions and rate must be integer powers of 2

   samplingRate = pow(2., int(log(samplingRate)/log(2.)+0.5));
   resolution = pow(2., int(log(resolution)/log(2.)+0.5));
   wResolution = pow(2., int(log(wResolution)/log(2.)+0.5));

// print out parameters
   cout <<" time stride, s            = "<<timeStride<<endl;
   cout <<" sampling rate, Hz         = "<<samplingRate<<endl;
   cout <<" resolution, Hz            = "<<resolution<<endl;
   cout <<" injection interval, s     = "<<injectionWindow<<endl; 
   cout <<" whitening resolution, Hz  = "<<wResolution<<endl;
   cout <<" black pixel probability   = "<<pFraction<<endl;       
   cout <<" significance cut          = "<<Threshold<<endl;         
   cout <<" false alarm rate, Hz      = "<<targetFAR<<endl;         
   cout <<" min cluster size          = "<<minClusterSize<<endl;         
   cout <<" data overlap buffer, s    = "<<offsetTime<<endl;       
   cout <<" dmtviewer history length  = "<<historyLength<<endl;       
   cout <<" time gate                 = "<<fabs(timeGate)<<endl;     
   cout <<" wavefilter length         = "<<waveOrder<<endl;     
   cout <<" number of injections      = "<<Injections<<endl;

   if (strlen(lockCondition))
   cout <<" lock condition            = "<<lockCondition<<endl;         

   readLockLoss(offLine);

// check directories defined by environment variables
// permissions must be 'rwx'

   if(check_directory_access(getenv("DMTHTMLOUT"), W_OK | R_OK | X_OK)){
     sprintf(htmlDir,"%s/",getenv("DMTHTMLOUT"));
   }

   check_directory_access(getenv("DMTOUTPUT"),  W_OK | R_OK | X_OK);

   if(check_directory_access(getenv("DMTRENDOUT"), W_OK | R_OK | X_OK)) { 
     sprintf(trendDir,"%s/",getenv("DMTRENDOUT"));
   }

// check initialization
// no channel is specified

   if (!(strlen(masterChannelName)>2 && masterChannelName[2] == ':'))
   {  
     cout <<"\n -> BurstMon error: no channel is specified.\n"<<endl;
     dumpHelp();
   }

// no waveform file is specified

   if (waveformFileName.size() == 0)
   {
      cout <<"\n -> BurstMon worning: no injection waveform is given.\n"<<endl;
   }

   string ifo(masterChannelName, 2);

// set first stride equal to 'offsetTime+timeStride' to fill BurstMon buffer

   getDacc().setStride(timeStride+offsetTime);
   getDacc().setIgnoreMissingChannel(true);

// set master channel name for input

   getDacc().addChannel(masterChannelName);
   masterTS = NULL;

   std::vector<double> empty_vector;	// empty vector for initialization
   Symlet<double>      sym(waveOrder,1);

   whiteLevel = int(log(samplingRate/wResolution)/log(2.)-0.5);
   analyLevel = int(log(samplingRate/resolution)/log(2.)-0.5);

   if (verbose>8)
     cout <<" whitening level = "<<whiteLevel
         <<", cluster analysis level = "<<analyLevel<<endl;

// read in response file if file name supplied

   if ( strlen(responseFile) != 0 ) {
     ezCal = true;
     mCalibrate = new FDEasyCalibrate(&getDacc(), responseFile, true, 
				      wResolution/2, wResolution, 
				      int(samplingRate/2./wResolution+0.5));
     if(strcmp(ifo.c_str(),"H1")==0) ifoLength = 4000.;
     if(strcmp(ifo.c_str(),"H2")==0) ifoLength = 2000.;
     if(strcmp(ifo.c_str(),"L1")==0) ifoLength = 4000.;
   }

// set up all injection waveforms

   for (size_t i=0; i<waveformFileName.size(); i++)
   {

// set up storage associated with each waveform

       wf.push_back(new wavearray<double>);	// for waveform
      wfT.push_back(new WSeries<double>);	// for tranformed waveform
      wfS.push_back(new WSeries<double>);	// for scaled waveform
      h50.push_back(0.);		// for amplitude estimation
      err_h50.push_back(0.);		// for errror
      peakTime.push_back(0.);		// injection peak time
      amp.push_back(0.);	        // vector of injection amplitudes
      ADC.push_back(0.);	        // injection amps in ADC counts

// create history time series for DMTviewer

      history_h50.push_back(new TSeries());
      history_noise.push_back(new TSeries());
      history_h50err.push_back(new TSeries());

// read waveform, resample and normalize to unity hrss,  store peak time

      peakTime.at(i) = read_waveform(*wf.at(i), waveformFileName.at(i)); 

// transform waveform down to whitening level

      wfT.at(i)->Forward(*(wf.at(i)), sym, whiteLevel);
      wfT.at(i)->getLayer(ww,0);
      ww = 0.;
      wfT.at(i)->putLayer(ww,0);

      if (verbose>8) {
        cout <<" waveform max level = "<<wfT.at(i)->getMaxLevel()<<endl;
        cout <<" waveform rms       = "<<wfT.at(i)->rms()<<endl;
      }

      if (whiteLevel>wfT.at(i)->getMaxLevel())
      {
        cout <<" BurstMon error: waveform in file '"<<waveformFileName.at(i)
            <<"' is too short for \n"
            <<" specified whitening frequency resolution"<<endl;
        exit(1);
      }
   }

// setting up trends and other output

// set Monitor Name

   string S("");
   if (strlen(suffix)<1) {
     sprintf(monitorName, "BurstMon_%s", masterChannelName);
   }
   else {
     sprintf(monitorName, "BurstMon_%s_%s", ifo.c_str(), suffix);
     S = string("_") + string(suffix);
   }

   while((w = strchr(monitorName, ':'))) w[0] = '_';
   setServerName(monitorName);
   cout <<" server name: "<< monitorName << endl;

   sprintf(frameFileName, "%s", monitorName);
   while((w = strchr(frameFileName, '-'))) w[0] = '_';

// set file name for summary

   if (strlen(summaryFile)){
     sprintf(dir, "%s%s", htmlDir, summaryFile);
     sprintf(summaryFile, "%s", dir);
   }
   else{
     sprintf(summaryFile, "%s%s.txt", htmlDir, monitorName);
   }

// replace ':' by '-'
   while((w = strchr(summaryFile, ':'))) w[0] = '-';

   cout <<" output summary file name: "<< summaryFile << endl;

// set up 1-minute trend for cluster rate, h50, etc..
   mTrend.setName(frameFileName);	// define trend file name
   mTrend.setType(Trend::kMinute);
//   mTrend.setFrameCount(1);
   mTrend.setFrameLen(60);		// one hour frame length
   mTrend.setIFO(ifo.c_str());
   mTrend.setAutoUpdate(false);

// set up 1-second trend for noise variability
   sTrend.setName(frameFileName);	// define trend file name
   sTrend.setType(Trend::kSecond);
//   sTrend.setFrameCount(1);
   sTrend.setFrameLen(3600);		// one hour frame length
   sTrend.setIFO(ifo.c_str());
   sTrend.setAutoUpdate(false);

   char theName[256];           	// temporary storage for names
   wavearray<float>* pf;
   pf = new wavearray<float>(historyLength);
   *pf = 0.;			        // array of zeroes to initialize history

   Time historyStart;
   if(startTime>0.) historyStart = Time(int(startTime));
   else             historyStart = Time(Now().getS());
   historyStart -= Interval(historyLength*timeStride);

   if(verbose>8) cout<<"history start: "<<historyStart.getS()<<endl;

// set up trend for burst quality factors
   sprintf(trendName_mpf1, "%s:DMT-BRST_pixelFraction_gt2%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_mpf1);
   cout <<" trend channel: "<<trendName_mpf1<<endl;
   sprintf(trendName_mpf2, "%s:DMT-BRST_pixelFraction_gt4%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_mpf2);
   cout <<" trend channel: "<<trendName_mpf2<<endl;
   sprintf(trendName_mpf3, "%s:DMT-BRST_pixelFraction_gt6%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_mpf3);
   cout <<" trend channel: "<<trendName_mpf3<<endl;

// set up DMTViewer series for MPF
   history_mpf1 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) fraction of pixels in clusters with size > 2", ifo.c_str(),S.c_str());
   serveData(theName, history_mpf1,
             "fraction of pixels in clusters with size > 2");
   cout <<" DMTViewer name: "<<theName<<endl;

   history_mpf2 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) fraction of pixels in clusters with size > 4", ifo.c_str(),S.c_str());
   serveData(theName, history_mpf2, 
             "fraction of pixels in clusters with size > 4");
   cout <<" DMTViewer name: "<<theName<<endl;

   history_mpf3 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) fraction of pixels in clusters with size > 6", ifo.c_str(),S.c_str());
   serveData(theName, history_mpf3, 
             "fraction of pixels in clusters with size > 6");
   cout <<" DMTViewer name: "<<theName<<endl;

// set up trend for burst pixel rates
   sprintf(trendName_pxr1, "%s:DMT-BRST_pixelRate_gt2%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_pxr1);
   cout <<" trend channel: "<<trendName_pxr1<<endl;
   sprintf(trendName_pxr2, "%s:DMT-BRST_pixelRate_gt4%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_pxr2);
   cout <<" trend channel: "<<trendName_pxr2<<endl;
   sprintf(trendName_pxr3, "%s:DMT-BRST_pixelRate_gt6%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_pxr3);
   cout <<" trend channel: "<<trendName_pxr3<<endl;

// pixel rate for DMT viewer

   history_pxr1 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) pixel rate (Hz) for clusters with size > 2", ifo.c_str(),S.c_str());
   serveData(theName, history_pxr1, 
             "pixel rate (Hz) for clusters with size > 2");
   cout <<" DMTViewer name: "<<theName<<endl;

   history_pxr2 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) pixel rate (Hz) for clusters with size > 4", ifo.c_str(),S.c_str());
   serveData(theName, history_pxr2, 
             "pixel rate (Hz) for clusters with size > 4");
   cout <<" DMTViewer name: "<<theName<<endl;

   history_pxr3 = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "%s(%s) pixel rate (Hz) for clusters with size > 6", ifo.c_str(),S.c_str());
   serveData(theName, history_pxr3, 
             "pixel rate (Hz) for clusters with size > 6");
   cout <<" DMTViewer name: "<<theName<<endl;

// set up trend for cluster rate
   sprintf(trendName_clusterRate, "%s:DMT-BRST_clusterRate%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_clusterRate);
   cout <<" trend channel: "<<trendName_clusterRate<<endl;

// set up DMTViewer series for cluster rate
   history_clusterRate = new TSeries(historyStart, Interval(timeStride),
                                     historyLength, pf->data);
   sprintf(theName, "%s(%s) cluster rate (Hz)", ifo.c_str(),S.c_str());
   serveData(theName, history_clusterRate, 
             "rate of clusters, Hz");
   cout <<" DMTViewer name: "<<theName<<endl;


// set up trend for frequency
   sprintf(trendName_freq, "%s:DMT-BRST_loudestClusterFreq%s", 
	   ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_freq);
   cout <<" trend channel: "<<trendName_freq<<endl;

// set up DMTViewer series for frequency
   history_freq = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "frequency (Hz) of loudest cluster in %s(%s)", 
	   ifo.c_str(),S.c_str());
   serveData(theName, history_freq, "frequency, Hz");
   cout << " DMTViewer name: " << theName << endl;

// set up trend for SNR
   sprintf(trendName_SNR, "%s:DMT-BRST_loudestClusterSNR%s", 
	   ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_SNR);
   cout <<" trend channel: "<<trendName_SNR<<endl;

// set up DMTViewer series for SNR
   history_SNR = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "SNR of loudest cluster in %s(%s)", ifo.c_str(),S.c_str());
   serveData(theName, history_SNR, 
             "cluster SNR");
   cout <<" DMTViewer name: "<<theName<<endl;

// set up trend for time stamp
   sprintf(trendName_time, "%s:DMT-BRST_loudestClusterTime%s", ifo.c_str(),S.c_str());
   mTrend.addChannel(trendName_time);
   cout <<" trend channel: "<<trendName_time<<endl;

// set up DMTViewer series for time stamp
   history_time = new TSeries(historyStart, Interval(timeStride),
			      historyLength, pf->data);
   sprintf(theName, "relative time of loudest cluster in %s(%s)", ifo.c_str(),S.c_str());
   serveData(theName, history_time, 
             "cluster time relative to stride start");
   cout <<" DMTViewer name: "<<theName<<endl;


// set up trend names and DMDTViewer time series for each waveform
   for (size_t i=0; i<trendName_h50.size(); i++)
   {

// create empty history
     history_h50.at(i)->setData(historyStart, Interval(timeStride),
                                pf->data, historyLength);
     history_noise.at(i)->setData(historyStart, Interval(timeStride),
                                pf->data, historyLength);
     history_h50err.at(i)->setData(historyStart, Interval(timeStride),
                                   pf->data, historyLength);

// set up DMTViewer series for h50 sensitivity

     if (H50_SCALE==ATTO_SCALE)
       sprintf(theName, "%s %s hrss@50%% (1.e-18 strain/sq(Hz))", 
	       ifo.c_str(), trendName_h50.at(i));
     else if(H50_SCALE==ZEPTO_SCALE)
       sprintf(theName, "%s %s hrss@50%% (1.e-21 strain/sq(Hz))", 
	       ifo.c_str(), trendName_h50.at(i));
     else
       sprintf(theName, "%s %s hrss@50%%", 
	       ifo.c_str(), trendName_h50.at(i));

     if (!ezCal) 
       sprintf(theName, "%s %s hrss@50%% (ADC counts)", 
	       ifo.c_str(), trendName_h50.at(i));

     serveData(theName, history_h50.at(i));
     cout <<" DMTViever name: '"<<theName<<"'"<<endl;

// set up DMTViewer series for noise

     if(H50_SCALE==ATTO_SCALE)
       sprintf(theName, "%s noise at %s (1.e-18 strain/sq(Hz))", 
	       ifo.c_str(), trendName_h50.at(i));
     else if(H50_SCALE==ZEPTO_SCALE)
       sprintf(theName, "%s noise at %s (1.e-21 strain/sq(Hz))", 
	       ifo.c_str(), trendName_h50.at(i));
     else
       sprintf(theName, "%s noise at %s", 
	       ifo.c_str(), trendName_h50.at(i));

     if (!ezCal) 
       sprintf(theName, "%s noise at %s (ADC counts)", 
	       ifo.c_str(), trendName_h50.at(i));

     serveData(theName, history_noise.at(i));
     cout <<" DMTViever name: '"<<theName<<"'"<<endl;

// hrss error
     sprintf(theName, "%s %s hrss50 error (%%)", ifo.c_str(), trendName_h50.at(i));
     serveData(theName, history_h50err.at(i));
     cout <<" DMTViever name: '"<<theName<<"'"<<endl;

// trends
     sprintf(theName, "%s:DMT-BRST_%s_noise%s", ifo.c_str(), trendName_h50.at(i),S.c_str());
     strcpy(trendName_noise.at(i), theName); 
     sprintf(theName, "%s:DMT-BRST_%s_hrss_50%s", ifo.c_str(), trendName_h50.at(i),S.c_str());
     strcpy(trendName_h50.at(i), theName);
     mTrend.addChannel(trendName_h50.at(i));
     cout <<" trend name: "<<trendName_h50.at(i)<<endl;
     mTrend.addChannel(trendName_noise.at(i));
     cout <<" trend name: "<<trendName_noise.at(i)<<endl;

   }

// set up trend for noise variability
   sprintf(trendName_nvar, "%s:DMT-BRST_variability%s", ifo.c_str(),S.c_str());
   sTrend.addChannel(trendName_nvar);
   cout <<" trend channel: "<<trendName_nvar<<endl;

// set up DMTViewer series for noise variability
   delete pf;

   double nvarStep = 1.;        // step for nvar history time series
   int nvarLength = 3600;       // length of noise variability history

   Time nvarStart;
   if(startTime>0.) nvarStart = Time(int(startTime));
   else             nvarStart = Time(Now().getS());
   nvarStart -= Interval(nvarLength+1);

   if(verbose>8) cout<<"variability history start: "<<nvarStart.getS()<<endl;

   pf = new wavearray<float>(nvarLength);
   *pf = 0.;
   history_nvar = new TSeries(nvarStart, Interval(nvarStep),
                              nvarLength, pf->data);

   sprintf(theName, "%s(%s) noise variability", ifo.c_str(),S.c_str());
   serveData(theName, history_nvar);
   delete pf;
   cout <<" DMTViewer name: "<<theName<<endl;

// get GPS time to construct output file names
   if (!offLine) sprintf(gpsTime, "%d", int(Now().getS()));

// set names for index files
   if (offLine)
     sprintf(indexFile,"%s%s_M.cfg",trendDir,monitorName);
   else
     sprintf(indexFile,"%s%s_M-%s.cfg",trendDir,monitorName,gpsTime);

   mTrend.writeIndex(indexFile);

   if (offLine)
     sprintf(indexFile,"%s%s_T.cfg",trendDir,monitorName);
   else
     sprintf(indexFile,"%s%s_T-%s.cfg",trendDir,monitorName,gpsTime);

   sTrend.writeIndex(indexFile);

}

//--------------------------------------  BurstMon destructor.
BurstMon::~BurstMon() 
{

// clear memory occupied by waveform data
  for (size_t i=0; i<waveformFileName.size(); i++)
  {
     delete wf.at(i);			// delete waveform
     delete wfT.at(i);			// delete tranformed waveform
     delete wfS.at(i);			// delete tranformed waveform
     delete [] trendName_h50.at(i);	// delete trend names
     delete [] trendName_noise.at(i);	// delete trend names
     delete history_h50.at(i);		// delete h50 history
     delete history_h50err.at(i);	// delete h50 error history
  }

  if (masterTS)    delete masterTS;
  if(mCalibrate)   delete mCalibrate;
  if(history_mpf1) delete history_mpf1;
  if(history_mpf2) delete history_mpf2;
  if(history_mpf3) delete history_mpf3;
  if(history_pxr1) delete history_pxr1;
  if(history_pxr2) delete history_pxr2;
  if(history_pxr3) delete history_pxr3;
  if(history_freq) delete history_freq;
  if(history_SNR)  delete history_SNR;
  if(history_time) delete history_time;
  if(history_nvar) delete history_nvar;
  if(history_clusterRate) delete history_clusterRate;

  mTrend.close();
  sTrend.close();

  cout << "BurstMon is finished" << endl;
}

//-- This skeleton function needed for writing to the dmt viewer

void BurstMon::Attention(void) {
  MonServer::Attention();
  return;
}

/*  Frame processing function.      *
 *  do this step every timestride.  */
void BurstMon::ProcessData(void) {

   size_t i,j,k,N;
   int n,m;
   int decimateLevel;		// DWT level for decimation of input
   bool inLock  = true;
   bool inTime = true;
   bool detected;
   bool tunehrss;
   double x, scale, hrss, step;
   double sm,sn,ss,rSF;
   double missTime;

   slice S;

   n = int(timeStride+offsetTime+0.1);    // start up variability buffer length
   
   Biorthogonal<double> B(waveOrder);
   Symlet<double>       sym(waveOrder,1);
   TSeries*             inputTS = NULL;	  // input for current stride	 
   WSeries<double>      U;                // temporary WS for injections
   WSeries<double>      W;                // temporary WS for injections
   WSeries<double>      wtmp;             // temporary WS for injections
   WSeries<double>*     wTmp;		  // WSeries for decimation	 
   WSeries<double>      wave;		  // WSeries for whitening	 
   WSeries<double>      wMaster(sym);	  // data after truncation of ends 
   WSeries<double>      pMaster(sym);	  // data after cluster analysis	 
   WSeries<double> 	wsNoise;          // WS for noise estimation	 
   WSeries<double> 	wsCalib;          // WS for calibration constants	 
   WSeries<float> 	wfCalib;          // WS for calibration constants	 
   wavearray<double>    *wInput = NULL ;  // input data from master channel
   wavearray<double>    master;		  // data after decimation	 
   wavearray<float>     noiseVar(n);	  // noise variability		 
   wavearray<float>     clusterTime;      // array of cluster times
   wavearray<float>     clusterSize;      // array of cluster energies
   wavearray<float>     clusterSign;      // array of cluster significances
   wavearray<float>     clusterFreq;      // array of cluster frequencies
   wavearray<float>     clusterSNR;       // array of cluster SNRs
   wavearray<float>     clusterID;        // array of cluster IDs
   wavearray<double>    buffer;           // wavearray buffer

   wavecluster          clusterList;      // cluster list

// set noiseVar for start up.
   noiseVar.rate(1.);

   if (verbose>9) tmark("ProcessData start"); 

   if (verbose>8) cout <<" read data, stride="
		       <<getDacc().getStride()<<" sec"<<endl;
   
   inputTS = getDacc().refData(masterChannelName);
   
   if (!inputTS) {cout <<"BurstMon: got no data!"<<endl; exit(1);}

   if (verbose>9)
   {
      cout <<" input data duration="<<inputTS->getInterval().GetSecs()
	   <<"."<<inputTS->getInterval().GetN()<<endl;
      cout <<" input data end="<<inputTS->getEndTime().getS()
	   <<"."<<inputTS->getEndTime().getN()<<endl;
   }

   missTime = 0.;     // missed time interval;
   if (firstCall)
   {
      firstCall = false;
      masterTS = new TSeries(*inputTS);
      startTime = masterTS->getStartTime().totalS();
      getDacc().setStride(timeStride);
      printf(" BurstMon start time: %20.9lf\n",startTime);
      
   } 
   else if (masterTS->Append(*inputTS)) 	// append failed
   {
      missTime = inputTS->getStartTime().totalS() - masterTS->getEndTime().totalS();
      *masterTS =  *inputTS;
      inTime = false;      
   } 
   else { 
      inTime = true;			// append succeded
   }

// check if input time is integer number of timeStrides from first data start

   double inputTime = masterTS->getStartTime().totalS();
   int    timeShift = int(inputTime-startTime+0.5) % int(timeStride+0.5);

   if(!inTime) deadTime += missTime + timeShift;
   
// if out of sync with previous data due to missed input
// then adjust input stride for next read operation

   if (timeShift>0)
   {
      inTime=false;
      masterTS->eraseStart(Interval(timeStride-double(timeShift)));
      getDacc().setStride(timeStride + offsetTime - 
			  masterTS->getInterval().GetSecs());
      if(verbose>0) printf("missing input: %f,  %d, time: %20.9lf - %20.9lf\n ",
			   missTime, timeShift,inputTime,startTime);
   } 
   else
   {
      inTime=true;
      getDacc().setStride(timeStride);
   }
   
// data start time to be processed (excluding padding)
   currentTime = masterTS->getStartTime()+Interval(offsetTime/2.);
   
// check if buffer is filled
   if (!masterTS->getInterval()==Interval(offsetTime+timeStride)) inTime=false;
   
   if (inTime)
   {
      wInput = new wavearray<double>(masterTS->getNSample());
      *wInput = *masterTS;
   }
   
// erase data start in input buffer leaving length = offsetTime
   if (inTime && masterTS->getInterval() > Interval(offsetTime))
      masterTS->eraseStart(masterTS->getInterval()-Interval(offsetTime));
   
   if (verbose>8 && !inTime)
      cout <<" input failure, need read more data "<<getDacc().getStride()
	   <<" sec long"<<endl;
   
   if (verbose>7){ printf(" lock condition: %s\n",lockCondition); }
   
// check lock condition
   
   if (strlen(lockCondition) && inLock) inLock = mOSC.satisfied(lockCondition);
   
   totalTime = masterTS->getEndTime().totalS()- startTime;
   
// fill in trends in case of failed lock condition
   
   if (!inLock && inTime)
   {
      noiseVar = 0.;		      // assign dummy values to variability
      update_trend_nvar(noiseVar);    // fill variability trend
      update_trend_mpf(NULL);	      // fill mpf trend with dummy values
      update_trend_rate(false);	      // fill rate trend with dummy values
      update_trend_freq(false);	      // fill frequency trend with dummy values
      update_trend_time(false);	      // fill time trend with dummy values
      update_trend_SNR(false);        // fill SNR trend with dummy values

      if(trendName_h50.size())
	update_trend_waveform(false); // fill waveform trends with dummy values

      mTrend.Update();
      sTrend.Update();
   }
   
   if (!inLock || !inTime) 
   {
      if (verbose>5) cout <<" inLock="<<inLock<<", inTime="<<inTime<<", "
			  <<" return to data input ..."<<endl;
      if (wInput) delete wInput;

      return;
   }

   liveTime += timeStride;
   if(verbose>8) cout<<"live time: "<<liveTime<<endl;
   
// decimate master channel 
   decimateLevel = int(log(wInput->rate()/samplingRate)/log(2.)+0.5);
   if (decimateLevel>0)
   {
      wTmp = new WSeries<double>(*wInput, B);
      wTmp->Forward(decimateLevel);
      wTmp->getLayer(master, 0);
      delete wTmp;
      
   } else 
      master = *wInput;
   
   if (wInput) delete wInput;
   
// transform data for whitening
   wMaster.Forward(master, sym, whiteLevel);
   
   if (verbose>8) cout <<" master channel has been transformed."<<endl;
   
//   wMaster=wave;
   
   if (verbose>8) cout <<" wMaster size="<<wMaster.size()<<endl;
   
   if (wMaster.getMaxLevel()<whiteLevel)
   {
      cout <<" BurstMon error: input data length "
	   << wMaster.size()/wMaster.rate()<<" sec"
	   <<" is too short for specified frequency resolution "
	   << resolution <<" Hz"<<endl;
      exit(1);
   }
   
   wsNoise  = wMaster.white(timeStride);

// get EZ calibration

   if (verbose>9) tmark("get calibration");

   if(ezCal) {

     mCalibrate->UpdateResponseFunction();
     modFResp = mCalibrate->getFSpectrum();

     if(modFResp.getNStep()+1 != wfCalib.size()) 
       wfCalib.resize(modFResp.getNStep()+1);

     if(verbose>9) 
       cout<<"FSeries length: "<<wfCalib.size()<<" layers: "<<wsNoise.size()<<endl;

     modFResp.getData(modFResp.getNStep()+1,wfCalib.data);

     if(wsCalib.size()!=wsNoise.size()) wsCalib=wsNoise;
     if(wsNoise.size() >wfCalib.size()) {
       cout<<"burstMon EZ calibration error"<<endl;
       exit(0);
     }

     for(i=1; i<wsNoise.size(); i++){
       S = wsCalib.getSlice(i);
       x = sqrt(wfCalib.data[i])/ifoLength/1.e9;
       wsCalib.data[S.start()] = x* wsNoise.data[S.start()];
     }

     if(verbose>9) {
       cout<<"first element="<<wsCalib.data[10]<<"  ";
       cout<<"low frequency="<<modFResp.getLowFreq()<<"  ";
       cout<<"hig frequency="<<modFResp.getHighFreq()<<"  ";
       cout<<"samples="<<modFResp.getNStep()+1<<endl;
     }     
   }
   else if(wsCalib.size() != wsNoise.size()) {
     wsCalib = wsNoise;
     wsCalib = 1.;
   }
  
   
   for (k=0; k<wfT.size(); k++)           // scale injections
   {
      amp.at(k) = getHRSS(*(wfT.at(k)), wsCalib);
      *(wfS.at(k)) = *(wfT.at(k));               // copy transformed waveforms
      ADC.at(k) = white(*(wfS.at(k)), wsNoise);  // store seed value of hrss

      if(whiteLevel > analyLevel) wfS.at(k)->Inverse(whiteLevel-analyLevel);
   }
   
   if (verbose>0) 
      printf(" data start time: %20.9lf, lock: %d\n", wMaster.start(), inLock);
   if (verbose>1) 
      printf(" data fill time: %ld\n",getDacc().getFillTime().getS());

// variability calculation
   if (verbose>9) tmark("start variability calculation");

   noiseVar = wMaster.variability(0.);
   update_trend_nvar(noiseVar);
      
// reconstruct data to lower frequency resolution for cluster analysis
   if (whiteLevel > analyLevel) 
      wMaster.Inverse(whiteLevel-analyLevel);
   

// cluster analysis
   if (verbose>9) tmark("start cluster analysis");
   
   S = wMaster.getSlice(0);
   step = S.stride()/wMaster.rate();    // time resolution

// clean-up approximation level
   if(!approxLevel){
     wMaster.getLayer(buffer,0);
     buffer = 0.;
     wMaster.putLayer(buffer,0);
   }
   
// cut off data edges, offsetTime/2. long at each end
   m  = int(offsetTime/step/2+0.1);              // offset number of towers
   n  = int(timeStride/step+0.1);                // total number of towers
   m *= S.stride();            // offset number of samples
   n *= S.stride();            // total number of samples
   
   if(size_t(m+n) <= wMaster.size()) W = wMaster[slice(m,n,1)];
   else W = wMaster;

   W.significance(injectionWindow,pFraction);
   clusterList.init(W,true);
   clusterList.cluster();
   clusterList.apush(wMaster,offsetTime/2.);

   clusterSign = clusterList.get("significance");  
   clusterFreq = clusterList.get("frequency,2");  
   clusterSize = clusterList.get("size");  
   clusterTime = clusterList.get("time,2");  
   clusterSNR  = clusterList.get("SNR",2);  
   clusterID   = clusterList.get("ID");
   
// update cluster rate
      
   N = clusterSize.size();
   double maxSign = 0.;
   double maxFreq = 0.;
   double maxSNR  = 0.;
   double maxTime = 0.;
   clusterRate = 0.;

   for (k=0; k<N; k++){

     if(clusterSign.data[k]>maxSign) {
       maxSign = clusterSign.data[k];
       maxFreq = clusterFreq.data[k];
       maxSNR  = clusterSNR.data[k];
       maxTime = clusterTime.data[k];
     }

     if(clusterSize.data[k] < minClusterSize) continue;

     if(clusterSign.data[k] < Threshold) {
       clusterList.ignore(size_t(clusterID.data[k]+0.5)); 
       continue;
     }

     clusterRate += 1./timeStride; 
   }
   
   if (verbose>8) cout <<" cluster rate="<<clusterRate<<endl;
   
   update_trend_rate(true, clusterRate);
   update_trend_freq(true, maxFreq);
   update_trend_time(true, maxTime);
   update_trend_SNR(true, sqrt(maxSNR));

// calculate significance threshold for a fixed rate

   if(N && targetFAR>0.) {
     x = targetFAR*timeStride/N;
     rSF = x<1. ? clusterSign.rank(x) : 0.;
   }
   else rSF = 0.;

   if (verbose>8) cout <<" significance threshold="<<rSF<<endl;

// update MPF

   clusterSize = clusterList.get("size");  
   update_trend_mpf(&clusterSize);
   

// ---------------------------
// start injection pipeline
// ---------------------------
   if (verbose>9 && wfT.size()) tmark("injections start");

// look over injections 
   for (k=0; k<wfT.size(); k++)
   {
      hrss = 4.*ADC.at(k);                 // copy seed value
      ss = sn = sm = 0.;
      wtmp = *(wfS.at(k));                 // copy waveform
      scale = hrss;
      x = offsetTime + wtmp.size()/wtmp.rate(); // overal offset
      tunehrss = true;

      n = m = 0;

      for (i=0; i<size_t(Injections);)
      {
	 j = size_t(offsetTime/step/2.+drand48()*(S.size()-x/step));
	 
//	 variability(wtmp, noiseVar,iTime);  // correct variability

	 if(scale != 1.) wtmp *= scale;
	 W = wtmp;
	 W.add(wMaster,wtmp.size(),j*S.stride());
	 U = W;

	 W.significance(0.,pFraction);
	 clusterList.init(W,true);
	 clusterList.cluster();
	 clusterList.apush(U);
	 clusterTime = clusterList.get("time",2);
	 clusterSign = clusterList.get("significance");  
	 clusterSize = clusterList.get("size");  
	 
	 N = clusterSize.size();
	 detected = false;
	 
	 for (j=0; j<N; j++){
	   if(fabs(clusterTime.data[j]-peakTime.at(k)) > timeGate) continue; 
	   if(clusterSign.data[j] < rSF) continue;
	   if(clusterSize.data[j] < minClusterSize) continue;
	   detected = true; break;
	 }

	 if(detected) { n++; sn += hrss; }
	 else         { m++; sm += hrss; }

	 if(tunehrss) {
	   scale = n>m ? 0.8 : 1.2;
	   if(n>1 && m>1){
	     if(verbose>7) cout<<"tune events: "<<n+m<<endl;
	     tunehrss = false;
	     n = m = 0;
	     sn = sm = 0.;
	   }
	 }

	 else {
	   ss += hrss*hrss;
	   if(n == m) scale = 1.;
	   else if(!detected && n>m) scale = 1.;
	   else if( detected && n<m) scale = 1.;
	   else scale = n>m ? 0.95 : 1.05;
	   i++;
	 }

	 hrss *= scale;
      }

      sm /= Injections;
      sn /= Injections;
      ss /= Injections;
      x = double(n-m)/(n+m);

      if(verbose>7) cout<<"asymmetry: "<<x<<endl;
      
      h50.at(k) = (x*ss + sm*sm - sn*sn)/(x*(sm+sn) + sm-sn);
      h50.at(k) *= amp.at(k)/ADC.at(k);
      err_h50.at(k) = sqrt((ss-pow(sn+sm,2))/(n+m))*100./(sn+sm);
     
   }	
   
   if(wfT.size()) update_trend_waveform(true);    // fill waveform trends with measured values
   
// -------------------------
// end of injection pipeline
// -------------------------

   mTrend.Update();
   sTrend.Update();

// dump summary
   dumpSummary(summaryFile);
   
   if (verbose >9) tmark("exit data processing");
}




/* procedures for injections */

double BurstMon::read_waveform(wavearray<double> &wa, char* file_name)
{
  ifstream infile; 
  infile.open(file_name);

  int level;

  int i,m;
  int n = -1;
  double x, peak_time=-1., f=-1.;
  double rms,mean,energy;		// for normalization

  char instr[256];
  char *w = NULL;
  char *trend;

  WSeries<double> ws;		// for decimation
  Biorthogonal<double> B(waveOrder);
  wavearray<double> wx;

  trend = new char[256];

  sprintf(trend,"%s","");

  i = 0;

  if (!infile.is_open())
  { 
     cout << " BurstMon::read_waveform() error: cannot open file '"
          << file_name << "'" << endl;
     exit(1);
  }

  if (verbose >0)
    cout <<" Reading waveform from file '"<< file_name << "'" << endl;

// skip file header, where each line starts with '#' character
  while ( infile.peek() == '#' && (! infile.eof()) )
  { 
     infile.getline(instr, 256);		// read in line
     pString = instr + 1;			// skip '#' symbol

// find and input waveform parameters
     while (strlen(w=getToken(pString)) > 1)
     {
       if (getParameter(w, "-size", n)) continue;
       if (getParameter(w, "-rate", f)) continue;
       if (getParameter(w, "-peak", peak_time)) continue;
       if (strcmp(w, "-name") == 0)
           sprintf(trend,"%s", w=getToken(pString, true));
     }
  }

  if (f <= 0.)
  {
    f = 16384.;			// use default
    cout <<" -> BurstMon:read_waveform() warning:\n"
        <<" -> default waveform rate used = " <<f<<" Hz"<<endl;
  }

  if (f < samplingRate)
  {
    cout <<"BurstMon::read_waveform() error:\n"
        <<"waveform in file '"<<file_name
        <<"'\n has sampling rate lower then required for analysis"<<endl;
    exit(1);
  }

  if (n <= 0.)
  {
    n = int(f);			// use default
    if (verbose>0) 
      cout <<" -> BurstMon::read_waveform() warning:\n"
          <<" -> default waveform length used = "
          <<n<<", "<<double(n)/f<<" sec"<<endl;
  }

  m = int(f*injectionWindow);

  wa.rate(f);
  wx.rate(f);
  if (int(wa.size()) != m ) wa.resize(m);
  if (int(wx.size()) != n ) wx.resize(n);
  wa = 0.;
  wa.start(0.);

  if (peak_time<0.)
  {
    peak_time = 0.5*double(m)/f;	// use default time of peak
    if (verbose>0)
      cout <<" -> BurstMon::read_waveform() warning:\n"
          <<" -> default peak time used = "
          << peak_time <<" sec"<<endl;
  }
  else {
    peak_time += (m-n)/f/2.;
  }

  if (timeStride < double(n)/f)
  {
    cout <<"BurstMon::read_waveform() error:\n"
        <<"waveform from file '"<<file_name
        <<"' is too long ("<<double(n)/f<<" sec)\n"
        <<"for selected time stride: "
        <<  timeStride <<" sec" <<endl;
    exit(1);
  }

// read data after header
  while ( i<n && (! infile.eof()) )
  {
     infile >> x;
     wx.data[i] = x;
     i++;
  }

  if (i<n)
  {
    cout <<"BurstMon::read_waveform() error: "
        <<"premature end of file '"<<file_name<<"'"<<endl;
    exit(1);
  }
  infile.close();

  wa.cpf(wx,n,0,(m-n)/2);    // put in the center

// decimate data if waveform sampling rate is higher then processing rate
  if (wa.rate() > samplingRate)
  {
     level = int(log(wa.rate()/samplingRate)/log(2.)+0.5);
     ws.Forward(wa, B, level);
     ws.getLayer(wa, 0);
  }

// normalize waveform
  wa.getStatistics(mean,rms);
  if (rms == 0)
  { cout <<"-> BurstMon::read_waveform() error: waveform from file \n'"
         <<file_name<<"' has zero RMS."<<endl;
    exit(1);
  }
  energy = wa.size()/wa.rate()*(rms*rms + mean*mean);
  wa *= 1./sqrt(energy);        // unity hrss
  
// create trend name
// if no "-name" option is specified then create trend name from file name
// removing directory name and file suffix
  if (strlen(trend) == 0)
  {
    w = strrchr(file_name,'/');   	// get pointer to last symbol '/'
    if ( w == NULL ) w = file_name;
    else w++;				// get trailing part of the file name
    strncat(trend, w, strcspn(w, ".")); // get file name without suffix
  }

  trendName_h50.push_back(trend);	// store trend name in vector

  if (verbose>9) 
    cout <<" waveform name '"<<trend
         <<"', peak-to-peak amplitude = "<<wa.max()-wa.min()<<endl;

  trend = new char[256];
  trendName_noise.push_back(trend);	// store trend name in vector

  return peak_time;
}




double BurstMon::white(WSeries<double> &signal, 
		       WSeries<double> &noise,
		       bool update)
{
// Layers of input signal are devided by corresponding values
// of noise produced  by WSeries::white()
//
// 'signal'	- signal in wavelet domain
// 'noise' 	- array of noise rms

  size_t i,j,n,m;
  double r,x;
  double sum = 0.;
  double hrss = 0.;
  slice S;

  if (noise.maxLayer() != signal.maxLayer())
  {
     cout << "BurstMon::white() error:"
          << " noise and signal number of layers do not match"
          << endl;
     exit(1);
  } 

  n = signal.size();

  for (i=1; i<=size_t(signal.maxLayer()); i++)
  {
     S = signal.getSlice(i);
     m = S.stride();
     r = noise.data[S.start()];         // noise in the layer

     for (j=S.start(); j<n; j+=m)	// for data samples in the layer
     {
	x = signal.data[j];
	hrss += x*x; x /= r; sum += x*x;
        if(update) signal.data[j] = x;		// normalize data
     }
  }

  if(verbose>8) cout<<"signal energy = "<<hrss<<endl;

  hrss = sqrt(1./sum);  // seed hrss                       

  return hrss;
}



void BurstMon::sync_history(TSeries *ts, float value )
{
   int err;
   wavearray<float> *dummy;
   if (!ts) return;
   Interval tsStep = ts->getTStep();  // sampling interval
   double step = tsStep.GetSecs();
   Interval tsSamp = ts->getInterval();  // ts duration
   int nsam = int(tsSamp.GetSecs()/60.);

   if(step > 2. && nsam != historyLength)
     cout<<"** history sync ERROR **: "<<nsam<<endl;

   double gap = currentTime - ts->getEndTime();
   Interval gapDuration(gap);
   if (verbose>9) {
     cout <<" history gap: " <<gapDuration.GetSecs()<<" sec"<<endl;
   }

// if gap is longer then history length or is negative then replace history
   if (gapDuration > ts->getInterval() || gap+ts->getInterval().GetSecs()<0.)
     gapDuration = ts->getInterval();

// if gap is <=0 then error - do not sync!
   if (gapDuration.GetSecs() < 0.) {
     cout <<" error: negative or zero history gap!" <<gapDuration.GetSecs()<<" sec"<<endl;
     return;
   }

   int gapSize = int(gapDuration/ts->getTStep());
   if(gapSize < 1) return;

   if (verbose>9) {
     cout <<" fill in gap size of "<<gapSize<<endl;
   }

   dummy = new wavearray<float>(gapSize);
   *dummy = value;
   
   if (gapDuration == ts->getInterval())
     {
       // completely replace data in history, performs Clear() and Append()
       ts->setData(currentTime-ts->getInterval(), ts->getTStep(),
                  dummy->data,  gapSize);
     } 
   else
     {
// or add data to history and erase start
       err=ts->Append(ts->getEndTime(), ts->getTStep(), dummy->data, gapSize);
       if (err==0) ts->eraseStart(gapDuration); 
     }
   delete dummy;

   return;
}

void BurstMon::update_trend_waveform(bool flag)
{
// update all waveform trends and histories
  float h50_last;
  float noise_last;
  float err_last;
  float dummy = 0.;

  if (verbose>8)
  { 
    cout <<" h50 history end="<<history_h50.at(0)->getEndTime().getS()<<endl;
    cout <<" current time="<<currentTime.getS()<<endl;
  }

  for (size_t k=0; k<history_h50.size(); k++)
  {
    if (flag && verbose>8)
    {
      cout <<"waveform="<<k<<", amp0="<<amp.at(k)<<", h50 ="<<h50.at(k)
          <<", err_h50="<<err_h50.at(k)<<" %"<<endl;
    }

    h50_last = (flag) ? h50.at(k): dummy;
    noise_last = (flag) ? amp.at(k): dummy;
    err_last = (flag) ? err_h50.at(k) : dummy;
   
// update trends
    mTrend.trendData(trendName_h50.at(k), currentTime, h50.at(k));
    mTrend.trendData(trendName_noise.at(k), currentTime, amp.at(k));

// history of h50 for DMTViewer
    sync_history(history_h50.at(k), dummy);		// fill gaps
    history_h50.at(k)->eraseStart(Interval(timeStride));
    if (ezCal) h50_last /= H50_SCALE;
    history_h50.at(k)->Append(history_h50.at(k)->getEndTime(),
                              Interval(timeStride), &h50_last, 1);

// history of noise hrss for DMTViewer
    sync_history(history_noise.at(k), dummy);		// fill gaps
    history_noise.at(k)->eraseStart(Interval(timeStride));
    if (ezCal) noise_last /= H50_SCALE;
    history_noise.at(k)->Append(history_noise.at(k)->getEndTime(),
                              Interval(timeStride), &noise_last, 1);

// history of h50 errors for DMTViewer
    sync_history(history_h50err.at(k), dummy);	// fill gaps
    history_h50err.at(k)->eraseStart(Interval(timeStride));
    history_h50err.at(k)->Append(history_h50err.at(k)->getEndTime(),
                                 Interval(timeStride), &err_last, 1);
  }

  return;
}

void BurstMon::update_trend_mpf(wavearray<float>* csize)
{
  float x;
  float mpf  = 0.;
  float mpf1 = 0.;
  float mpf2 = 0.;
  float mpf3 = 0.;

  size_t i,n;

  if(csize) {
    n = csize->size();
    for(i=0; i<n; i++) {
      x = csize->data[i]; 
      mpf  += x/timeStride; 
      if(x>2) mpf1 += x/timeStride; 
      if(x>4) mpf2 += x/timeStride; 
      if(x>6) mpf3 += x/timeStride; 
    }
  }

  sync_history(history_mpf1, 0.);
  sync_history(history_mpf2, 0.);
  sync_history(history_mpf3, 0.);
  sync_history(history_pxr1, 0.);
  sync_history(history_pxr2, 0.);
  sync_history(history_pxr3, 0.);

  history_mpf1->eraseStart(Interval(timeStride));
  history_mpf2->eraseStart(Interval(timeStride));
  history_mpf3->eraseStart(Interval(timeStride));
  history_pxr1->eraseStart(Interval(timeStride));
  history_pxr2->eraseStart(Interval(timeStride));
  history_pxr3->eraseStart(Interval(timeStride));

  history_pxr1->Append(history_pxr1->getEndTime(),
		       Interval(timeStride), &mpf1, 1);
  history_pxr2->Append(history_pxr2->getEndTime(),
		       Interval(timeStride), &mpf2, 1);
  history_pxr3->Append(history_pxr3->getEndTime(),
		       Interval(timeStride), &mpf3, 1);
  mTrend.trendData(trendName_pxr1, currentTime, mpf1);
  mTrend.trendData(trendName_pxr2, currentTime, mpf2);
  mTrend.trendData(trendName_pxr3, currentTime, mpf3);

    if(mpf>0.) mpf1 /= mpf; 
    if(mpf>0.) mpf2 /= mpf; 
    if(mpf>0.) mpf3 /= mpf; 

  history_mpf1->Append(history_mpf1->getEndTime(),
		       Interval(timeStride), &mpf1, 1);
  history_mpf2->Append(history_mpf2->getEndTime(),
		       Interval(timeStride), &mpf2, 1);
  history_mpf3->Append(history_mpf3->getEndTime(),
		       Interval(timeStride), &mpf3, 1);
  mTrend.trendData(trendName_mpf1, currentTime, mpf1);
  mTrend.trendData(trendName_mpf2, currentTime, mpf2);
  mTrend.trendData(trendName_mpf3, currentTime, mpf3);
}

void BurstMon::update_trend_freq(bool flag, float value)
{
  float dummy = 0.;
  float last_value;

  last_value = (flag)? value: dummy;

  sync_history(history_freq, dummy);
  history_freq->eraseStart(Interval(timeStride));
  history_freq->Append(history_freq->getEndTime(),
		       Interval(timeStride), &last_value, 1);

  mTrend.trendData(trendName_freq, currentTime, last_value);
}

void BurstMon::update_trend_time(bool flag, float value)
{
  float dummy = 0.;
  float last_value;

  last_value = (flag)? value: dummy;

  sync_history(history_time, dummy);
  history_time->eraseStart(Interval(timeStride));
  history_time->Append(history_time->getEndTime(),
		       Interval(timeStride), &last_value, 1);

  mTrend.trendData(trendName_time, currentTime, last_value);
}

void BurstMon::update_trend_SNR(bool flag, float value)
{
  float dummy = 0.;
  float last_value;

  last_value = (flag)? value: dummy;

  sync_history(history_SNR, dummy);
  history_SNR->eraseStart(Interval(timeStride));
  history_SNR->Append(history_SNR->getEndTime(),
		       Interval(timeStride), &last_value, 1);

  mTrend.trendData(trendName_SNR, currentTime, last_value);
}

void BurstMon::update_trend_rate(bool flag, float value)
{
  float dummy = 0.;
  float last_rate;

  last_rate = (flag)? value: dummy;

  sync_history(history_clusterRate, dummy);
  history_clusterRate->eraseStart(Interval(timeStride));
  history_clusterRate->Append(history_clusterRate->getEndTime(),
                              Interval(timeStride), &last_rate, 1);

  mTrend.trendData(trendName_clusterRate, currentTime, last_rate);
}


void BurstMon::update_trend_nvar(wavearray<float> &var)
{
  Time historyEndTime;
  TSeries *tmp;
  size_t k = size_t(offsetTime/2.+0.1);  // offset samples
  size_t n = size_t(var.size()/var.rate()+0.5)-2*k;
  size_t m = size_t(var.rate()+0.5);
  wavearray<float> temp(n);
  float dummy = 1.;
  temp.rate(1.);
  size_t i,j;

  if(n != size_t(timeStride+0.1))
    cout<<"burstMon warning: wrong variability buffer length: "<<n<<endl;

  temp = 0.;
  for(i=0; i<n; i++){
    for(j=0; j<m; j++) temp.data[i] += var.data[(i+k)*m+j];
    temp.data[i] /= m;
  }


// update history for noise variability
  sync_history(history_nvar, dummy);
  historyEndTime = history_nvar->getEndTime();

  tmp = new TSeries(historyEndTime, history_nvar->getTStep(), temp.size(), temp.data);

  if (history_nvar->Append(*tmp)==0)
     history_nvar->eraseStart(tmp->getInterval());

  if (tmp) delete tmp;
  
// trend to file
  tmp = new TSeries(historyEndTime, Interval(1./var.rate()), n*m, var.data+k*m);

  sTrend.trendData(trendName_nvar, *tmp);

  if (tmp) delete tmp;
  return;
}


void BurstMon::dumpSummary(char* fname)
{
  wavearray<int> q(10);
  wavearray<double> r(10);

  char cbuf[256];
  FILE *fp;

  if ( (fp = fopen(fname, "w")) == NULL ) {
     cout << " BurstMon::dumpSummary() error: cannot open file " 
          << fname <<". \n";
     return;
  };

  LocalStr(Now(), cbuf, "%M %d, %Y %02H:%02N");

  fprintf(fp, " BurstMon statistics at: %25s\n", cbuf);
  fprintf(fp, " \n");
  fprintf(fp, " channel name            %25s\n", masterChannelName);
  if (strlen(lockCondition))
  fprintf(fp, " lock condition          %25s\n", lockCondition);
  fprintf(fp, " start of data processing at GPS time %12lu\n",
         (long unsigned)startTime);
  fprintf(fp, " now processing data at GPS time %17lu\n", currentTime.getS());
  fprintf(fp, " \n");
  fprintf(fp, " total time, sec        %26d\n",int(totalTime));
  fprintf(fp, " live  time, sec        %26d\n",int(liveTime));
  fprintf(fp, " dead  time, sec        %26d\n",int(deadTime));
  fprintf(fp, " \n");
  fprintf(fp, " Input parameters:\n\n");
  fprintf(fp, "    time stride, sec           = %6.0f\n", timeStride);
  fprintf(fp, "    resolution, Hz             = %6.0f\n", resolution);
  fprintf(fp, "    injection window, sec      = %6.0f\n", injectionWindow);
  fprintf(fp, "    whitening resolution, Hz   = %6.0f\n", wResolution);
  fprintf(fp, "    sampling rate, Hz          = %6.0f\n", samplingRate);
  fprintf(fp, "    wavemon buffer, s          = %6.0f\n", offsetTime);
  fprintf(fp, "    dmtviewer history length   = %6d\n", historyLength);
  fprintf(fp, "    significance cut           = %6.1f\n", Threshold);
  fprintf(fp, "    false alarm rate, Hz       = %6.1f\n", targetFAR);
  fprintf(fp, "    min cluster size           = %6d\n", minClusterSize);
  fprintf(fp, "    black pixel probability    = %6.2f\n", pFraction);
  if (timeGate > 0.)
  fprintf(fp, "    coincidence gate, sec      = %6.4f\n", timeGate);
  fprintf(fp, "    wavelet filter length      = %6d\n", waveOrder);
  fprintf(fp, "    number of injections       = %6d\n", Injections);

  if ( strlen(responseFile) != 0 )
  fprintf(fp, "\n    response function from file : %s\n", responseFile);

  fprintf(fp, "\n");

  for (size_t k=0; k<h50.size(); k++)
  {
    fprintf(fp, "    waveform #%zd, file name: %s\n",
            k, waveformFileName.at(k));
  }
  fprintf(fp, "\n");
  
  fprintf(fp, " current numbers:\n");
  fprintf(fp, " cluster rate     = %f Hz\n", clusterRate);

  fprintf(fp, "\n");


  for (size_t k=0; k<h50.size(); k++)
  {
    fprintf(fp, " waveform #%zd, noise = %8.3e, h50%% = %8.3e, err_h50 = %4.1lf %%\n",
             k, amp.at(k), h50.at(k), err_h50.at(k));
  }

  fclose(fp); 
}

char *BurstMon::getToken(pchar &s, bool ignore)
// ignore = true - ignore punctuation except space
{
   char *w = 0;
   char *eos = s+strlen(s); 

   s--;
   while(++s < eos){

      if (ispunct(*s) && !ignore && *s != '-' && *s != '.' ){
	*s = '\0'; break;
      }

      if (!isspace(*s) && !w) w = s;  // first character
      if ( isspace(*s) &&  w){ *(s++) = '\0'; break; }

   }

   return (w) ? w : eos;
}

// read Monitor parameters from configuration file
bool BurstMon::readConfig(char *)
{
  FILE* fp;

  if ( (fp=fopen(configFile,"r")) == NULL ){ 
    cout << endl;
    cout << " -> BurstMonr error: cannot open configuration file \""
         << configFile <<"\""<<endl<<endl;
    dumpHelp();
  } 

  char s[256];
  char *w = NULL;
  char *pname = NULL;
    
  while(fgets(s,100,fp) != NULL){
    pString = s;
    if (!strlen(s)) break;
       
    while(strlen(w=getToken(pString))>1){

	if (getParameter(w,"-stride", timeStride))       continue;
	if (getParameter(w,"-window", injectionWindow))  continue;
	if (getParameter(w,"-bpp", pFraction))           continue;
	if (getParameter(w,"-f", resolution))            continue;
        if (getParameter(w,"-white", wResolution))       continue;
	if (getParameter(w,"-rate", samplingRate))       continue;
	if (getParameter(w,"-history", historyLength))   continue;
	if (getParameter(w,"-offset", offsetTime))       continue;
	if (getParameter(w,"-v", verbose))               continue;
	if (getParameter(w,"-wavelet", waveOrder))       continue;
	if (getParameter(w,"-size", minClusterSize))     continue;
	if (getParameter(w,"-gate", timeGate))           continue;
        if (getParameter(w,"-events", Injections))       continue;
        if (getParameter(w,"-scut", Threshold))	         continue;
        if (getParameter(w,"-far", targetFAR))	         continue;
        if (getParameter(w,"-gps", startTime))	         continue;
	
	if (strcmp(w,"-c") == 0){
	  sprintf(masterChannelName,"%s",w=getToken(pString, true));
	  cout <<" master channel            = "<<masterChannelName<<endl;
	}
	
	else if ( strcmp(w,"-suffix") == 0 )
	  sprintf(suffix,"%s",w=getToken(pString, true));

        else if (strcmp(w,"-inject") == 0){
	  pname = new char[256];
	  sprintf(pname,"%s",w=getToken(pString, true));
	  waveformFileName.push_back(pname);
	  cout <<" waveform file             = "<<pname<<endl;
	}

        else if (strcmp(w,"-resp") == 0){
	  sprintf(responseFile,"%s",w=getToken(pString, true));
	  cout <<" response file             = "<<responseFile<<endl;
	  ezCal = true;
	}

	else if (strcmp(w,"-d") == 0)
	  sprintf(summaryFile,"%s",w=getToken(pString, true));

	else if (strcmp(w,"-lock") == 0)
	  sprintf(lockCondition,"%s",w=getToken(pString, true));
	
      }
  }
  fclose(fp);
  return true;
}

void BurstMon::readLockLoss(bool offLine)
{
  FILE* fp;
  string s;

// read OperStateCondition file
   if (strlen(lockCondition))
   {
     mOSC.setStride(int(timeStride+0.1));
     if ( (fp=fopen("/export/home/ops/pars/LockLoss.conf","r"))==NULL 
          || offLine ) 
     {
       if (!offLine) 
         cout << "Error: can't open /export/home/ops/pars/LockLoss.conf "
              << endl;

       fclose(fp);

       if ( (fp=fopen("LockLoss.conf","r"))==NULL )
       {
	 cout << "Error:         can't open LockLoss.conf file"<< endl;
	 cout << "copy LockLoss.conf file into local directory"<< endl;
	 exit(0);
       }
       else
       {
	 fclose(fp);
	 s += string("LockLoss.conf");
       }
     }
     
     else{
       fclose(fp);
       s += string("/export/home/ops/pars/LockLoss.conf");
     }
   }
     
   if (!s.empty()) 
   {
     cout <<" read lock loss file: "<<s.c_str()<<endl;
     mOSC.readConfig(s.c_str());
   }

}


std::string BurstMon::getCalFileName(std::string ifo)
{
  FILE* fp;
  std::string s = "ReferenceCalibration_"+ifo+".xml";  
  std::string S = "/export/home/ops/pars/"+s;

  if ((fp=fopen(S.c_str(),"r"))==NULL) 
    {
       if (!offLine) cout << "Error: can't open "<<S.c_str()<< endl;

       fclose(fp);

       if ( (fp=fopen(s.c_str(),"r"))==NULL )
       {
	 cout << "copy "<<s.c_str()<<" file into local directory"<< endl;
	 exit(0);
       }
 
       fclose(fp);
       return s;
    }
     
  else{
    fclose(fp);
    return S;
  }
}


void BurstMon::dumpHelp(void)
{
   cout << "                   Wave Burst Monitor                    \n\n"

        << "reference: http://www.phys.ufl.edu/LIGO/burstmon/        \n"
        << "    S.Klimenko, A. Sazonov University of Florida         \n\n"

        << "Purpose: Using software injections of waveform           \n"
        << "         provided by user BurstMon calculates amplitude  \n"
        << "         of injection at which 50\% of detection         \n"
        << "         efficiency is achieved.                         \n\n"

        << "Method: The BurstMon performs Monte-Carlo simulation of  \n"
        << "        burst detection using time-frequency analysis    \n"
        << "        in wavelet domain.                               \n\n"

        << "Output: The BurstMon has the following output:           \n"
        << "        - cluster rate,                                  \n"
        << "        - detection sensitivity, h50                     \n"
        << "        - noise variability                              \n"
        << "        which are served to the dmtviewer and saved in   \n"
        << "        trend files (minute and second trends).          \n\n"

        << "command line options :                                   \n\n"

        << "  -c <channel name>: channel name is mandatory           \n"
        << "     Only one channel name is permitted.                 \n\n"

        << "  -i <input config file>: The BurstMon configuration file\n"
        << "     All options except -i, -infile, -inlist can be used \n"
        << "     to build entry (one per line) in the file.          \n"
        << "     For example,                                        \n"
        << "        -c L1:LSC-AS_Q        // channel name            \n"
        << "        -lock L1:X_arm_locked // when x arm locked       \n"
        << "        -T 60                 // stride 32 sec           \n"
        << "        -r 64.                // resolution 64 Hz        \n"
        << "        -p 1.                 // pixel fraction 1%       \n"
        << "        -j sgauss235q9.txt    // input waveform from file\n\n"

        << "  -lock <lock condition>:  Process data only if all      \n"
        << "     specified lock conditions are satisfied.            \n"
        << "     For example:                                        \n"
        << "           H2:Both_arms_locked                           \n"
        << "           H1:Mode_cleaner_locked                        \n"
        << "           L1:X_arm_locked                               \n\n"

        << "  -stride <time stride (sec)> [60]: Length of time       \n"
        << "     series requested by BurstMon                        \n\n"

        << "  -f <frequency resolution (Hz)>: defines frequency scale\n"
        << "     resolution, for cluster analysis [64]               \n\n"

        << "  -white <whitening frequency resolution (Hz)> [8]       \n\n"

        << "  -rate <processing sampling rate (Hz)> [8192.]:         \n\n"

        << "  -bpp <pixel fraction>, % [1]: Fraction of pixels,      \n"
        << "     which pass the percentile selection                 \n\n"

        << "  -size <minimum cluster size>, [1]: only clusters with  \n"
        << "     bigger size will be taken into analysis             \n\n"

        << "  -v <printout verbosity level> [0]                      \n\n"

        << "  -wavelet <length of wavelet filter> [32]               \n\n"

        << "  -gate <coincidence time gate, sec> [0.02]              \n\n"

        << "  -d <name>: file name to dump summary to                \n\n"

        << "  -infile <frame file>: If  -infile is specified, the    \n"
        << "   BurstMon will process data from the input frame file  \n\n"

        << "  -inlist <file name>: If  -inlist is specified, the     \n"
        << "   BurstMon will process data from the frame files       \n"
        << "   listed in file <file name>                            \n\n"

        << "  -history <length of the DMTVIEWER buffer> [720]        \n\n"

        << "  -offset <length of the BurstMon buffer> [2 sec]        \n\n"

        << "  -suffix <suffix added to BurstMon name>                \n\n"

        << "  -inject <waveform file name>, mandatory		     \n\n"

        << "  -events <int>, number of njections/stride/waveform     \n\n"

        << "  -cal <frequency>, calibration line frequency, Hz [0]   \n\n"

        << "  -resp <response file>, get response function from file \n\n"
  ;
  exit(0);
  return;
}

bool BurstMon::check_directory_access(char *name, int permission)
{
    if (name)
    {
      if (strlen(name)) {
        if (access(name, permission) != 0)
        {
           cout << "BurstMon error, checking output directory: "<< name
                << "\n"<< strerror(errno) << endl;
           return false;
        }
	else return true;
      }
    }
    return false;
}
