#include "TrigEnv.hh"
#include "TCanvas.h"
#include <string>
#include <list>

class Pipe;
class TSeries;
namespace trig {
    class TrigBase;
}

class TrigID {
public:
    TrigID(const std::string& ID, const std::string& subID, const Pipe* f=0);
    TrigID(const TrigID& t);
    ~TrigID(void);
    const char* getDisplay(void) const;
    Interval getSettle(void) const;
    bool isTrig(const trig::TrigBase& t);
    TSeries Filter(const TSeries& t);
    void setDisplay(const std::string& d);
    void setPipe(const Pipe* p);
    void setSettle(double dt);

private:
    std::string mTrigID;
    std::string mSubID;
    std::string mDisplay;
    Pipe*       mFilter;
    Interval    mSettle;
};

inline const char*
TrigID::getDisplay(void) const {
    return mDisplay.c_str();
}

inline Interval
TrigID::getSettle(void) const {
    return mSettle;
}

/**  @name TrigDsply
  *  TrigDsply displays the raw data corresponding to each selected trigger
  *  in an xml list. The displayed data may be pre-filtered with any of the 
  *  filter types in the DMT signal processing library. The channel to be 
  *  displayed defaults to the trigger sub-ID, but may be specified 
  *  separately for each trigger type. The time interval displayed for
  *  each trigger starts at the specified trigger time and runs for the 
  *  trigger duration. In addition the signal is shown for a specified 
  *  "pad" time before and after the trigger epoch. The pad time used 
  *  is the greates of a minimum pad time or a constant framction of the 
  *  trigger duration. the minimum time and padding framction are specified 
  *  in the TriDsply command line.
  *
  *  \begin{center}{\bf {\Large Running TrigDsply}}\end{center}
  *
  *  The TrigDsply command line is as follows:
  *  \begin{verbatim}
     TrigDsply -ttable <trigs> [-conf <cfile>] [-infile <frames>] \
                               [-mpad <tmin>] [-fpad <tfrac>] [+toc | -toc] \
			       [-outfile <ps-file>] [-sleep <secs>]
     \end{verbatim}
  *  The arguments have the following meanings:
  *
  *  \begin{center} \begin{tabular}{ll}
  *  <cfile> & Configuration file name. \\
  *  <trigs> & Ligo-lightweight (xml) format table of trigger to process. \\
  *  <frames> & List of files to be searched for the raw data frames. \\
  *  <tmin>   & is the minimum paddding time to be used [0.25s]. \\
  *  <tfrac>  & default pad time as a fraction of the trigger duration [0.2]. \\
  *  <ps-file> & Postscript output file. \\
  *  <secs> & Delay between successive trigger displays. \\
  *  \end{tabular} \end{center}
  *
  *  The following example displays selected raw data from a list of 
  *  triggers in "trigs.xml" as selected by the configuration file in 
  *  "trig_cf.cf". The raw data are serch for in the /qfs directory 
  *  tree (i.e. the LDAS cache disks if running on decatur).
  *
  *  \begin{verbatim}
      TrigDsply -ttable trigs.xml -conf trig_cf.cf -infile '/qfs*\/frames/Data*\/\*.gwf'
     \end{verbatim}
  *
  *  \begin{center}{\bf {\Large The TrigDsply Configuration File}}\end{center}
  *
  *  Configuration files are optional. If no configuration file is specified,
  *  the raw data corresponding to all triggers in the trigger tables are 
  *  displayed without any pre-filtering.
  *
  *  The configuration file consists of filter specifiers and trigger 
  *  selection and deselection statements. Each configuration statement
  *  is a single line. Comments start with a pound sign ('\#'). The 
  *  configuration statements start with one of the following keywords.
  *  \begin{center}
  *  \begin{tabular}{ll}
  *  DeSelect & Disable display of the specified trigger type. \\
  *  Filter & Specify a filter. \\
  *  Select & Enable display of a specified trigger type. \\
  *  \end{tabular}
  *  \end{center}
  *
  *  {\bf Deselect}
  *
  *  The Deselect statement disables plotting of a specified trigger type.
  *  The deselection command has the following format:
  *  \begin{verbatim}
     Deselect <trigger-id> <sub-id>
     \end{verbatim}
  *  The Trigger-ID and Sub-ID fields may be specified as '*' to specify
  *  all values of that field.
  *
  *  {\bf Filter}
  *
  *  The Filter statement specifies a filter that may be used to preprocess
  *  the raw data to be displayed. Filters are associated with individual
  *  triggers with the Select command. Each trigger type has a separate 
  *  instance of the named filter with independent history data. The Filter 
  *  command has the following format:
  *  \begin{verbatim}
     Filter <name> <type> <parameters>
     \end{verbatim}
  *
  *  The implemented filter types are listed in the following table.
  *  Each entry is followed by a list of parameters need to define the filter.
  *
  *  \begin{center}
  *  \begin{tabular}{ll} \hline
  *  BaseLine   & Average baseline suppression \\
  *             & P1: [optional] time constant [1.0] \\
  *  Design     & Arbitrary filter design \\
  *             & P1: Sample rate  \\
  *             & P2: FilterDesign class filter specifier in quotes \\
  *  Difference & Difference between adjacent samples \\
  *             & P1: Sample rate \\
  *  IIRFilter  & Infinite impulse filter \\
  *             & P1: Sample rate  \\
  *             & P2: List of poles (2*pi*f) in quotes \\
  *             & P3: List of zeros (2*pi*f) in quotes \\
  *  MultiPipe  & Chain of filters \\
  *             & P1-PN: Names of filters to be used to form a pipeline \\
  *  \end{tabular} 
  *  \end{center}
  *
  *  {\bf Select}
  *
  *  The Select statement enables plotting of a specified trigger type.
  *  The Select command has the following format:
  *
  *  \begin{verbatim}
     Select { <trig-ID> | * } { <sub-id> | * } [-display <channel>] \
            [-filter <filter-name>] [-settle <settling-time>]
     \end{verbatim}
  *  The raw data associated with the specified trigger type(s) are 
  *  displayed. The channel specified by the trigger sub-ID is displayed 
  *  unless the "-display" argument is specified. The raw data are filtered
  *  before being displayed if the optional "-filter" argument is specified.
  *
  *  {\bf Example configuration file} \\
  *  \begin{verbatim}
     #
     #    Configuration file to plot H2:LSC-AS_Q glitches prefiltered with
     #    a 4th order 100Hz high-pass IIR filter.
     #
     Filter HP16kHz100 IIRFilter 16384 "-474.906839355070  271.905402086509 \
                                        -474.906839355070 -271.905402086509 \
					-126.983988592287  426.388253881100 \
					-126.983988592287 -426.388253881100" \
				       "-0.00000000001095  31.82015846649922 \
				        -0.00000000001095 -31.82015846649922 \
					-0.00000000000763  76.58315106360777 \
					-0.00000000000763 -76.58315106360777"
     #
     Select Glitch H2:LSC-AS_Q  -filter HP16kHz100
     #
     \end{verbatim}
  *
  *  @memo Display raw data corresponding to a list of triggers.
  *  @author J. Zweizig
  *  @version 1.1; Last modified April 4, 2002
  */
//@{
//@}

class TrigDsply : public TrigEnv {
public:
    TrigDsply(int argc, const char* argv[]);
    ~TrigDsply(void);
    bool ProcessTrigger(const trig::TrigBase& t);
    void ProcessData(void);
    void Plot(const TSeries& t);
    bool ReadConfig(const std::string& file);
private:
    TCanvas* mCanvas;
    double minPad;
    double mFracPad;
    std::string mChannel;
    long mSleep;

    //-----------------------------------  Trigger lists
    typedef std::list<TrigID> TrigList;
    typedef TrigList::iterator trig_iter;
    TrigList  mSelect;
    TrigList  mDeSelect;
    trig_iter mTrig;

    typedef std::map<std::string, Pipe*> FiltrList;
    typedef FiltrList::iterator  FiltrIterator;
    FiltrList mFiltr;
    std::string mOutfile;
};



