/* -*- mode: c++; c-basic-offset: 3; -*- */
//
//    Rayleigh Monitor, version 6.2
//      author: Patrick J. Sutton
//        date: Oct 27, 2003 
//  
//    For documentation, see source code file RayleighMonitor.cc 
//    Based on DatTemplate routine by J. Zweizig.
//
////////////////////////////////////////////////////////////////////////////

#ifndef RayleighMonitor_HH
#define RayleighMonitor_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "fSeries/PSD.hh"
#include <string>

//=====================================  Rayleigh class forward references
class MainDialog;
class RayleighConfig;

//=====================================  Root class forward references
class TCanvas;
class THTFD;
class TLTimeFrequency;
class TPaveText;

//=====================================  DMT class forward references
class Hanning;
class TSeries;
namespace containers {
  class fSeries;
  class PSD;
}

const char* const MonitorName = "RayleighMonitor";    


/** The RayleighMonitor class is the basis for a DMT monitor of 
  * the same name which produces a frequency-dependent measure 
  * of the non-Gaussianity of data from an arbitrary collection 
  * of channels.  
  * @memo RayleighMonitor class for DMT monitor of Gaussianity. 
  * @author Patrick J. Sutton
  * @version 6.2; Last modified Oct 27, 2003.
  */  
class RayleighMonitor : public DatEnv, MonServer {

  public:

    /** Construct a RayleighMonitor object.
      * @memo Constructor.
      */
    RayleighMonitor(int argc, const char *argv[]); 

    /** Destroy a RayleighMonitor object.
      * @memo Destructor.
      */
    ~RayleighMonitor();   

    /** Process data statement.  The monitor reads a specified amount 
      * of timeseries data for each channel and computes the average 
      * periodogram (essentially the power spectrum) and the ratio of 
      * its standard deviation to its mean (the ``Rayleigh statistic'').
      * @memo Process data statement.  
      */ 
    void ProcessData(void);

    /** Attention interrupt method.  Does one of two things: (1) 
      * Causes the current time-frequency plot to be printed as a 
      * GIF file named RM.gif. 
      * This is activated manually by the command 
      *      {\bf kill -USR1 <pid>}  
      * where <pid> is the process ID for the Rayleigh monitor. 
      * (2) Automatically serves data to the dmtviewer.
      * Currently set to option (2).
      * @memo Attention interrupt method.  
      */
    virtual void Attention(void) {
         MonServer::Attention();  
         //Canvas->Print("RM.gif"); 
    }

    /** Output monitor status to .html page named RayleighMonitorSummary.html 
      * includes output from recent strides, time-frequency plots, 
      * error messages.
      * @memo Output monitor status to .html page. 
      */
    virtual void Summary(const Time& t);  

    /** Construct simulated data for testing the monitor.
      * \note The caller must delete the returned time series to avoid leaking
      * memory. 
      * \brief Construct simulated data for testing the monitor. 
      * \param n Number of samples
      * \param Sample step time.
      * \return pointer to filled time series.
      */
   virtual TSeries* SimulateData(int n, double tstep);  

private:
    char input_param_file_name[ 100 ];   

    /** Construct time-frequency plots of the averaged periodogram
      * and Rayleigh statistic for each channel.   
      * Uses first stride of data only; plots updated with UpdatePlot() method. 
      * @memo Construct time-frequency plot of PSD data. 
      */
    void CreatePlot(void); 
    void CreateFastPlot(void); 

    /** Update time-frequency plots of the averaged periodogram
      * and Rayleigh statistic for each channel.   
      * @memo Udate time-frequency plot of PSD data. 
      */
    void UpdatePlot(void);  
    void UpdateFastPlot(void);  
     
    /** Add current PSD for first channel to reference spectrum.
      * Used to whiten PSD time-frequency plot.
      * @memo Add current PSD to reference spectrum.
      */
    void AddToReferenceSpectra(void);
     
    /** Record current PSD for first channel as reference spectrum.
      * Used to whiten PSD time-frequency plot.
      * @memo Record current PSD as reference spectrum.
      */
    void SetReferenceSpectra(void);

    bool batch;              // Batch job; no GUI front-end window.
    bool detrend;            // Flag indicating data to be detrended. 
    bool errors;             // Flag indicating error messages have been written
    bool fast_update;        // Use TFPainter code for fast plotting.
    bool fatal_error;        // Fatal error flag; exit when true.
    bool fixedrange;         // Fix the ranges of the rayleigh & psd plots.
    bool greyscale;          // Use greyscale for plots if true. 
    bool initial;            // Flag signaling initialization of data storage
    bool local;              // Divert output files to $DMTHTMLOUT.
    bool logf;               // Flag for logscale in frequency. 
    bool missing_data;       // Flag for gap in data. 
    bool record;             // Save .gif images of all data.  
    bool seek;               // Seek specified GPS time befroe starting.
    bool simulated_data;     // Run monitor on simulated Gaussian noise. 
    bool slow_update;        // Update t-f plots only every 10th timestep.
    bool stderr_diverted;    // Flag indicating stderr remapped to file 
    bool stdout_diverted;    // Flag indicating stdout remapped to file 
    bool whiten;             // Flag signaling whitening of PSD
    bool window;             // Flag indicating data to be windowed. 
    char* current_channel_name;  // Output channel names have _psd or _ray 
                                 // appended to specify type of output.
    double* f_lower;         // minimum frequency examined 
    double* f_upper;         // maximum frequency examined
                             // NOTE: f_lower, f_upper may differ from those 
    Time   prev_endtime;     // End time of last stride (for error checking)    
    Time   seektime;         // Start time to look for.
    double expected_raymean; // Expected value of rayleigh statistic (raymean) 
    double expected_rayvar;  // and its standard deviation (rayvar) for N 
                             // samples of Gaussian noise. 
    containers::PSD PSDdata_ref; // Holds reference averaged power spectrum 
    double whiten_scale;     // Set PSD range to [1/X,X] when whitening. 
    containers::PSD* psd;    // Holds averaged power spectrum of the data
    containers::PSD* psd_ref;  // Holds reference averaged power spectrum 
    containers::PSD* rayleigh; // Holds rayleigh statistic of processed data
    Hanning* win;            // Default window type.
    int canvas_height;       // Height of single plot in pixels.
    int canvas_width;        // Width of single plot in pixels.
    size_t MaxNStep;         // Largest *NStep value (determines size of 
                             //   storage arrays). 
    int NStride;             // Number of strides already processed
    size_t* NStep;           // (Number of frequency points - 1) or (number  
                             // of time-series points/2) for each channel
    int Nwhiten;             // Number of strides over which to average PSDs 
			     // for whitening.
    MainDialog* mainWindow;  // GUI window.
    RayleighConfig* parameters;  // Class for fetching parameters from input 
                                 // file, such as:
                                 // int N (Number of data segments to average)
                                 // int MaxStride (Max # of strides to process)
                                 // double Stride (Length of data segments-sec)
    std::string error_file;  // Strings containing names (including path) of
    std::string gif_file;    // summary, error, log html files and gif image 
    std::string log_file;    // files.
    std::string record_file; // 
    std::string summary_file;// 
    TCanvas* Canvas;         // ROOT canvas for time-frequency plot.
    TCanvas* Canvas2;        // ROOT canvas for fast time-frequency plot.
    TCanvas* Canvas3;        // ROOT canvas for fast time-frequency plot.
    THTFD** PSDhist;         // ROOT histogram to hold plot data.
    THTFD** RAYhist;         // ROOT histogram to hold plot data.
    time_t start_time;       // Store system times for time trials of monitor.
    time_t end_time;
    THTFD* PSD_TFHist;       // Histogram for fast painting.
    THTFD* RAY_TFHist;       // Histogram for fast painting.
    TPaveText* legend;       // Plot legend object (holds N, Stride, etc.)
    TSeries* noise;          // Holds simulated time-series data
    TSeries** ts;            // Holds time-series data to be analysed
};


#endif     //  RayleighMonitor_HH
