/* -*- mode: c++; c-basic-offset: 4; -*- */
//   Class PulsarMon: Pulsar Sensitivity Monitor
///
//          based on:
//   Class SenseMonitor: Base class for Binary Inspiral Sensitivity Monitor 
//   version: 5.1 (Nov15, 2008)
//   authors:   Giovanni Santostasi  (gsantostasi@mcneese.edu)
//              Kevin C. Schlaufman (kcs149@psu.edu)
//            Patrick J. Sutton (psutton@ligo.caltech.edu)
//           
//
///////////////////////////////////////////////////////////////////////////


#ifndef PulsarMon_HH
#define PulsarMon_HH



#ifndef __CINT__

//----- Include standard library stuff.
#include <string>
#include <fstream>
#include <iostream>
#include <stdio.h> 
#include <signal.h>

//----- Include DMT-specific headers.
#include "Blackman.hh"
#include "DatEnv.hh"
#include "FlatTop.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "Hamming.hh"
#include "Hanning.hh"
#include "MonServer.hh" 
//#include "LineFilter.hh"
#include "OperStateCondList.hh"
#include "Time.hh"
#include "Trend.hh"
#include "TSeries.hh"
//#include "Window.hh"

#endif  //------ !def(__CINT__)

//----- Include monitor-specific headers.

#include "FDEasyCalibrate.hh"
#include "PSD.hh"
#include "RangeSummary.hh"



/** Helper struct for PulsarMon class. 
  * Holds various parameters for the current 
  * invocation of SenseMonitor.
  * @memo Holds parameters for the current invocation of SenseMonitor.
  */
struct Parameter
{        

    /// If true then generate new alpha, beta calibration parameters on-the-fly.
    bool genCalParam;


      /** Number of intervals to divide time-series data into when calculating 
      * the noise PSD.  Must divide the length 'T' of the time 
      * series for the PSD class to be invoked. 
      */
    int num_ints;
    
    int CalPSD_Dump_Period;

    /** The ratio of the current value of the 
      * calibration parameter beta to its value
      * when the open-loop gain and sensing function calibrations 
      * were measured (must be supplied by hand;
      * not supposed to change often).
      */
    double beta;
    
    /// Length (sec) of data to use for each range estimate.
    double T; 

    /// Highest frequency used in range estimate.
    double h_freq;  

    /// Lowest frequency used in range estimate.
    double l_freq;  

    /// Name of uber xml file holding reference calibration data.
    std::string cal_file_name; 


    /// Name of uber xml file holding reference calibration data.
    std::string sf_file_name; 

    /// Name of uber xml file holding reference calibration data.
    std::string xml_file_name; 

    /// Name of configuration file holding monitor parameters.
    std::string config_file_name;  

    ///  Interferometer (one of H1, H2, or L1).
    std::string IFO;
    ///  Prefix name of environmental channels (L0 or H0)
    std::string IFOPEM;

    /// Channel name.  Holds <IFO>:LSC-AS_Q, where <IFO> is one of H1, H2, or L1.
    std::string mChannel;  
 
    ///PEM channels
    std::string PEMChannel1;
    std::string PEMChannel2;
    std::string PEMChannel3;
    std::string PEMChannel4;
    std::string PEMChannel5;
    std::string PEMChannel6;
    ///  std::string PEMChannel7;

   /// osc condition
   std::string osc_cond;
   /// monitor name
   std::string MonitorName;
 

    /// Name of file holding open-loop-gain data for dynamic calibrations.
    std::string olg_file_name; 


    /** Name of configuration file holding operating state condition (OSC) 
      * definitions for the IFO.
      */
    std::string osc_file_name; 

    std::string CalPSD_file_name;

    /// Name of window type to use in calculating the noise PSD.
    std::string window_name; 
};


/** Helper struct for SenseMonitor class. 
  * Holds all information on where and what 
put files will be written.
  * @memo Holds information on output files to be written.
  */
struct Output_Files
{
    //  If true then the monitor will write error messages to file 
    //  <IFO>_SenseMonitor_Error.html (default true).
    bool error_log; 

    //  If true then dump output to local directory
    //  instead of to directory specified by 
    //  environment variable DMTHTMLOUT. 
    bool local; 

    //  If true, output is echoed to the screen (default false).
    bool screen_output;  

    //  If true, SenseMonitor writes to a trend file (default false).
    bool trend;

    //  If true then summary file <IFO>_SenseMonitor_Log.html
    //  is written (default true).
    bool write_log; 

    //  If true, configuration file will be re-read 
    //  Output stream to file holding calibrated noise spectrum.
    std::ofstream calpsd_file;  

    //  Name of file holding calibrated noise spectrum.
    //  Holds name of calibrated pSD file name.
    std::string calpsd_file_name;     

    /// Name (including path) for DMTViewer history file.
    std::string dmtviewer_file_name; 

    //----- Notes: 
    //      1. By default cout is redirected to the log file,
    //         and cerr to the error file. 
    //      2. dmtviewer_file is used as both input stream 
    //         (in the constructor) and output stream (in the 
    //         destructor) so the stream objects are declared 
    //         locally.
    //      3. Naming conventions:  
    //          file streams typically end with _file. 
    //          file names typically end with _file_name (with/without path). 
    //          file names ending with _file_link do not include a path. 

    /// Directory specified in DMTHTMLOUT, where output files go.
    char *html_dir; 
    
    /// Name (including path) for current log file.
    std::string log_file_name;       

    /// Name (excluding path) for current log file.
    std::string log_file_link;  

    /// Output stream for CumLog.txt file.
    std::ofstream cumlog_file;

    /// Name (excluding path) for cumulative log file.
    std::string cumlog_file_link;  

    /// Name (including path) for cumulative log file.
    std::string cumlog_file_name;  

    /// Name (including path) for error file.
    std::string error_file_name;     

    /// Name (excluding path) for error file
    std::string error_file_link;

    //  Name (including path) for html revolver file.
    std::string revolver_file_name;	

    //  Name (including path) for html summary file.
    std::string summary_file_name;   
};


/** Helper struct for PulsarMon class. 
  * Holds data related to recent range estimates. 
  * @memo Holds data related to recent range estimates. 
  */
struct Range_Data
{
    /// Current value of calibration parameter alpha.
    double alpha;

    /// Current value of calibration parameter beta.
    double beta; 

    /// Current value of calibration line amplitude.
    double ampl;

    /** Current value of effective ellipticity detectability (Pulsar at 500 Hz, 1kpc, 1 
year IT),
      * adjusted by calibration-line amplitude.
      */ 
    double ellipticity; 

    /** Current value of effective ellipticity detectability.
      * If used with static calibration (-calfile
      * option) then this range is NOT adjusted by
      * the calibration-line amplitude.
      * If used with dynamic calibration (-dyncal
      * option) then this ellipticity is already adjusted
      * by the calibration-line amplitude, and is 
      * identical to 'ellipticity'. 
      */
        
    double CrabTime;
    double CrabTimePEM1;
    double CrabTimePEM2;
    double CrabTimePEM3;
    double CrabTimePEM4;
    double CrabTimePEM5;
    double CrabTimePEM6;
    /// double CrabTimePEM7;

    double pulsar1Time ;
    double pulsar2Time ;
    double pulsar3Time ;
    double pulsar4Time ;

    double pulsar1TimePEM1 ;
    double pulsar2TimePEM1 ;
    double pulsar3TimePEM1 ;
    double pulsar4TimePEM1 ;
    double pulsar1TimePEM2 ;
    double pulsar2TimePEM2 ;
    double pulsar3TimePEM2 ;
    double pulsar4TimePEM2 ;
    double pulsar1TimePEM3 ;
    double pulsar2TimePEM3 ;
    double pulsar3TimePEM3 ;
    double pulsar4TimePEM3 ;
    double pulsar1TimePEM4 ;
    double pulsar2TimePEM4 ;
    double pulsar3TimePEM4 ;
    double pulsar4TimePEM4 ;
    double pulsar1TimePEM5 ;
    double pulsar2TimePEM5 ;
    double pulsar3TimePEM5 ;
    double pulsar4TimePEM5 ;
    double pulsar1TimePEM6 ;
    double pulsar2TimePEM6 ;
    double pulsar3TimePEM6 ;
    double pulsar4TimePEM6 ;
    ////double pulsar1TimePEM7 ;
    /// double pulsar2TimePEM7 ;
    /// double pulsar3TimePEM7 ;
    /// double pulsar4TimePEM7 ;

    /** Value of the Integration Time needed to reach Energy Conservation limit for Crab 
Pulsar
    */ 
    
    /// Contains last 5 ellipticity estimates. (CAL or UNCAL?)
    double T5_array[5];

    /// Contains last 15 ellipticity estimates. (CAL or UNCAL?)
    double T15_array[15];   

    /// Average ellipticity over last 5 time steps.
    double T5_ellipticity;

    /// Average ellipticity over last 15 time steps.
    double T15_ellipticity;
};






/** The PulsarMon class is the base class for a DMT monitor of the 
  * same name which produces estimates of the Figure of Merit of known and unknown Pulsars  
  * @memo PulsarMon class for DMT pulsar monitor.
  */ 
class PulsarMon : public DatEnv, MonServer {

  public:

    /** Construct a PulsarMon object.
      * @memo Constructor.
      */ 
     PulsarMon(int argc, const char *argv[]); 

    /** Destroy a PulsarMon object.
      * @memo Destructor.
      */ 
    ~PulsarMon(void); 

    /** Process data statement.  The monitor reads a specified amount 
      * of time-series data for the LSC-AS_Q channel and computes the different Pulsars 
FOMs 
      * @memo Process data statement.  
      */
    void ProcessData(void);

    /** Write a brief html page summarizing the monitor and interferometer 
      * status.  The file is named <IFO>_PulsarMon_Summary.revolver.html, 
      * where <IFO> is one of L1, H1, or H2.  FILE IS WRITTEN TO WHAT DIRECTORY?
      * @memo Write an abbreviated html page summarizing the monitor and interferometer 
status.
      * @param t Start time of data for which summary is written. 
      * @param Run_Par Struct Parameter holding SenseMonitor parameters.
      * @param R_Dat Struct Range_Data holding data on recent range and calibration.
      */ 
    virtual void RevolverSummary(const Time& t, Parameter Run_Par,
				 Range_Data R_Dat);

    /** Write an html page summarizing the monitor and interferometer 
      * status.  The file is named index.html.
      * FILE IS WRITTEN TO WHAT DIRECTORY?
      * @memo Write an html page summarizing the monitor and interferometer status.
      * @param t Start time of data for which summary is written. 
      * @param Run_Par Struct Parameter holding SenseMonitor parameters.
      * @param R_Dat Struct Range_Data holding data on recent range and calibration.
      */ 
    virtual void Summary(const Time& t, Parameter Run_Par, Range_Data R_Dat);

    /** Read run parameters from specified configuration file.
      * See documentation for syntax.
      * @memo Read run parameters from specified configuration file.
      * @param filename Name of configuration file. 
      */ 
    virtual void Configure(std::string& filename);  

    /** Read run parameters from command-line arguments.  
      * See documentation for syntax.
      * @memo Read run parameters from command-line arguments.
      */ 
    virtual void Configure(int argc, const char *argv[]);

    /** Perform various checks on monitor parameters and set  
      * bool fatal_error to TRUE if any of the following hold:  
      * stride is not an integer multiple of number of averages;
      * low frequency is not less than high frequency;
      * calibration file(s) cannot be opened;
      * operating state condition (OSC) configuration file cannot be opened. 
      * Also prints warnings under various conditions.
      * @memo Perform various checks on monitor parameters.
      * @param Run_Par Struct Parameter holding PulsarMon parameters.  
      * @param fatal_error Set to TRUE if parameters in Run_Par are not valid.
      */
    virtual void VerifyParameters(Parameter& Run_Par, bool& fatal_error); 

    /** Dump run parameters to specified output stream. 
      * Dumps all members of specified Parameter except dyncal and 
      * config_file_name, plus MaxStride, 
      * frequency of the calibration line tracked (if any), 
      * frequency and amplitude of the simulated calibration line (if any), 
      * plus open-loop gain and sensing function data if dyncal is TRUE.  
      * Output is human-readable, and each line begins with  
      * the # character (except for calbration data).
      * @memo Dump run parameters to specified output stream.
      * @param file Ouput stream.   
      */ 
    virtual void DumpParameters(std::ostream& file); 
      /** Write column headers for range data to specified output stream.  The precise 
      * header information depends on whether the calibration line is tracked.
      * @memo Write column headers for range data to specified output stream.
      * @param out Output stream to which to dump headers.
      */ 
    virtual void DumpReportHeader(std::ostream& out);
  ///HERE out1?

    /** Write column headers for range data to specified output stream.  The precise 
      * header information depends on whether the calibration line is tracked.
      * @memo Write column headers for range data to specified output stream.
      * @param out Output stream to which to dump headers.
      * @param frequency Frequency of calibration line to be tracked for calibration 
adjustments (use zero if no such line).
      */ 
    virtual void DumpReportHeader(std::ostream& out, double frequency);

    /** Write current range and calibration information to the specified 
      * output stream.
      * @memo Write current range and calibration information to the specified output 
stream.
      * @param out Output stream to which to write information. 
      */
    virtual void ReportResults(std::ostream& out, const Time& t, Parameter& Run_Par);

    /** Reread configuration file and extract new values for `beta' and 
      * nominal calibration-line amplitude (only).
      * @memo Reread configuration file.
      */

    /** Act on SIGUSR1 signal (reread configuration file).
      * @memo Act on SIGUSR1 signal.
      */
   // virtual void HandleUsr1();     

    /** Handle signals to monitor.
      * @memo Handle signals to monitor.
      */
    virtual void Attention(void) {
        //----- This call serves data to the DMTViewer. 
    //--- Replaced SIGPOLL with SIGIO Keith Thorne June, 2011
    // on Linux, Solaris SIGPOLL same as SIGIO, Mac OS X only has SIGIO 
	if (testAttnSignal(SIGIO)) MonServer::Attention();
        //----- This call triggers re-reading of calibration information.
         }

  private:

    //  If true then quit cleanly ASAP.
    bool fatal_error; 

    //  Holds last 12 hrs of calibrated ellipticity values.
    double *calvalues;
   // Holds last 12 hours of calibrated TEC crab values 
    double *crabvalues;
 
    //  Holds last 12 hrs of cal-line-ampl values.
    double *amplvalues;	


    //  Holds last 12 hrs of alpha values.
    double *alphavalues;

    //  Holds last 12 hrs of beta values.
    double *betavalues;

    //  Holds last 12 hrs of pulsar1values.
    double *pulsar1values;

    //  Holds last 12 hrs of pulsar2values.
    double *pulsar2values;

    //  Holds last 12 hrs of pulsar3values.
    double *pulsar3values;

    //  Holds last 12 hrs of pulsar4values.
    double *pulsar4values;

    //  Holds end time of last stride of data read.  Used to check for gaps in data.
    double prev_endtime;

    //  1000-sec average range reported to 2-week summary file.
    float running_average;

    //  Number of strides contributing to running_avg
    float running_count; 

    //  Total number of strides to be processed.
    int MaxStride; 

    //  Number of strides of data missing between last and current stride.
    int missing_strides;

    //  Number of strides already processed.
    int NStride;

    //  Time since last update of 2-week summary.
    int time_since_update;

    //  Number of time steps in DMTViewer history.
    int tserieslength;
    
        //  "Easy" calibration object for handling ASQ->strain(nm) conversion.
    FDEasyCalibrate *mCalibrate;

    //  FSpectrum object that holds the integrand of the range integral,  
    //  (f^(7_3) * PSD)^{-1}
    //  See Finn & Chernoff, PRD, 1993.
    FSpectrum f_7_3; 

    // Holds the PSD of the AS_Q data.  It is filled by PSD::generate().
    FSpectrum PSDed_data;  
    // Holds the PSD of the PEM data.  It is filled by PSD::generate().
    FSpectrum PSDed_PEM1;
    FSpectrum PSDed_PEM2;
    FSpectrum PSDed_PEM3;
    FSpectrum PSDed_PEM4;
    FSpectrum PSDed_PEM5;
    FSpectrum PSDed_PEM6;
    /// FSpectrum PSDed_PEM7;

    // Old Spectrum for exponential averaging
    FSpectrum  SpOld;  
    
    //Holds the Ellipticity upper limit for Pulsar at 1kpc, 1 year Integration Time (IT)
    FSpectrum Ellipticity_1kpc;  

    //Holds the Integrated Spectrum over Run Time in terms of h
    FSpectrum FOM4_PSD;
    //Inverse of new FOM4 Spectrum, used for caculation of FOM4_PSD
    FSpectrum CumS_New;
    // Inverse of old FOM4 Spectrum, used for calculation of FOM4_PSD
    FSpectrum CumS_Old;
   
    // Holds the calibrated amplitude spectrum of the detector noise (PSD^0.5).
    FSpectrum strain_noise_ampl;

   FSpectrum  TimeInt_knownpul;
// Holds integration time for known pulsars   
   FSpectrum  EpsiRun;
// Holds ellipticity for known pulsars  
   FSpectrum  hP_EC;
// Holds strain for known pulsars 
   FSpectrum  hIFO_year;
// Holds strain for IFO 

  

 /// GPS time (sec) at which the monitor is started.
    long int starttime; 
    
    //  Monitors lock state of IFO.
    OperStateCondList *mOsclist;

    //  Struct holds metadata on output files.
    Output_Files mOutput; 

    //  Struct (defined above) that keeps all the run parameters in one place.
    Parameter Run_Par;		

    //  Object calculates the power-spectral density
    //  (stored in an FSpectrum object) of the
    //  TSeries data fed to it.  See PSD.hh for 
    //  further explanation.
    PSD *psd;                   
  //------------------------------------------------------------------------marc line
    // PSD *psd_2;

    //  Struct (defined above) that transports all
    //  of the calculated range data in one piece
    //  from the process data method to the summary
    //  method, where the webpage is created.
    Range_Data R_Dat;

    //  Range summary data for Szabi's status web page
    RangeSummary mRangeSumm;	


    /** Stores monitor arguments as read from command-line 
      * or configuration file; used in writing summary html files.
      */
    std::string command_line;

    /// Directory where monitor was started.
    std::string pwd_dir;  

    //  Trend object to which trend information is sent.
    Trend mTrend;

    //  History of alpha parameter sent to DMTViewer.
    TSeries *history_alpha; 

    //  History of beta parameter sent to DMTViewer.
    TSeries *history_beta; 

    //  History of calibration-line amplitude sent to DMTViewer.
    TSeries *history_ampl;  


    //  History of calibration-adjusted ellipticity estimates sent to DMTViewer.
    TSeries *history_ellipticity; 

    
     //  History of calibration-adjusted T_EC estimates sent to DMTViewer.
    TSeries *history_crab;

    //   History of calibration-adjusted T_EC estimates sent to DMTViewer.
    TSeries *history_pulsar1;
    TSeries *history_pulsar2;
    TSeries *history_pulsar3;
    TSeries *history_pulsar4;


    //  Pointer to time series asq data.
    TSeries *ts_asq;                
    // Pointer to time series PEM channel
    TSeries *ts_PEM1;
    TSeries *ts_PEM2;
    TSeries *ts_PEM3;
    TSeries *ts_PEM4;
    TSeries *ts_PEM5;
    TSeries *ts_PEM6;
    /// TSeries *ts_PEM7;

    //  Pointer to Hamming window.
    Hamming *ham;               

    //  Pointer to Hanning window.
    Hanning *han;               

    //  Pointer to FlatTop window.
    FlatTop *flat;              

    //  Pointer to Blackman window.
    Blackman *black;            

    int N_unlock;
    bool UpLastTime;
};

#endif     //----------  PulsarMon_nhat_HH
