/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    PSL Monitor
//
#include "PSLmon.hh"
#include "ParseLine.hh"

#ifdef ROOT_HISTOS
#include "TFile.h"
#include "TH1.h"
#endif

#ifndef __CINT__

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Id: PSLmon.cc 7155 2014-08-23 02:55:50Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "PreStablized LASER monitor"
#include "ProcIdent.hh"

#include <cstdlib>
#include "TSeries.hh"
#include "FSeries.hh"
#include "DaccAPI.hh"
#include "TrigRslt.hh"
#include <iostream>
#include <sstream>
#include <iomanip>
#include <time.h>

//-------------------------------------  Filter implemented
#include "BaseLine.hh"
#include "Blackman.hh"
#include "Difference.hh"
#include "FilterDesign.hh"
#include "Hamming.hh"
#include "Hanning.hh"
#include "MultiPipe.hh"
#include "FIRFilter.hh"
#include "IIRFilter.hh"

//--------------------------------------  Generate the main routine.
EXECDAT(PSLmon)
#endif               // !def(__CINT__)

using namespace std;

//--------------------------------------  Skeleton object constructor.
PSLmon::PSLmon(int argc, const char *argv[]) 
  : DatEnv(argc, argv), TrigClient(trig::TrigWriter::kNone), mStride(1.0), 
    mOSC(getDacc()), mAlarm("PSLmon"), mSegAcct(*this),
    mStartData(0), mStartProcess(0), mProcessedTime(0.0)
{

    //----------------------------------  Look for arguments
    const char* cfile = "PSLmon.conf";
    bool windex=false, exit=false;
    const char* ifile = 0;
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-conf") {
	    cfile = argv[++i];
	} else if (argi == "-ndsindex") {
	    if (i+1<argc && *(argv[i+1]) != '-') ifile = argv[++i];
	    windex = true;
	} else if (argi == "-exit") {
	    exit = true;
	} else if (argi == "-version") {
	    cout << "PSLmon version: " << PIDCVSHDR << endl;
	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    cerr << "Error in command. Syntax:" << endl;
	    cerr << argv[0] << " [-conf <cfile>] [-ndsindex [<file>]] "
		 << "[-exit] [-version]" << endl;
	    finish();
	    return;
	}
    }
    mCFile = cfile;
#ifndef DMTOFFLINE
    MonServer::setDebug(Debug());
#endif
    getDacc().setDebug(Debug());
    mTrend.setType(Trend::kMinute);
    mTrend.setAutoUpdate(false);

    //----------------------------------  Get default monitor name
    const char* name = argv[0];
    for (const char* j=name ; *j ; j++) if (*j == '/') name = j+1;

    //----------------------------------  Define and initialize parameters.
    mDict.addPar("StatMask",   11111);
    mDict.addPar("StatFile",   "");
    mDict.addPar("Stride",     double(1.0));
    mDict.addPar("TrigEnable", 0);
    mDict.addPar("MonName",    string(name));
    mDict.addPar("ChanPfx",    string(name,0,4));
    mDict.addPar("OSCFile",    string(""));
    mDict.addPar("TrendCount", 1);
    mDict.addPar("TrendFile",  string(""));
    mDict.addPar("SummaryTime", double(43200.0));
    mDict.addPar("BandHisto",  0);
    mDict.addPar("HistoFile",  string(""));
    mDict.addPar("SegmentMode",  string("None"));
    mDict.addPar("SegmentTime",  double(0.0));

    //----------------------------------  Read configuration and set up
    if (ReadConfig()) {
        finish();
	return;
    }

    //----------------------------------  Set up the rest
    string MonName     = mDict.getString("MonName");
    const char* myName = MonName.c_str();
    setServerName(myName);
    if (mDict.getString("StatFile").empty()) {
        string statfile;
        const char* htmldir = getenv("DMTHTMLOUT");
	if (htmldir) statfile = string(htmldir) + "/";
	statfile += MonName;
	statfile += "-stats.txt";
	mDict.setPar("StatFile", statfile);
    }

    //----------------------------------  Set up trigger writing.
    if (mDict.getInt("TrigEnable")) {
	TrigClient::setDebug(Debug());
	TrigClient::enroll(trig::TrigWriter::kNone);
    }

    //----------------------------------  Set up segment accountant
    trig::SegAccountant::flush_mode  fm =  
	trig::SegAccountant::flush_char_to_mode(mDict.getString("SegmentMode"));
    mSegAcct.set_mode(fm, mDict.getDouble("SegmentTime"));

    //----------------------------------  Set up trend writing
    mTrend.setName(myName);
    mTrend.setFrameCount(mDict.getInt("TrendCount"));
    if (windex) mTrend.writeIndex(ifile);

    if (!mDict.getString("OSCFile").empty()) {
        mOSC.readConfig(mDict.getString("OSCFile").c_str());
    }
    getDacc().setIgnoreMissingChannel(true);
    if (exit) finish();
    mUTimer.set_start();
}

#ifdef ROOT_HISTOS
//======================================  Histogram conversion
inline TH1*
makeRootHist(const char* name, const Histogram1& h) {
    int nBin = h.GetNBins();
    Histogram1::xbin_t lo = h.GetBinLowEdge(0);
    Histogram1::xbin_t hi = h.GetBinLowEdge(nBin+1);
    TH1D* rh = new TH1D(name, h.GetTitle(), nBin, lo, hi);
    Histogram1::histdata_t* bins = new Histogram1::histdata_t[nBin+2];
    h.GetBinContents(bins);
    rh->SetContent(bins);
    delete[] bins;
    rh->SetEntries(h.GetNEntries());
    return rh;
}
#endif

//======================================  Skeleton object destructor.
PSLmon::~PSLmon() {
    printStats();
#ifdef ROOT_HISTOS
    if (!mDict.getString("HistoFile").empty()) {
        TFile f(mDict.getString("HistoFile").c_str(), 
		"RECREATE", "PSLmon Histograms");
        for (const_dict_iter i=refDictionary().begin(); 
	     i != refDictionary().end(); ++i) {
	    const DataDesc& dd(i->second);
	    if (dd.getType() == DataDesc::t_Histogram1) {
	        const Histogram1* hist = reinterpret_cast<const Histogram1*>(dd.getAddr());
		TH1* rhist = makeRootHist(i->first.c_str(), *hist); 
		if (rhist) rhist->Write(rhist->GetName());
	    }
	}
        f.Close();
     }
#endif
    Reset();
}

//======================================  Print Statistics
void
PSLmon::printStats(void) const {
    char tString[40];
    const char* dateFmt = "%2m/%02d/%02y %2H:%02N:%02S (GPS %s)";
    int mask = mDict.getInt("StatMask");
    if (!mask) return;
    ofstream out(mDict.getString("StatFile").c_str(), ios::out);

    //----------------------------------  Global statistics
    if (mask%10 != 0) {
        LocalStr(Now(), tString, dateFmt);
        out << "             PSLmon Statistics at: " << tString << endl;
	out << endl;
        LocalStr(mStartProcess, tString, dateFmt);
	out << "Job started at:        " << tString << endl;
        LocalStr(mStartData, tString, dateFmt);
	out << "First data processed:  " << tString << endl;
        LocalStr(getDacc().getCurrentTime(), tString, dateFmt);
	out << "Latest data processed: " << tString << endl;
	out << "Total data processed:  " << mProcessedTime << " secs" << endl;
	out << "User processing time:  " << mUTimer.get_time() << " secs" << endl;
	out << endl;
    }

    //----------------------------------  Glitch tool statistics
    if (!mGlitches.empty() && (mask/10)%10) {
        double obsRate=0.0;
        mGlitches.begin()->printStatHdr(out);
        for (GlitchConstIter i=mGlitches.begin(); i!=mGlitches.end(); ++i) {
	    i->printStats(out);
	    obsRate += i->getNTotal();
	}
	if (!mProcessedTime) obsRate  = 0.0;
	else                 obsRate /= double(mProcessedTime);
	out << "Total observed glitch rate: " << obsRate << endl;
	out << endl;
    }

    //----------------------------------  Band tool statistics
    if (!mBands.empty() && (mask/100)%10) {
        mBands.begin()->printStatHdr(out);
        for (BandConstIter i=mBands.begin() ; i!=mBands.end() ; i++) {
	    i->printStats(out);
	}
    }
    out.close();
}

//======================================  Interrupt handler
void
PSLmon::Attention(void) {
    if (Debug()) cout << "PSLmon: Attention Interrupt" << endl;
    MonServer::Attention();
}

//======================================  Read in the configuration file.
bool
PSLmon::ReadConfig(void) {
    Reset();
    int nConfig = 0;

    //----------------------------------  Set up the line parser
    ParseLine lp(mCFile.c_str());
    if (!lp.isOpen()) {
        cerr << "Unable to open configuration file " << mCFile << endl;
	return true;
    }

    ofstream* conf = 0;
    if (mDict.getInt("StatMask")%10 != 0) {
        string outfile;
	if (getenv("DMTHTMLOUT")) outfile=string(getenv("DMTHTMLOUT")) + "/";
	outfile += mDict.getString("MonName");
	outfile += "-config.txt";
	conf = new ofstream(outfile.c_str(), ios::out);
	lp.setLog(*conf);
    } else if (Debug()) {
        lp.setLog(cout);
    }

    bool gAlarm = false, bAlarm=false;
    for (int nArg=lp.getLine() ; nArg>=0 ; nArg=lp.getLine()) {

        //------------------------------  Empty command line
        if (!lp[0]) {
	    continue;

        //------------------------------  Parse parameter line
        } else if (!strcmp(lp[0], "Parameter")) {
	    for (int i=1 ; i<nArg ; i+=2) {
	        if (!mDict.exists(lp[i])) {
		    cout << "No such parameter: " << lp[i] << endl;
		    if (conf) *conf << "**** No such parameter: " << lp[i] 
				    << endl;
		    break;
		} else {
		    if (Debug()) cout << "Set parameter: " << lp[i] 
				      << " to: " << lp[i+1] << endl;
		    mDict.setPar(lp[i], lp[i+1]);
		    if (string(lp[i]) == "MonName") {
		        mDict.setPar("ChanPfx", string(lp[i+1], 0, 4));
		    }
		}
		if (string(lp[i]) == "MonName") mTrend.setName(lp[i+1]);
	    }

        //------------------------------  Parse channel definition
        } else if (!strcmp(lp[0], "Channel")) {
	    if (nArg < 2) {
	        cout << "Channel not specified" << endl;
		if (conf) *conf << "**** Channel not specified" << endl;
		continue;
	    }
	    string cname(lp[1]);

	    //--------------------------  Create a PSLChan object for chanid
	    string chanid(cname);
	    int inx = 2;
	    if (nArg > 2 && *lp[2] != '-') {
		inx    = 3;
	        chanid = lp[2];
	    }
	    PSLChan pChan(chanid);

	    //--------------------------  Loop over options.
	    bool syntax = false;
	    char *p;
	    for (int i=inx ; i<nArg ; i++) {
	        string lpi=lp[i];
		if (lpi == "-bias") {
		    double bias = strtod(lp[++i], &p);
		    syntax |= *p;
		    pChan.setBias(bias);
		} else if (lpi == "-overlap") {
		    double ovlp = strtod(lp[++i], &p);
		    syntax |= *p;
		    pChan.setOverlap(ovlp);
		} else if (lpi == "-scale") {
		    double scale = strtod(lp[++i], &p);
		    syntax |= (!scale || *p);
		    pChan.setScale(scale);
		} else if (lpi == "-window") {
		    FiltrIterator f = mFiltr.find(lp[++i]);
		    if (f == mFiltr.end()) syntax = true;
		    else                   pChan.setPipe(f->second);
		} else {
		    syntax = true;
		    cout << "Invalid channel parameter" << lpi << endl;
		    if (conf) *conf << "**** Invalid channel parameter" << lpi 
				    << endl;
		}
		if (syntax) {
		    cout << "Command syntax is:" << endl;
		    cout << "Channel [<short-name>] <channel> [-bias <bval>] "
			 << "[-scale <sval>] [-window <filter>] [-overlap <sec>]"
			 << endl;
		    break;
		}
	    }
	    if (!syntax) mChannels[cname] = pChan;

        //------------------------------  Parse Filter definition
        } else if (!strcmp(lp[0], "Filter")) {
	    if (nArg < 3) {
	        cout << "**** Invalid Filter specifier. Syntax: " << endl;
		cout << "    Filter <name> <type> [-settle <time>] ";
		cout << "[-quip <nick>] <parameters>" << endl;
	        if (conf) *conf << "**** Invalid Filter specifier." << endl;
		continue;
	    }
	    string fName(lp[1]);
	    string fType(lp[2]);
	    int inx = 3;
	    PSLfilter filt(fName);
	    Interval settle(0.0);

	    while (inx+1 < nArg) {
	        if (string(lp[inx]) == "-settle") {
		    inx++;
		    filt.setSettle(lp.getDouble(inx++));
		} else if (string(lp[inx]) == "-quip") {
		    inx++;
		    filt.setQuip(lp[inx++]);
		} else {
		    break;
		}
	    }

	    //--------------------------  BaseLine filter
	    if (fType == "BaseLine") {
	        double tConst(1.0);
		if (nArg > inx) tConst = lp.getDouble(inx);
		BaseLine bas(tConst);
	        filt.set(bas);

	    //--------------------------  Blackman window
	    } else if (fType == "Blackman") {
	        Blackman blk;
	        filt.set(blk);

	    //--------------------------  Design a filter
	    } else if (fType == "Design") {
	        if (nArg-inx < 2) {
		    cout << "*** Filter design parameters are: "
			 << "<sample-rate> <string>" << endl;
		    continue;
		}
		FilterDesign fd(lp[inx+1], lp.getDouble(inx));
	        filt.set(fd.release());

	    //--------------------------  Difference FIR filter
	    } else if (fType == "Difference") {
	        if (nArg-inx < 1) {
		    cout << "*** Difference filter requires a frequency!" 
			 << endl;
		    continue;
		}
		Difference dif(lp.getDouble(inx));
	        filt.set(dif);

	    //--------------------------  FIR filter
	    } else if (fType == "FIRFilter") {
	        if (nArg-inx < 2) {
		    cout << "Specify rate, coefficients" << endl;
		    continue;
		}
		double rate = strtod(lp[inx], 0);
		if (Debug()) cout << "Filter: " << fName << " rate " 
				  << rate << endl;
		vector<double> coefs;
		const char* ptr = lp[inx+1];
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    coefs.push_back(re);
		    if (Debug()) cout << "Coefficient " <<coefs.size() 
				      << ": " << re << endl;
		}
		FIRFilter* fir = new FIRFilter(coefs.size(), rate);
		fir->setCoefs(coefs.size(), &coefs[0]);
		filt.set(fir);

	    //--------------------------  Hamming window
	    } else if (fType == "Hamming") {
	        Hamming ham;
	        filt.set(ham);

	    //--------------------------  Hanning window
	    } else if (fType == "Hanning") {
	        Hanning han;
	        filt.set(han);

	    //--------------------------  IIR Filter
	    } else if (fType == "IIRFilter") {
	        if (nArg-inx < 3) {
		    cout << "Specify rate, poles and zeros" << endl;
		    continue;
		}
		double rate = strtod(lp[inx], 0);
		if (Debug()) cout << "Filter: " << fName << " rate " 
				  << rate << endl;
	        vector<dComplex> poles, zeros;
		const char* ptr = lp[inx+1];
		double twopi=2*3.1415926535;
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    poles.push_back(dComplex(re,im)/twopi);
		    if (Debug()) cout << "pole " << poles.size() << ": "
				      << re << " + " << im << "i" << endl;
		}
		ptr = lp[inx+2];
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    zeros.push_back(dComplex(re,im)/twopi);
                    if (Debug()) cout << "zero " << zeros.size() << ": "
                                      << re << " + " << im << "i" << endl;
		}
		filt.set(IIRFilter(poles.size(), &poles[0], 
				   zeros.size(), &zeros[0], rate));

	    //--------------------------  Multiple filter
	    } else if (fType == "MultiPipe") {
	        MultiPipe* mPipe = new MultiPipe;
	        for (int i=inx ; i<nArg ; i++) {
		    if (mFiltr.find(lp[i]) != mFiltr.end()) {
		        cout << "Filter " << lp[i] 
			     << " could not be added to " << fName << endl;
		        continue;
		    }
		    mPipe->addPipe(*(mFiltr[lp[i]]));
		}
		filt.set(mPipe);

	    //--------------------------  Line removal filter
	    } else if (fType == "LineFilter") {
	        cout << "Line Filters not implemented" << endl;

	    //--------------------------  Unknown filter type
	    } else {
	        if (conf) *conf << "**** Unrecognized filter type: " 
				<< fType << endl;
	        cout << "**** Unrecognized filter type: " << fType << endl;
		continue;
	    }
	    if (!filt.null()) mFiltr[fName] = filt;

        //------------------------------  Parse Spectrum definition
        } else if (!strcmp(lp[0], "Spectrum")) {
	    if (nArg < 3 || !chanExists(lp[2])) {
	        cout << "***  Spectrum not named. Syntax: " << endl;
		cout << "     Spectrum <spec-name> <chan> <args..>" << endl;
		if (conf) *conf << "**** Too few arguments or unknown channel"
				<<endl;
		continue;
	    }
	    if (Debug()) cout << "Add spectrum: " << lp[1] << endl;
	    mSpectra.push_back(PSLSpec(lp[1], &mChannels[lp[2]]));
	    Interval avTime(1.0);
	    Interval svTime(3600.0);
	    const char* svFile=0;
	    for (int i=3 ; i<nArg ; i++) {
	        char* p;
		bool syntax = false;

	        if (string(lp[i]) == "-avtime") {
		    avTime = strtod(lp[++i], &p);
		    syntax = (!avTime || *p);
		} else if (string(lp[i]) == "-svtime") {
		    svTime = strtod(lp[++i], &p);
		    syntax = (!svTime || *p);
		} else if (string(lp[i]) == "-svfile") {
		    svFile = lp[++i];
		} else if (string(lp[i]) == "-while") {
		    mSpectra.back().setEnableCond(lp[++i]);
		} else {
		    syntax = true;
		    cout << "Undefined spectrum argument: " << lp[i] << endl;
		}
		if (syntax) {
		    cout << "Command syntax is:" << endl;
		    cout << "Spectrum <spec-name> <chan> [-avTime <avsec>] "
			 << "[-svtime <svsec>] [-svfile <file>]" << endl;
		    cout << "         [-while <osc-cond>]" << endl;
		    mSpectra.pop_back();
		    break;
		}
	    }

	    //--------------------------  Set the spectrum parameters
	    if (svFile) mSpectra.back().setSave(svTime, svFile);
	    mSpectra.back().setAvg(avTime);
	    nConfig++;

	//------------------------------  Configure a Band
        } else if (!strcmp(lp[0], "Band")) {
	    if (nArg < 3 || !chanExists(lp[2])) {
	        cout << "***  Band not named. Syntax: " << endl;
		cout << "     Band <band-name> <chan> <args..>" << endl;
		if (conf) *conf << "**** Too few arguments or unknown channel"
				<< endl;
		continue;
	    }
	    if (Debug()) cout << "Add Band: " << lp[1] << endl;
	    
            PSLBand band(lp[1], &mChannels[lp[2]]);
	    band.setDebug(Debug());
	    band.setHistEnable(mDict.getInt("BandHisto"));

            //--------------------------  Get the arguments
            double fmin=0, fmax=0;
	    double abslo=0, abshi=0, delt=0, fraclo=0, frachi=0;
	    bool   alarm = false;
            int setfl(0);
	    const char* baseChan(mChannels[lp[2]].getChannel());
	    const char* chanSuffix(0);
	    const char* trendChan(0);
	    const char* viewerChan(0);
	    for (int i=3 ; i<nArg ; i++) {
	        char* p;
		bool syntax = false;
		string argi = lp[i];

	        if (argi == "-abslow") {
		    abslo = strtod(lp[++i], &p);
		    syntax = (*p);
                    setfl |= 1;
		} else if (argi == "-abshi") {
		    abshi = strtod(lp[++i], &p);
		    syntax = (*p);
                    setfl |= 2;
		} else if (argi == "-alarm") {
		    alarm = true;
		} else if (argi == "-comment") {
		    band.setComment(lp[++i]);
	        } else if (argi == "-delt") {
		    delt = strtod(lp[++i], &p);
		    syntax = (!delt || *p);
                    setfl |= 4;
	        } else if (argi == "-exclude") {
		    double xmin = strtod(lp[++i], &p);
		    if (! (syntax = *p++ != '-')) {
		        double xmax = strtod(p, &p);
			syntax = (*p);
			band.excludeBand(xmin, xmax);
		    }
	        } else if (argi == "-fmin") {
		    fmin = strtod(lp[++i], &p);
		    syntax = (*p);
		} else if (argi == "-fmax") {
		    fmax = strtod(lp[++i], &p);
		    syntax = (*p);
		    band.setBand(fmin, fmax);
	        } else if (argi == "-fraclow") {
		    fraclo = strtod(lp[++i], &p);
		    syntax = *p;
                    setfl |= 8;
		} else if (argi == "-frachi") {
		    frachi = strtod(lp[++i], &p);
		    syntax = (!frachi || *p);
                    setfl |= 16;
		} else if (argi == "-hist") {
		    band.setHistEnable(true);
		} else if (argi == "-nohist") {
		    band.setHistEnable(false);
		} else if (argi == "-notrend") {
		    baseChan = 0;
		} else if (argi == "-segment") {
		    band.setSegment(lp[++i]);
		} else if (argi == "-suffix") {
		    chanSuffix = lp[++i];
		} else if (argi == "-trend") {
		    if (i+1 < nArg && *(lp[i+1]) != '-') trendChan = lp[++i];
		} else if (argi == "-trigger") {
		    band.setTrigEnable(true);
		} else if (argi == "-tavg") {
		    band.setAvgTime(lp.getDouble(++i));
		} else if (argi == "-viewer") {
		    if (i+1 < nArg && *(lp[i+1]) != '-') viewerChan = lp[++i];
		} else if (argi == "-while") {
		    band.setEnableCond(lp[++i]);
		} else {
		    syntax = true;
		    cout << "**** Undefined Band argument: " << argi << endl;
		    if (conf) *conf << "**** Undefined argument: " << argi 

				    << endl;
		}
		if (syntax) {
		    cout << "Command syntax is:" << endl;
		    cout << "Band <band-name> <chan> [-alarm] [-fmin <min>] "
		         << " [-fmax <max>] [-abslo <ymin>] [-abshi <ymax>] "
			 << endl;
		    cout << "     [-delta <dymax>] [-fraclow <fydown>] "
			 << "[-frachi <fyup>] [-tavg <time>] "
			 << "[-trend <trndChan>] [-viewer <v-chan>]" << endl;
		    if (conf) *conf << "**** Syntax error in line." << endl;
		    break;
		}
	    }

	    //--------------------------  Set band parameters.
	    if ((setfl & 3)) band.setAbsLimit(abslo, abshi);
	    if ((setfl & 4)) band.setDeltLimit(delt);
	    if ((setfl & 24) == 24) {
	        band.setFracLimit(fraclo, frachi);
	    } else if ((setfl & 24) == 8) {
	        band.setFracLimit(fraclo, 0);
	    } else if ((setfl & 24) == 16) {
	        band.setFracLimit(frachi);
	    }

	    //-------------------------  set up output channels
	    OutChan ochan(mTrend, "", mDict.getString("ChanPfx"));
	    if (trendChan) {
		ochan.setTrendChannel(trendChan);
	    } else if (baseChan) {
		if (!chanSuffix) {
		    ostringstream sufx;
		    sufx << "BAND_" << band.getFLow()<< "_" << band.getFHigh();
		    ochan.setChannel(baseChan, sufx.str());
		    if (Debug()) cout << "Trend: " << ochan.getTrendChannel()
				      << endl;
		} else {
		    ochan.setChannel(baseChan, chanSuffix);
		}
	    }
	    if (viewerChan) {
	        ochan.setViewerChannel(viewerChan);
	    } else if (!*ochan.getViewerChannel()) {
	        ochan.setViewerChannel(lp[1]);
	    }
	    band.refOutChan() = ochan;

	    //--------------------------  set alarms
	    const char* alDesc = 
	        "Power in $1 band from $2 - $3 Hz exceeds limit";
	    if (alarm) {
	        band.setAlarmEnable(alarm);
		if (!bAlarm) {
		    AlarmData al("PSLmon", "BandPower", Interval(300.0),
				 4, alDesc, "");
		    al.setWebFile("Monitors/PSLmon/index.html");
		    al.jamFlags(AlarmData::kReTrigger);
		    mAlarm.defineAlarm(al);
		}
		bAlarm = true;
	    }
	    mBands.push_back(band);
	    nConfig++;

	//------------------------------  Configure a Coherence tool
        } else if (!strcmp(lp[0], "Coherence")) {
	    if (nArg < 4 || !chanExists(lp[2])) {
	        cout << "***  Coherence not named. Syntax: " << endl;
		cout << "     Coherence <tool-name> <chanA> <chanB> <args..>"
		     << endl;
		if (conf) *conf << "**** Too few arguments or unknown channel"
				<<endl;
		continue;
	    }
	    if (Debug()) cout << "Add Coherence: " << lp[1] << endl;
            mCorrs.push_back(PSLCorr(lp[1], &mChannels[lp[2]], &mChannels[lp[3]]));

            //--------------------------  Get the arguments
            double fmin=0, fmax=0;
	    double abslo=0, abshi=0, delt=0, fraclo=0, frachi=0;
            int setfl(0);
	    const char* trendChan=lp[1];
	    for (int i=4 ; i<nArg ; i++) {
	        char* p;
		bool syntax = false;

	        if (string(lp[i]) == "-abslow") {
		    abslo = strtod(lp[++i], &p);
		    syntax = (*p);
                    setfl |= 1;
		} else if (string(lp[i]) == "-abshi") {
		    abshi = strtod(lp[++i], &p);
		    syntax = (*p);
                    setfl |= 2;
	        } else if (string(lp[i]) == "-delt") {
		    delt = strtod(lp[++i], &p);
		    syntax = (!delt || *p);
                    setfl |= 4;
	        } else if (string(lp[i]) == "-fmin") {
		    fmin = strtod(lp[++i], &p);
		    syntax = (*p);
		} else if (string(lp[i]) == "-fmax") {
		    fmax = strtod(lp[++i], &p);
		    syntax = (*p);
	        } else if (string(lp[i]) == "-fraclow") {
		    fraclo = strtod(lp[++i], &p);
		    syntax = (!fraclo || *p);
                    setfl |= 8;
		} else if (string(lp[i]) == "-frachi") {
		    frachi = strtod(lp[++i], &p);
		    syntax = (!frachi || *p);
                    setfl |= 16;
		} else if (string(lp[i]) == "-trend") {
		    trendChan = lp[++i];
		} else if (string(lp[i]) == "-while") {
		    mCorrs.back().setEnableCond(lp[++i]);
		} else {
		    syntax = true;
		    cout << "Undefined spectrum argument: " << lp[i] << endl;
		}
		if (syntax) {
		    cout << "Command syntax is:" << endl;
		    cout << "Cohere <band-name> <chanA> <chanB> [-fmin <min>] "
		         << " [-fmax <max>] [-abslo <ymin>] [-abshi <ymax>] "
			 << endl;
		    cout << "     [-delta <dymax>] [-fraclow <fydown>] "
			 << "[-frachi <fyup>] [-trend <trndChan>]" << endl;
		    break;
		}
	    }

	    //--------------------------  Set band parameters.
	    mCorrs.back().setBand(fmin, fmax);
	    if ((setfl & 3)) mCorrs.back().setAbsLimit(abslo, abshi);
	    if ((setfl & 4)) mCorrs.back().setDeltLimit(delt);
	    if ((setfl & 24) == 24) {
	        mCorrs.back().setFracLimit(fraclo, frachi);
	    } else if ((setfl & 24) == 8) {
	        mCorrs.back().setFracLimit(fraclo, frachi);
	    } else if ((setfl & 24) == 16) {
	        mCorrs.back().setFracLimit(frachi);
	    }
	    if (trendChan) {
	        mCorrs.back().setTrend(trendChan);
		mTrend.addChannel(trendChan);
	    }
	    nConfig++;

	//------------------------------  Configure a glitch
        } else if (!strcmp(lp[0], "Glitch")) {
	    if (nArg < 3 || !chanExists(lp[2])) {
	        cout << "***  Glitch not named. Syntax: " << endl;
		cout << "     Glitch <glitch-name> <chan> <args..>" << endl;
		if (conf) *conf << "**** Too few arguments or unknown channel"
				<<endl;
		continue;
	    }

	    //--------------------------  Check for duplicate name
	    GlitchConstIter itsav = mGlitches.end();
	    for(GlitchConstIter it=mGlitches.begin(); it != itsav; ++it){
	        if (it->getName() == lp[1]) {
		    itsav = it;
		    break;
		}
	    }
	    if (itsav != mGlitches.end()) {
	        cout << "**** Duplicate tool name: " << lp[1] << endl;
		if (conf) *conf << "**** Duplicate tool name" << endl;
		continue;
	    }

	    //--------------------------  Construct the glitch tool
	    if (Debug()) cout << "Add Glitch: " << lp[1] << endl;
	    const char* baseChan(mChannels[lp[2]].getChannel());
	    PSLGlitch glitch(lp[1], &mChannels[lp[2] ]);
	    glitch.setDebug(Debug());

	    //--------------------------  Get the arguments
	    Interval snipTime(0.001);
	    double   sigma = 4.0;
	    double   asize = 0.0;
	    double   alarm = 0.0;
	    Interval aTime = 0.0;
	    const char* trendChan(0);
	    const char* chanSuffx(0);
	    const char* fName(0);
	    for (int i=3 ; i<nArg ; i++) {
	        char* p;
		bool syntax = false;
		string lpStr(lp[i]);

	        if (lpStr == "-alarm") {
		    alarm = strtod(lp[++i], &p);
		    syntax = (alarm == 0.0 || *p);
		} else if (lpStr == "-atime") {
		    aTime = Interval(strtod(lp[++i], &p));
		    syntax = (!aTime || *p);
		} else if (lpStr == "-comment") {
		    glitch.setComment(lp[++i]);
		} else if (lpStr == "-filter") {
		    FiltrIterator f = mFiltr.find(lp[++i]);
		    if (f == mFiltr.end()) {
		        syntax = true;
		    } else {
		        glitch.setFilter(f->second);
			fName = lp[i];
		    }
		} else if (lpStr == "-notrend") {
		    baseChan = 0;
		} else if (lpStr == "-segment") {
		    glitch.setSegment(lp[++i]);
		} else if (lpStr == "-sigma") {
		    sigma = strtod(lp[++i], &p);
		    syntax = (!sigma || *p);
		} else if (lpStr == "-size") {
		    asize = strtod(lp[++i], &p);
		    syntax = (!asize || *p);
		} else if (lpStr == "-snip") {
		    snipTime = strtod(lp[++i], &p);
		    syntax = (!snipTime || *p);
		} else if (lpStr == "-suffix") {
		    chanSuffx = lp[++i];
		} else if (lpStr == "-trend") {
		    if (i < nArg-1) trendChan = lp[++i];
		} else if (lpStr == "-trigger") {
		    if (i+1 >= nArg || lp[i+1][0] == '-') glitch.setTrigger(1);
		    else                     glitch.setTrigger(lp.getInt(++i));
		} else if (string(lp[i]) == "-while") {
		    glitch.setEnableCond(lp[++i]);
		} else {
		    syntax = true;
		    cout << "Undefined glitch argument: " << lpStr << endl;
		}
		if (syntax) {
		    cout << "Command syntax is:" << endl;
		    cout << "Glitch <glitch-name> <chan> [-snip <ssec>] "
			 << "[-trend <trndChan> | -notrend] [-sigma <nSig>]" 
			 << endl;
		    cout << "        [-while <osc-cond>] [-trigger <N>] [-alarm] " 
			 << "[-filter <filter>] [-atime <avg-time>]" << endl;
		    break;
		}
	    }

	    //--------------------------  Set glitch parameters.
	    glitch.setSigmaT(sigma);
	    glitch.setSize(asize);
	    glitch.setTime(snipTime);

	    //-------------------------  set up output channels
	    OutChan ochan(mTrend, "", mDict.getString("ChanPfx"));
	    if (trendChan) {
		ochan.setTrendChannel(trendChan);
	    } else if (baseChan) {
		if (!chanSuffx) {
		    ostringstream sufx;
		    sufx << "S" << glitch.getSigmaT();
		    if (fName) sufx << "_f" << mFiltr[fName].getQuip();
		    ochan.setChannel(baseChan, sufx.str());
		    if (Debug()) cout << "Trend: " << ochan.getTrendChannel()
				      << endl;
		} else {
		    ochan.setChannel(baseChan, chanSuffx);
		}
	    }
	    if (!*ochan.getViewerChannel()) {
	        ochan.setViewerChannel(lp[1]);
	    }
	    glitch.refHistoryChan() = ochan;

	    //--------------------------  Set alarm Parameters
	    const char* alDesc = 
	          "Glitch rate on channel $1 exceeds limit [$2]";
	    if (alarm > 0.0) {
	        glitch.setAlarmParams(alarm, aTime);
		if (!gAlarm) {
		    AlarmData al("PSLmon", "GlitchRate", Interval(300.0), 
				 4, alDesc, "");
		    al.setWebFile("Monitors/PSLmon/index.html");
		    al.jamFlags(AlarmData::kReTrigger);
		    mAlarm.defineAlarm(al);
		}
		gAlarm = true;
	    }
	    mGlitches.push_back(glitch);
	    nConfig++;

	//------------------------------  Invalid statement.
	} else {
	    cout << "Invalid command in " << mCFile << ": " << lp[0] << endl;
	}
    }

    //----------------------------------  Close out the configuration output.
    if (conf) {
        conf->close();
	delete conf;
    }

    //----------------------------------  Check that >=1 tool configured.
    if (nConfig == 0) {
        cerr << "No PSLmon tools configured!" << endl;
	return true;
    }

    //----------------------------------  Act on the parameter values.
    getDacc().setStride(Interval(mDict.getDouble("Stride")));

    //----------------------------------  Connect channels to Dacc
    for (ChanIterator i=mChannels.begin() ; i != mChannels.end() ; i++) {
        i->second.setup(getDacc());
    }

    //----------------------------------  Serve average spectra
    for (SpecIterator i=mSpectra.begin() ; i != mSpectra.end() ; i++) {
        if (Debug() > 1) cout << "Serving spectrum: " << i->getName() << endl;
        serveData(i->getName(), &(i->refAverage()), "Spectrum");
    }

    //----------------------------------  Serve band rms
    for (BandIterator i=mBands.begin() ; i != mBands.end() ; i++) {
	i->refOutChan().setHistoryLen(mDict.getDouble("SummaryTime"));
	i->postConfig(*this);
	i->confSegment(mSegAcct);
    }

    //----------------------------------  Serve glitch series
    for (GlitchIterator i=mGlitches.begin() ; i != mGlitches.end() ; i++) {
	i->refHistoryChan().setHistoryLen(mDict.getDouble("SummaryTime"));
	i->postConfig(*this);
	i->confSegment(mSegAcct);
    }
    return false;
}

//======================================  Clear all lists.
void
PSLmon::Reset(void) {

    //----------------------------------  Delete Dacc channels
    while (mChannels.begin() !=mChannels.end()) {
        mChannels.erase(mChannels.begin());
    }

    //----------------------------------  Delete Spectrum requests
    while (mSpectra.begin() != mSpectra.end()) {
        mSpectra.erase(mSpectra.begin());
    }

    //----------------------------------  Delete requested bands
    while (mBands.begin() != mBands.end()) {
        mBands.erase(mBands.begin());
    }

    //----------------------------------  Delete requested bands
    while (mCorrs.begin() != mCorrs.end()) {
        mCorrs.erase(mCorrs.begin());
    }

    //----------------------------------  Delete Glich search requests
    while (mGlitches.begin() != mGlitches.end()) {
        mGlitches.erase(mGlitches.begin());
    }
}

//======================================  Frame processing function.
void
PSLmon::ProcessData(void) {

    //---------------------------------  Check the stride alignment
    Time tEnd   = getDacc().getCurrentTime();
    double tSec = tEnd.totalS();
    double Stride = getDacc().getStride();
    if (fmod(tSec, Stride) != 0) {
        tSec = Stride - fmod(tSec, Stride);
	cout << "PSLmon: Skipping to " << setprecision(13) 
	     << (tEnd+Interval(tSec)).totalS() << " GPS." << endl;
        getDacc().seek(tEnd+tSec);
	return;
    }

    //----------------------------------  Get the time info.
    Time tStart = getDacc().getFillTime();
    Interval dT(tEnd - tStart);
    if (!mStartData) {
        mStartProcess = Now();
	mStartData    = tStart;
    }
    AlarmHandle han;
    mSegAcct.start_stride(tStart);

    //----------------------------------  Get channel data
    for (ChanIterator i=mChannels.begin(); i!=mChannels.end(); i++) {
        i->second.start();
    }

    //----------------------------------  Process spectra
    for (SpecIterator i=mSpectra.begin() ; i!=mSpectra.end() ; i++) {
        const char* condName = i->getEnableCond();
        if (!*condName || mOSC.satisfied(condName)) i->crunch();
    }

    //----------------------------------  Process Bands
    for (BandIterator i=mBands.begin() ; i!= mBands.end() ; i++) {
        i->reset();
        if (i->testCondition(mOSC)) {
	    bool trig_bit = i->crunch();
	    if (trig_bit) {
	        if (Debug()) cout << "Trigger occurred in band: " 
				  << i->getName() << endl;
		if (mDict.getInt("TrigEnable")) i->sendTrigger(*this);
		if (i->getAlarmEnable()) {
		    ostringstream param;
		    param << i->getChannel() << " " << i->getFLow()  
			  << " " << i->getFHigh() << ends;
		    AlarmData al("PSLmon", "BandPower", 0, 4, "", param.str());
		    mAlarm.setAlarm(al, han);
		}
	    }
	    i->sendSegment(mSegAcct);
	} else if (! (tStart.getS()%60) ) {
	     i->refOutChan().setHistoryTime(tStart, dT);
	}
    }

    //----------------------------------  Process Coherence
    for (CorrIterator i=mCorrs.begin() ; i!=mCorrs.end() ; i++) {
        i->reset();
        const char* condName = i->getEnableCond();
        if (!*condName || mOSC.satisfied(condName)) {
	    if (i->crunch()) {
	        if (Debug()) cout << "Trigger occurred in correlation tool: " 
				  << i->getName() << endl;
		if (mDict.getInt("TrigEnable")) {
		    trig::TrigRslt t("PSLCorr", i->getName(), 1);
		    t.setTime(tStart);
		    t.setPriority(trig::p_error);
		    t.setDisposition(trig::d_metaDB);
		    t.setDuration(dT);
		    t.setIfos(i->getIFO().c_str());
		    t[0] = i->getCorr();
		    int rc = sendTrigger(t);
		    if (rc && rc!=12 ) {
		        cout << "Error " << rc << " logging Corr " 
			     << i->getName() << " trigger" << endl;
		    }
		}
	    }
	    const char* tChan = i->getTrend();
	    if (*tChan) mTrend.trendData(tChan, tStart, i->getCorr());
	}
    }

    //----------------------------------  Process Glitches
    for (GlitchIterator i=mGlitches.begin() ; i!=mGlitches.end() ; i++) {
        i->reset();
        if (i->testCondition(mOSC)) {
	    i->crunch();
	    int nGlitch = i->getCount();
	    i->refHistoryChan().fillData(tStart+0.5*dT, double(nGlitch)/dT, dT);

	    if (nGlitch && mDict.getInt("TrigEnable")) {
	        i->sendTrigger(*this);
	    }

	    if (mDict.getInt("TrigEnable")) {
	        i->sendSegment(mSegAcct);
	    }

	    if (i->testAlarm()) {
	        ostringstream param;
		param << i->getChannel() << " " << i->getAlarmRate() << ends;
		AlarmData al("PSLmon", "GlitchRate", 0, 4, "", param.str());
		mAlarm.setAlarm(al, han);
	    }

	//-----------------------------  Update even if condition failed!
	} else if (! (tStart.getS()%60) ) {
	    i->refHistoryChan().setHistoryTime(tStart, dT);
	}
    }

    //----------------------------------  Clean up channel data
    for (ChanIterator i=mChannels.begin() ; i!=mChannels.end() ; i++) {
        i->second.finish();
    }

    //----------------------------------  End of stride processing
    mProcessedTime += dT;
    if (tStart.getS()%300 > tEnd.getS()%300) printStats();
    mTrend.Update(tEnd);
    mSegAcct.update(tEnd);
}

//======================================  Test if channel exists
bool
PSLmon::chanExists(const string& key) const {
    return mChannels.find(key) != mChannels.end();
}
