#include "PSLChan.hh"
#include "DVecType.hh"
#include "DaccAPI.hh"

using namespace std;

//======================================  Channel constructors
PSLChan::PSLChan(void) 
  : mDacc(0), mTS(0), mBias(0.0), mScale(1.0), mOverlap(0.0), 
    mSpecOK(false), mEndStride(0)
{
}

PSLChan::PSLChan(const string& name) 
  : mChannel(name), mDacc(0), mTS(0), mBias(0.0), mScale(1.0), 
    mOverlap(0.0), mSpecOK(false), mEndStride(0)
{
}

//======================================  Destructor
PSLChan::~PSLChan(void) {
    if (mDacc && mTS) mDacc->rmChannel(mChannel.c_str());
    if (mTS) delete mTS;
}

//======================================  Test if channel data is valid
bool
PSLChan::exists(void) const {
    return (mTS != 0 && !mTS->isEmpty());
}

//======================================  Get channel name
const char* 
PSLChan::getChannel(void) const {
    return mChannel.c_str();
}

//======================================  Get channel average value
double
PSLChan::getAverage(void) const {
    if (!mStatOK) setStats();
    return mAverage;
}

//======================================  Get channel average value
string
PSLChan::getIFO(void) const {
    if (mChannel[2] == ':') return mChannel.substr(0,2);
    return string("");
}

//======================================  Get channel sigma
double
PSLChan::getSigma(void) const {
    if (!mStatOK) setStats();
    return mSigma;
}

//======================================  Get channel Bias
void 
PSLChan::setBias(double bias) {
    mBias = bias;
}

//======================================  Get channel Bias
void 
PSLChan::setOverlap(Interval dt) {
    mOverlap = dt;
}

//======================================  Get channel Scale
void 
PSLChan::setPipe(const PSLfilter& p) {
    mPipe = p;
}

//======================================  Get channel Scale
void 
PSLChan::setScale(double scale) {
    mScale = scale;
}

//======================================  Calculate the statistics.
void
PSLChan::setStats(void) const {
    if (mTSeries.isEmpty()) {
        mAverage = 0.0;
	mSigma   = 0.0;
    } else {
        mAverage = mTSeries.getAverage();
	double sigma = (mTSeries * mTSeries)/double(mTSeries.getNSample())
	             - mAverage * mAverage;
	if (sigma <= 0) mSigma = 0.0;
	else            mSigma = sqrt(sigma);
    }
    mStatOK = true;
}

//======================================  Get FSeries data
const FSeries& 
PSLChan::refFSeries(void) const {
    if (!mSpecOK) {
        if (mPipe.null()) mFS.setData(mTSeries);
	else              mFS.setData(const_cast<PSLfilter&>(mPipe)(mTSeries));
	mSpecOK = true;
    }
    return mFS;
}

//======================================  Mark FSeries as invalid
void 
PSLChan::start(void) {
    mTSerOK   = false;
    mSpecOK   = false;
    mStatOK   = false;
    mLastData = mEndStride;
    if (!mTS || mTS->isEmpty()) return;
        
    //-----------------------------------  Convert the input series for effcy
    mTS->Convert(DVecType<chan_data>::getDataType());

    //-----------------------------------  Discard unneeded data
    if (!mOverlap) {
        if (mBias == 0.0 && mScale == 1.0) {
	    mTSeries = *mTS;
	} else {
	    mTSeries.copy(*mTS);
	    if (mBias  != 0.0) mTSeries += mBias;
	    if (mScale != 1.0) mTSeries *= mScale;
	}
	mTSerOK = true;
    } else if (mTSeries.getEndTime() != mTS->getStartTime()) {
        mTSeries.copy(*mTS);
	if (mBias  != 0.0) mTSeries += mBias;
	if (mScale != 1.0) mTSeries *= mScale;	
    } else {
        Interval dT = mTSeries.getInterval() -  mOverlap;
        if (double(dT) > 0.0) mTSeries.eraseStart(dT);
	TSeries::size_type N=mTSeries.getNSample();
	if (mTSeries.Append(*mTS)) throw runtime_error("Sample rate changed");
	TSeries::size_type M=mTS->getNSample();
	if (mBias  != 0.0) mTSeries.refDVect()->bias(N, mBias, M);
	if (mScale != 1.0) mTSeries.refDVect()->scale(N, mScale, M);
	if (double(dT) >= 0.0) mTSerOK = true;
    }
    if (mTSerOK) mEndStride = mTSeries.getEndTime();
}


//======================================  Mark FSeries as invalid
void 
PSLChan::finish(void) {
    mTSerOK   = false;
    mSpecOK   = false;
    mStatOK   = false;
    if (!mOverlap) mTSeries.Clear();
}

//======================================  Specify a Dacc, request the channel.
void 
PSLChan::setup(DaccAPI& In) {
    mDacc = &In;
    if (!In.isChannelRead(mChannel.c_str())) {
        In.addChannel(mChannel.c_str(), 1, &mTS);
    }
}
