/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wfigure.hh"
#include "wtile.hh"
#include "wtransform.hh"

#include "spline.hh"
#include "matlab_fcs.hh"
#include "lcl_array.hh"
#include "DVecType.hh"

#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>


using namespace wpipe;
using namespace std;
// Shourov K. Chatterji <shourov@ligo.mit.edu>
// $Id: wspectrogram.m 2753 2010-02-26 21:33:24Z jrollins $

//======================================  Plot a spectrogram.
void 
wfigure::spectro_plot(const qTransform& transform, const wtile& tiling, 
		      const Time& referenceTime, const dble_vect& timeRange, 
		      const dble_vect& requestedFrequencyRange, size_t iQmin,
		      size_t iQmax, const dble_vect& normalizedEnergyRange, 
		      long horizontalResolution)
{

   // spectrogram boundary
   double spectrogramLeft = 0.14;
   double spectrogramWidth = 0.80;
   double spectrogramBottom = 0.28;
   double spectrogramHeight = 0.62;

   //-----------------------------  set pad position, size (x0, y0, wx, wy)
   dble_vect spectrogramPosition(4, 0.0);
   spectrogramPosition[0] = spectrogramLeft;
   spectrogramPosition[1] = spectrogramBottom;
   spectrogramPosition[2] = spectrogramWidth;
   spectrogramPosition[3] = spectrogramHeight;

   // colorbar position
   double colorbarLeft = spectrogramLeft; 
   double colorbarWidth = spectrogramWidth;
   double colorbarBottom = 0.12;
   double colorbarHeight = 0.02;
   dble_vect colorbarPosition(4, 0.0);
   colorbarPosition[0] = colorbarLeft;
   colorbarPosition[1] = colorbarBottom;
   colorbarPosition[2] = colorbarWidth;
   colorbarPosition[3] = colorbarHeight;

   double dInf = 1.0 / 0.0;

   //-----------------------------------  Set up the time range.
   Time     tStart = referenceTime + timeRange[0];
   Interval plotDt = timeRange[1] - timeRange[0];

   // set x-axis scale.
   double tScale = _plot.xTimeScale(double(plotDt), "Time");

   // vector of times to display
   //times = linspace(min(timeRange), max(timeRange), horizontalResolution);
   lcl_array<double> times(horizontalResolution+1);
   double dt = (timeRange[1] - timeRange[0]) / double(horizontalResolution);
   for (int i=0; i<=horizontalResolution; ++i) {
      times[i] = (timeRange[0] + dt * double(i)) / tScale;
   }

   ////////////////////////////////////////////////////////////////////////////
   //                        begin loop over q planes                        //
   ////////////////////////////////////////////////////////////////////////////

   // loop over planes to display
   for (size_t planeIndex=iQmin; planeIndex<iQmax; planeIndex++) {
      const qplane& tileplanei(tiling.planes(planeIndex));

      /////////////////////////////////////////////////////////////////////////
      //                   identify frequency rows to display                //
      /////////////////////////////////////////////////////////////////////////

      // default minimum frequency is minimum frequency of available data
      double fMin = 0;
      if (requestedFrequencyRange[0] == -dInf || requestedFrequencyRange[0]<=0){
	 fMin = tiling.planes(planeIndex).minimumFrequency;
      } else {
	 fMin = requestedFrequencyRange[0];
      }

      // default maximum frequency is maximum frequency of available data
      double fMax = 0;
      if (requestedFrequencyRange[1] == dInf || requestedFrequencyRange[1] < 0){
	 fMax = tiling.planes(planeIndex).maximumFrequency;
      } else {
	 fMax = requestedFrequencyRange[1];
      }

      // validate selected frequency range
      if ((fMin < tileplanei.minimumFrequency) || 
	  (fMax > tileplanei.maximumFrequency)) {
	 cout << " Request: [" << fMin << ", " << fMax << "]  available: ["
	      << tileplanei.minimumFrequency << ", " 
	      << tileplanei.maximumFrequency << "]." << endl;
	 error("requested frequency range exceeds available data");
      }

      /////////////////////////////////////////////////////////////////////////
      //                       initialize display matrix                     //
      /////////////////////////////////////////////////////////////////////////

      // initialize matrix of normalized energies for display
      size_t numberOfRows = tileplanei.numberOfRows;
      lcl_array<double> frequencies(numberOfRows);
      size_t iFmin = numberOfRows;
      size_t iFmax = 0;
      for (size_t irow=0; irow < numberOfRows; irow++) {
	 double frow = tileplanei.row(irow).frequency;
	 frequencies[irow] = frow;
	 if (frow >= fMin && irow <  iFmin) iFmin = irow;
	 if (frow <= fMax && irow >= iFmax) iFmax = irow+1;
      }
      if (iFmin >= iFmax) {
	 error("Couldn't find rows in range");
      }

      // initialize maximum normalized energy
      size_t nArray = (iFmax-iFmin) * horizontalResolution;
      lcl_array<double> normalizedEnergies(nArray);

      //////////////////////////////////////////////////////////////////////////
      //                     begin loop over frequency rows                   //
      //////////////////////////////////////////////////////////////////////////

      size_t row = 0;
      // loop over rows
      for (size_t rowIndex=iFmin; rowIndex < iFmax; rowIndex++) {
	 //const qrow& tileij = tileplanei.row(rowIndex);
	 const trow& rowij(transform.planes(planeIndex).rows(rowIndex));
	 const TSeries& tsNormE(rowij.normalizedEnergies);

	 // find tiles within requested time range
	 //double padTime = 1.5 * tileij.timeStep;
	 Interval step = tsNormE.getTStep();

	 // corresponding tile normalized energies
	 TSeries rowNormErgs = tsNormE.extract(tStart-step, plotDt + step*3.0);
	 convertUnits(rowNormErgs);

	 size_t nPoints = rowNormErgs.getNSample();
	 lcl_array<double> rowTimes(nPoints);
	 double t0= double(rowNormErgs.getStartTime() - referenceTime) / tScale;
	 double rowDt = double(rowNormErgs.getTStep()) / tScale;
	 for (size_t i=0; i<nPoints; ++i) {
	    rowTimes[i] = t0 + double(i)*rowDt;
	 }

	 // interpolate to desired horizontal resolution, copy to display matrix
	 lcl_array<double> rowYp(nPoints);
	 const double* rowY = 
	    dynamic_cast<const DVectD&>(*rowNormErgs.refDVect()).refTData();
	 spline_pchip_set(nPoints, rowTimes, rowY, rowYp);
	 spline_pchip_val(nPoints, rowTimes, rowY, rowYp, 
			  horizontalResolution, times, 
			  normalizedEnergies + row*horizontalResolution);
	 row++;
      }	// end loop over rows
      if (!row) continue;

      double maximumNormalizedEnergy = 0;
      for (size_t i=0; i<nArray; i++) {
	 double tmp =  normalizedEnergies[i];
	 if (tmp < 0.0) {
	    normalizedEnergies[i] = 0.0;
	 } else if (tmp > maximumNormalizedEnergy) {
	    maximumNormalizedEnergy = tmp;
	 }	
      }
    
      //------------------------------- Set up frequency binning
      lcl_array<double> fBins(row+1);
      double hwid = sqrt(frequencies[1]/frequencies[0]);
      for (size_t i=0; i<row; i++) {
	 fBins[i] = frequencies[iFmin + i]/hwid;
      }
      fBins[row] = frequencies[iFmax - 1]*hwid;

      //////////////////////////////////////////////////////////////////////////
      //                           set colormap scaling                       //
      //////////////////////////////////////////////////////////////////////////

      // if normalized energy range is not specified, normalized energy 
      // range to code on colormap
      dble_vect colormapScale(2, 0.0);
      if (normalizedEnergyRange.empty()) {
	 colormapScale[1] = maximumNormalizedEnergy;
      }

      // or if autoscaling of upper limit is requested, normalized energy 
      // range to code on colormap
      else if (normalizedEnergyRange[1] == dInf) {
	 colormapScale[0] = normalizedEnergyRange[0];
	 colormapScale[1] = maximumNormalizedEnergy;
      }

      // otherwise, use specified range
      else {
	 colormapScale = normalizedEnergyRange;
      }

      /////////////////////////////////////////////////////////////////////////
      //                            plot spectrogram                         //
      /////////////////////////////////////////////////////////////////////////

      // set title properties
      ostringstream titleString;
      titleString << transform._channelName << " at "
		  << fixed << setprecision(3) << referenceTime.totalS() 
		  << " with Q of " << setprecision(1) 
		  << tiling.planes(planeIndex).q << endl;

      // titleString = strrep(titleString, "_", "\_");

      _plot.title(titleString.str());
      _plot.ylabel("Frequency [Hz]");

      // plot spectrogram
      _plot.ylog();
      _plot.set_zrange(colormapScale[0], colormapScale[1]);
      if (_zUnits.empty() || _zUnits == "NormE") {
	 _plot.zlabel("Normalized Energy");
      } else {
	 _plot.zlabel(_zUnits);
      }
      _plot.surf(horizontalResolution, times, row, fBins, normalizedEnergies);

   }      // end loop over planes
}

//======================================  Plot a spectrogram.
void 
wfigure::spectro_data(const qTransform& transform, const wtile& tiling, 
		      const Time& referenceTime, const dble_vect& timeRange, 
		      const dble_vect& requestedFrequencyRange, size_t iQmin,
		      size_t iQmax, const dble_vect& normalizedEnergyRange, 
		      long horizontalResolution)
{
   double dInf = 1.0 / 0.0;
   // set title properties
   ostringstream titleString;
   titleString << transform._channelName << " at "
	       << fixed << setprecision(3) << referenceTime.totalS() 
	       << " with Q of " << setprecision(1) 
	       << tiling.planes(iQmin).q;
   _data.set_title("wplot data", titleString.str());

   _data.add_param("eventTime", referenceTime.totalS(), "gps");
   _data.add_param("Qmin", tiling.planes(iQmin).q, "");
   _data.add_param("Qmax", tiling.planes(iQmax-1).q, "");

   _data.add_coord("frequency", "Hz");
   _data.add_coord("time", "s");
   _data.add_coord("Amplitude", _zUnits);

   Time tStart = referenceTime + timeRange[0];
   Interval plotDt = timeRange[1] - timeRange[0];
   
   //-----------------------------------  loop over planes to display
   for (size_t planeIndex=iQmin; planeIndex<iQmax; planeIndex++) {
      const qplane& tileplanei(tiling.planes(planeIndex));

      /////////////////////////////////////////////////////////////////////////
      //                   identify frequency rows to display                //
      /////////////////////////////////////////////////////////////////////////

      // default minimum frequency is minimum frequency of available data
      double fMin = 0;
      if (requestedFrequencyRange[0] == -dInf || requestedFrequencyRange[0]<=0){
	 fMin = tiling.planes(planeIndex).minimumFrequency;
      } else {
	 fMin = requestedFrequencyRange[0];
      }

      // default maximum frequency is maximum frequency of available data
      double fMax = 0;
      if (requestedFrequencyRange[1] == dInf || requestedFrequencyRange[1] < 0){
	 fMax = tiling.planes(planeIndex).maximumFrequency;
      } else {
	 fMax = requestedFrequencyRange[1];
      }

      // validate selected frequency range
      if ((fMin < tileplanei.minimumFrequency) || 
	  (fMax > tileplanei.maximumFrequency)) {
	 cout << " Request: [" << fMin << ", " << fMax << "]  available: ["
	      << tileplanei.minimumFrequency << ", " 
	      << tileplanei.maximumFrequency << "]." << endl;
	 error("requested frequency range exceeds available data");
      }

      /////////////////////////////////////////////////////////////////////////
      //                       initialize display matrix                     //
      /////////////////////////////////////////////////////////////////////////

      // initialize matrix of normalized energies for display
      size_t numberOfRows = tileplanei.numberOfRows;
      lcl_array<double> frequencies(numberOfRows);
      size_t iFmin = numberOfRows;
      size_t iFmax = 0;
      for (size_t irow=0; irow < numberOfRows; irow++) {
	 double frow = tileplanei.row(irow).frequency;
	 frequencies[irow] = frow;
	 if (frow >= fMin && irow <  iFmin) iFmin = irow;
	 if (frow <= fMax && irow >= iFmax) iFmax = irow+1;
      }
      if (iFmin >= iFmax) {
	 error("Couldn't find rows in range");
      }

      //////////////////////////////////////////////////////////////////////////
      //                     begin loop over frequency rows                   //
      //////////////////////////////////////////////////////////////////////////

      // loop over rows
      for (size_t rowIndex=iFmin; rowIndex < iFmax; rowIndex++) {
	 const trow& rowij(transform.planes(planeIndex).rows(rowIndex));
	 const TSeries& tsNormE(rowij.normalizedEnergies);

	 // corresponding tile normalized energies
	 Interval step = tsNormE.getTStep();
	 TSeries rowNormErgs = tsNormE.extract(tStart + step*0.5, plotDt+step);
	 convertUnits(rowNormErgs);
	 DVectD tsdv(*rowNormErgs.refDVect());

	 size_t nPoints = rowNormErgs.getNSample();
	 double t0= double(rowNormErgs.getStartTime() - referenceTime);
	 double rowDt = double(rowNormErgs.getTStep());
	 dble_vect dv(3, 0.0);
	 dv[0] = tileplanei.row(rowIndex).frequency;
	 for (size_t i=0; i<nPoints; ++i) {
	    dv[1] = t0 + double(i)*rowDt;
	    dv[2] = tsdv[i];
	    _data.fill_row(dv);
	 }
	 
      }	// end loop over rows

   }      // end loop over planes
}

//======================================  Plot a spectrogram.
void 
wfigure::wspectrogram(const qTransform& transform, const wtile& tiling, 
		      const Time& referenceTime, const dble_vect& tRange, 
		      const dble_vect& fRange, const dble_vect& Qrnge, 
		      const dble_vect& normalizedEnergyRange, 
		      long horizontalResolution)
{

   /////////////////////////////////////////////////////////////////////////////
   //                        process command line arguments                   //
   /////////////////////////////////////////////////////////////////////////////
   double dInf = 1.0 / 0.0;

   dble_vect timeRange = tRange;
   if (timeRange.empty()) {
      timeRange.resize(2);
      timeRange[0] = -dInf;
      timeRange[1] =  dInf;
   }

   dble_vect requestedFrequencyRange = fRange;
   if (requestedFrequencyRange.empty()) {
      requestedFrequencyRange.resize(2);
      requestedFrequencyRange[0] = -dInf;
      requestedFrequencyRange[1] =  dInf;
   }

   //------------------------------------  Check requested q range
   dble_vect qRange = Qrnge;
   if (qRange.empty()) {
      qRange.resize(2);
      qRange[0] = -dInf;
      qRange[1] =  dInf;
   }
   else if (qRange.size() > 2) {
      error("Q range must be scalar or two component vector [Qmin Qmax].");
   }
  
   /////////////////////////////////////////////////////////////////////////////
   //                       validate command line arguments                   //
   /////////////////////////////////////////////////////////////////////////////

   // Check for two component range vectors
   if (timeRange.size() != 2) {
      error("Time range must be two component vector [tmin tmax].");
   }

   if (requestedFrequencyRange.size() != 2 ) {
      error("Frequency range must be two component vector [fmin fmax].");
   }

   if (!normalizedEnergyRange.empty() && normalizedEnergyRange.size() != 2) {
      error("Normalized energy range must be two component vector [Zmin Zmax].");
   }

   /////////////////////////////////////////////////////////////////////////////
   //                         identify q planes to display                    //
   /////////////////////////////////////////////////////////////////////////////

   // number of planes to display
   size_t numberOfPlanes = tiling.numberOfPlanes();

   // if (non-zero Q range is requested,
   size_t iQmin = numberOfPlanes;
   size_t iQmax = 0;
   if (qRange.size() == 2) {
      // find planes within requested Q range
      for (size_t i=0; i < numberOfPlanes; ++i) {
	 double qval = tiling.planes(i).q;
	 if (qval >= qRange[0] && qval <= qRange[1]) {
	    if (i <  iQmin) iQmin = i;
	    if (i >= iQmax) iQmax = i+1;
	 }
      }
   }

   // otherwise, if (only a single Q is requested,
   else {
      // find plane with Q nearest the requested value
      iQmin = tiling.nearest_plane(qRange[0]);
      iQmax = iQmin + 1;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                          identify times to display                      //
   /////////////////////////////////////////////////////////////////////////////

   // default start time for display is start time of available data
   Time startTime=transform.planes(0).rows(0).normalizedEnergies.getStartTime();
   double startDt = double(startTime - referenceTime);
   if (timeRange[0] == -dInf) {
      timeRange[0] = startDt;
   }

   // default stop time for display is stop time of available data
   if (timeRange[1] == dInf) {
      timeRange[1] = startDt + tiling.duration();
   }

   // validate requested time range
   if ((timeRange[0] < startDt) || (timeRange[1] > startDt+tiling.duration())) {
      error("requested time range exceeds available data");
   }

   cout << "Writing data to " << _format << " format file" << endl;
   if (graphic()) {
      cout << "Plotting data" << endl;
      spectro_plot(transform, tiling, referenceTime, timeRange, 
		   requestedFrequencyRange, iQmin, iQmax,
		   normalizedEnergyRange, horizontalResolution);
   } else {
      cout << "Spewing data" << endl;
      spectro_data(transform, tiling, referenceTime, timeRange, 
		   requestedFrequencyRange, iQmin, iQmax,
		   normalizedEnergyRange, horizontalResolution);
   }
}
