/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wparameters.hh"
#include "matlab_fcs.hh"
#include <iostream>
#include <fstream>
#include <cstdlib>

using namespace wpipe;
using namespace std;

const char* trigTypeList[] = {"DOWNSELECT", "CLUSTER", "COINCIDE", "EVENTS"};
str_vect wparameters::validFileTypes(trigTypeList, trigTypeList+4);
static wparameters null_parameter_list;

//======================================  Construct a Null parameter list.
wparameters::wparameters(void) {

   //-----------------------------------  Initialize all parameters
   sampleFrequency = 0;
   maximumMismatch = 0;
   falseEventRate = 0;
   eventThreshold = 0;
   blockDuration = 0;
   conditionDuration = 0;
   errorOnStateError = false;
   highPassCutoff = -1.0;
   lowPassCutoff = -1.0;
   whiteningDuration = 0;
   transientFactor = 4.0;
   doubleWhiten = -1;
   extraBlockOverlap = 0;
   outlierFactor = 2.0;
   maximumSignificants = 0;
   maximumTriggers = 0;
   minTriggerSNR  = 0;
   durationInflation = 1.0;
   bandwidthInflation = 1.0;
   randomSeed = 0.0;

   applyClustering  = -1;
   clusterRadius    = 0.0;
   clusterDensity   = 0.0;
   clusterSingles   = 0;
   clusterThreshold = 0.0;
   writeClusters    = -1;

   coincidenceNumber  = 0;
   maximumCoincidents = 999999999;
  
   applyVeto            = 1;
   falseVetoRate        = 0;
   vetoThreshold        = 0;
   uncertaintyFactor    = 0;
   correlationFactor    = 0;
   vetoDurationFactor   = 0.5;
   vetoBandwidthFactor  = 0.5;
   maximumConsistents   = 1000;

   //-----------------------------------  OSC parameters
   oscStride = 1.0;

   //-----------------------------------  Follow-up parameters
#ifndef NO_FOLLOWUP
   bayesian = true;
   maxFollowTriggers = 5;
   writeSkymap = false;
   gzipSkymap = false;
   snrRatioCut = 50.0;

   xCoherentCheck = false;
   maxEnt = false;

   combinedStat = false;
   //combinedParameters = [];
#else
   bayesian = false;
#endif
}

//======================================  Construct parameter list from file.
wparameters::wparameters(const std::string& parFile, int debugLevel) {

   //-----------------------------------  Get default parameter list file.
   string parameterFile = parFile;
   if (parameterFile.empty()) parameterFile = "./parameters.txt";

   // validate command line arguments
   if (!exist(parameterFile, "file")) {
      error(string("could not find parameter file: ") + parameterFile);
   }

   wparameters default_pars;
   default_pars.set_defaults();

   // initialize parameters to the "null" values
   *this = null_parameter_list;

   // open parameter file for reading
   ifstream parameterFileID(parameterFile.c_str());

   try  {
      read_params(parameterFileID, debugLevel);
      merge_defaults(default_pars);
   } 
   catch (std::exception& e) {
      parameterFileID.close();
      throw;
   }
   // close parameter file
   parameterFileID.close();

   // validate
   validate();
}

//======================================  Build network string from channel list
std::string
wparameters::buildNetworkString(const str_vect& chan) {
   string net;
   size_t N = chan.size();
   for (size_t i=0; i<N; ++i) {
      string site = chan[i].substr(0,1);
      if (net.empty()) {
	 net  = site;
      } else if (net.find(site) == string::npos) {
	 net += site;
      }
   }
   return net;
}

//====================================== Read a single line of a parameter file
std::string
wparameters::read_line(std::istream& parameterFileID) {
   std::string parameterLine;
   bool continuation = true;
   while (continuation && !parameterFileID.fail()) {
      string line;
      getline(parameterFileID, line);
      continuation = false;
      if (!line.empty()) {
	 if (line[line.size()-1] == '\\') {
	    line.erase(line.size()-1,1);
	    continuation=true;
	 }
	 deblank(line);
	 if (!parameterLine.empty()) parameterLine += " ";
	 parameterLine += line;
      }
   }

   if (!parameterLine.empty()) {
      // remove any comments
      string::size_type commentIndex = parameterLine.find_first_of("#%");
      string::size_type altIndex     = parameterLine.find("//");
      if (altIndex < commentIndex) commentIndex = altIndex;
      if (commentIndex != string::npos) parameterLine.erase(commentIndex);

      // remove leading and trailing blanks
      parameterLine = deblank(parameterLine);
   }
   return parameterLine;
}

//======================================  Construct parameter list from file.
int
wparameters::read_params(std::istream& parameterFileID, int debugLevel) {
   // begin loop over parameter file
   int rc = 1;
   while (!parameterFileID.eof()) {
      string parameterLine = read_line(parameterFileID);
      // if (empty line, skip to the next line
      if (parameterLine.empty()) continue;

      string::size_type braceIndex = parameterLine.find("{");
      if (braceIndex == parameterLine.size()-1) {
	 parameterLine.erase(braceIndex);
	 groupName = deblank(parameterLine);
	 continue;
      }

      if (parameterLine == "}") {
	 rc = 0;
	 break;
      }

      // locate field separator
      string::size_type colonIndex = parameterLine.find(":");

      // if (field separator not located, report syntax error
      if (colonIndex == string::npos) {
	 error(string("syntax error processing parameter file: ")+parameterLine);
      }

      // parse parameter line
      string parameterName  = parameterLine.substr(0, colonIndex);
      string parameterValue = parameterLine.substr(colonIndex + 1);
      parameterName = deblank(parameterName);
      parameterValue = deblank(parameterValue);

      // report parameter settings
      wlog(debugLevel, 3, parameterName + ": " + parameterValue);

      // assign parameters based on name
      // required parameters
      if (parameterName == "channelNames") {
	 channelNames = eval_str(parameterValue);
      } else if (parameterName ==  "frameTypes") {
	 frameTypes = eval_str(parameterValue);
      }

      //--------------------------------  Optional parameters
      else if (parameterName ==  "analysisMode") {
	 analysisMode = unquote(parameterValue);
      } else if (parameterName ==  "sampleFrequency") {
	 sampleFrequency = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "qRange") {
	 qRange = eval(parameterValue);
      } else if (parameterName ==  "frequencyRange") {
	 frequencyRange = eval(parameterValue);
      } else if (parameterName ==  "maximumMismatch") {
	 maximumMismatch = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "falseEventRate") {
	 falseEventRate = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "eventThreshold") {
	 eventThreshold = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "blockDuration") {
	 blockDuration = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "conditionDuration") {
	 conditionDuration = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "dataFactors") {
	 dataFactors = eval(parameterValue);
      } else if (parameterName ==  "timeShifts") {
	 timeShifts = eval(parameterValue);
      } else if (parameterName ==  "stateNames") {
	 stateNames = eval_str(parameterValue);
      } else if (parameterName ==  "stateTypes") {
	 stateTypes = eval_str(parameterValue);
      } else if (parameterName ==  "stateMasks") {
	 stateMasks = eval(parameterValue);
      } else if (parameterName ==  "errorOnStateError") {
	 errorOnStateError = strtol(parameterValue.c_str(), 0, 0);
      }

      //--------------------------------  OSC condition parameters
      else if (parameterName == "oscFile") {
	 oscFile = unquote(parameterValue);
      }
      else if (parameterName == "oscFrameType") {
	 oscFrameType = unquote(parameterValue);
      }
      else if (parameterName == "oscConditions") {
	 oscConditions = eval_str(parameterValue);
      }
      else if (parameterName == "oscStride") {
	 oscStride = strtod(parameterValue.c_str(), 0);
      }
      
      //--------------------------------  Injection parameters
      else if (parameterName ==  "injectionNames") {
	 injectionNames = eval_str(parameterValue);
      } else if (parameterName ==  "injectionTypes") {
	 injectionTypes = eval_str(parameterValue);
      } else if (parameterName ==  "injectionFactors") {
	 injectionFactors = eval(parameterValue);
      } else if (parameterName ==  "injectionTimeShifts") {
	 injectionTimeShifts = eval(parameterValue);
      } else if (parameterName ==  "highPassCutoff") {
	 highPassCutoff = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "lowPassCutoff") {
	 lowPassCutoff = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "whiteningDuration") {
	 whiteningDuration = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "transientFactor") {
	 transientFactor = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "doubleWhiten") {
	 doubleWhiten = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "extraBlockOverlap") {
	 extraBlockOverlap = strtod(parameterValue.c_str(), 0);
      } 
      else if (parameterName ==  "outlierFactor") {
	 outlierFactor = strtod(parameterValue.c_str(), 0);
      }
      //----------------------------------  Input maximum significants
      //  Note: large ints are read in as doubles to allow exp notation (e.g. 1e5)
      else if (parameterName ==  "maximumSignificants") {
	 maximumSignificants = int(strtod(parameterValue.c_str(), 0));
      }
      else if (parameterName ==  "maximumTriggers") {
	 maximumTriggers = int(strtod(parameterValue.c_str(), 0));
      } 
      else if (parameterName ==  "minTriggerSNR") {
         minTriggerSNR = strtod(parameterValue.c_str(), 0);
      }
      //-----------------------------------
      else if (parameterName ==  "durationInflation") {
	 durationInflation = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "bandwidthInflation") {
	 bandwidthInflation = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "triggerFields") {
	 triggerFields = eval_str(parameterValue);
      } else if (parameterName ==  "triggerPrefix") {
	 triggerFiles = eval_str(parameterValue);
      } else if (parameterName ==  "triggerFiles") {
	 triggerTypes = eval_str(parameterValue);
      } else if (parameterName ==  "triggerFormat") {
	 triggerFormat = unquote(parameterValue);
      } else if (parameterName ==  "randomSeed") {
	 randomSeed = strtod(parameterValue.c_str(), 0);
      }

      //--------------------------------  clustering parameters
      else if (parameterName ==  "applyClustering") {
	 applyClustering = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "clusterMethod") {
	 clusterMethod = tolower(unquote(parameterValue));
      } else if (parameterName ==  "clusterRadius") {
	 clusterRadius = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "clusterDensity") {
	 clusterDensity = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "clusterSingles") {
	 clusterSingles = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "clusterLinkage") {
	 clusterLinkage = tolower(unquote(parameterValue));
      } else if (parameterName ==  "clusterCriterion") {
	 clusterCriterion = tolower(unquote(parameterValue));
      } else if (parameterName ==  "clusterThreshold") {
	 clusterThreshold = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "distanceMetric") {
	 distanceMetric = unquote(parameterValue);
      } else if (parameterName ==  "writeClusters") {
	 writeClusters = strtol(parameterValue.c_str(), 0, 0);
      }

      //--------------------------------  "independent" mode parameters
      else if (parameterName ==  "coincidenceNumber") {
	 coincidenceNumber = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "maximumCoincidents") {
	 maximumCoincidents = strtol(parameterValue.c_str(), 0, 0);
      }

#ifndef NO_COHERENT
      // "coherent" mode parameters
      else if (parameterName ==  "skyPosition") {
	 skyPosition = eval(parameterValue);
      } else if (parameterName ==  "skyCoordinateSystem") {
	 skyCoordinateSystem = unquote(parameterValue);
      }
#endif

      //--------------------------------  "coherent" mode parameters used in 
      //                                  function calls.
      else if (parameterName ==  "applyVeto") {
	 applyVeto = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "falseVetoRate") {
	 falseVetoRate = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "vetoThreshold") {
	 vetoThreshold = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "uncertaintyFactor") {
	 uncertaintyFactor = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "correlationFactor") {
	 correlationFactor = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "vetoDurationFactor") {
	 vetoDurationFactor = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "vetoBandwidthFactor") {
	 vetoBandwidthFactor = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "maximumConsistents") {
	 maximumConsistents = strtol(parameterValue.c_str(), 0, 0);
      }

      //--------------------------------  Bayesian mode parameters
#ifdef NO_FOLLOWUP
      else if (parameterName ==  "bayesian") {
	 cerr << "*** Warning *** Bayesian followup is not implemented" << endl;
      }
#else
      else if (parameterName ==  "bayesian") {
	 bayesian = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "maxFollowTriggers") {
	 maxFollowTriggers = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "snrRatioCut") {
	 snrRatioCut = strtod(parameterValue.c_str(), 0);
      } else if (parameterName ==  "writeSkymap") {
	 writeSkymap = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "gzipSkymap") {
	 gzipSkymap = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "prcInjectionFile") {
	 prcInjectionFile = unquote(parameterValue);
      }

      // xCoherentCheck
      else if (parameterName ==  "xCoherentCheck") {
	 xCoherentCheck = strtol(parameterValue.c_str(), 0, 0);
      }

      // MaxEnt waveform recovery
      else if (parameterName ==  "maxEnt") {
	 maxEnt = strtol(parameterValue.c_str(), 0, 0);
      }

      // Combined statistic
      else if (parameterName ==  "combinedStat") {
	 combinedStat = strtol(parameterValue.c_str(), 0, 0);
      } else if (parameterName ==  "combinedParameters") {
	 combinedParameters = eval(parameterValue);
      }
#endif

      // handle unknown parameters
      else if (debugLevel >= 1) {
	 cout <<  "unknown parameter " << parameterName << ".  skipping." << endl;
      }  // end assign parameters based on name
    
   } // end loop over parameter file entries

   //------------------------------------  Build site list.
   sites = buildNetworkString(channelNames);
   return rc;
}

//======================================  Merge default values
void
wparameters::merge_defaults(const wparameters& d) {
   size_t nChan = numberOfChannels();
   // default values
   if (analysisMode.empty())   analysisMode    = d.analysisMode;
   if (sampleFrequency == 0)   sampleFrequency = d.sampleFrequency;
   if (qRange.empty())         qRange          = d.qRange;
   if (frequencyRange.empty()) frequencyRange  = d.frequencyRange;
   if (maximumMismatch == 0)   maximumMismatch = d.maximumMismatch;
   if (falseEventRate == 0)    falseEventRate  = d.falseEventRate;
   if (eventThreshold == 0)    eventThreshold  = d.eventThreshold;
   if (blockDuration == 0)     blockDuration   = d.blockDuration;
   if (conditionDuration == 0) conditionDuration = blockDuration;
   if (!maximumSignificants)   maximumSignificants = d.maximumSignificants;
   if (!maximumTriggers)       maximumTriggers = d.maximumTriggers;
   if (minTriggerSNR == 0)     minTriggerSNR = d.minTriggerSNR;
   if (dataFactors.size() < nChan)    dataFactors.resize(nChan, 1.0);
   if (timeShifts.size() < nChan)     timeShifts.resize(nChan, 0.0);
   if (injectionNames.size() < nChan) injectionNames.resize(nChan);
   if (injectionTypes.size() < nChan) injectionTypes.resize(nChan, "NONE");
   if (injectionFactors.size() < nChan) injectionFactors.resize(nChan, 0.0);
   if (injectionTimeShifts.size() < nChan) injectionTimeShifts.resize(nChan, 0.0);
   if (highPassCutoff < 0)     highPassCutoff  = d.highPassCutoff;
   if (lowPassCutoff < 0)      lowPassCutoff   = d.lowPassCutoff;
   if (doubleWhiten < 0)       doubleWhiten    = d.doubleWhiten;
   if (triggerFormat.empty())  triggerFormat   = d.triggerFormat;

   //if (randomSeed == 0) {
   //  randomSeed = sum(1e6 * clock);
   //}

   //-----------------------------------  Defaults for clustering
   if (applyClustering < 0)      applyClustering  = d.applyClustering;
   if (writeClusters < 0)        writeClusters    = d.writeClusters;
   if (clusterMethod.empty())    clusterMethod    = d.clusterMethod;
   if (clusterRadius  == 0)      clusterRadius    = d.clusterRadius;
   if (clusterDensity == 0)      clusterDensity   = d.clusterDensity;
   if (clusterSingles == 0)      clusterSingles   = d.clusterSingles;
   if (clusterLinkage.empty())   clusterLinkage   = d.clusterLinkage;
   if (clusterCriterion.empty()) clusterCriterion = d.clusterCriterion;
   if (clusterThreshold == 0.0)  clusterThreshold = d.clusterThreshold;
   if (distanceMetric.empty())   distanceMetric   = d.distanceMetric;

#ifndef NO_COHERENT
   // defaults for coherent analysis
   if (skyCoordinateSystem.empty()) skyCoordinateSystem = d.skyCoordinateSystem;
   if (combinedParameters.empty())  combinedParameters  = d.combinedParameters;
#endif
  
   // default trigger fields
   if (triggerFields.empty()) {
      triggerFields = d.triggerFields;

      if (analysisMode == "coherent") {
	 triggerFields.push_back("incoherentEnergy");
      }

      if (applyClustering && writeClusters) {
	 triggerFields.push_back("clusterSize");
	 triggerFields.push_back("clusterNormalizedEnergy");
      }
   }

   // default trigger files
   if (triggerFiles.empty()) triggerTypes = d.triggerTypes;
   if (triggerTypes.empty() && !triggerFiles.empty()) {
      triggerTypes.swap(triggerFiles);
   }
   if (triggerTypes.empty()) {
      if (applyClustering) {
	 triggerTypes.push_back("CLUSTER");
      } else {
	 triggerTypes.push_back("DOWNSELECT");
      }
   }

   for (size_t i=0; i<nChan; ++i) {
      timeShifts[i] = long(timeShifts[i]*sampleFrequency + 0.5) / sampleFrequency;
   }
}

//======================================  Merge default values
void
wparameters::set_defaults(void) {
 
   // default values
   analysisMode    ="independent";
   sampleFrequency = 4096;

   doubleWhiten = 1;

   double qDefault[2] = {sqrt(11), 100.0};
   qRange = dble_vect(qDefault, qDefault+2);

   double fDefault[2] = {48.0, 1.0/0.0};
   frequencyRange = dble_vect(fDefault, fDefault+2);

   maximumMismatch = 0.2;
   falseEventRate  = 1;
   blockDuration   = 64;
   conditionDuration = blockDuration;
   maximumSignificants = 100000;
   maximumTriggers = 1000;
   dataFactors     = dble_vect(1, 1.0);
   timeShifts      = dble_vect(1, 0.0);
   injectionNames.resize(1);
   injectionTypes.resize(1, "NONE");
   injectionFactors.resize(1, 0.0);
   injectionTimeShifts = dble_vect(1, 0.0);
   minTriggerSNR = 0;
   triggerFormat   = "xml";

   // defaults for clustering
   applyClustering  = 0;
   writeClusters    = 0;
   clusterMethod    = "density";
   clusterRadius    = 4.0;
   clusterDensity   = 3.0;
   clusterSingles   = 1;
   clusterLinkage   = "single";
   clusterCriterion = "distance";
   clusterThreshold = 4.0;
   distanceMetric   = "integratedMismatch";

#ifndef NO_COHERENT
   // defaults for coherent analysis
   skyCoordinateSystem = "equatorial";

   double defCPar[2] = {1.1, 1.0};
   combinedParameters = dble_vect(defCPar, defCPar+2);
#endif
  
   // default trigger fields
   const char* defFields[5] = {"time", "frequency", "duration", 
			       "bandwidth", "normalizedEnergy"};
   triggerFields = str_vect(defFields, defFields+5);
}

//======================================  Validate parameters
void
wparameters::validate(void ) const {

   size_t nChan = numberOfChannels();
   if (!nChan) {
      error("channelNames not specified");
   }
   
   if (analysisMode == "coherent") {
      if (nChan < 2) {
	 error("coherent analysis mode requires at least two channels");
      }
   } else if (analysisMode != "independent") {
      error(string("unknown analysis mode ") + analysisMode);
   }

   // validate number of frame types
   if (frameTypes.size() != nChan) {
      error("number of frameTypes and channelNames are inconsistent");
   }

   // validate number of data factors
   if (dataFactors.size() != nChan && !dataFactors.empty()) {
      error("number of dataFactors and channelNames are inconsistent");
   }

   // validate number of time shifts
   if (timeShifts.size() != nChan && !timeShifts.empty()) {
      error("number of timeShifts and channelNames are inconsistent");
   }

   // validate number of state names
   size_t nState = stateNames.size();
   if (nState > nChan) {
      error("number of stateNames and channelNames are inconsistent");
   }

   // validate number of state types
   if (stateTypes.size() != nState) {
      error("number of stateTypes and stateNames are inconsistent");
   }

   // validate number of state masks
   if (stateMasks.size() != nState) {
      error("number of stateMasks and stateNames are inconsistent");
   }

   // validate number of injection names
   size_t nInject = injectionNames.size();
   if (nInject > nChan) {
      error("number of injectionNames and channelNames are inconsistent");
   }

   // validate number of injection types
   if (injectionTypes.size() != nInject) {
      error("number of injectionTypes and injections are inconsistent");
   }

   // validate number of injection factors
   if (injectionFactors.size() != nInject && !injectionFactors.empty()) {
      error("number of injectionFactors and injections are inconsistent");
   }

   //-----------------------------------  Warn of incomplete injection spec.
   for (size_t i=0; i<nInject; i++) {
      if (!injectionNames[i].empty() && injectionNames[i] != "NONE") {
	 if (injectionTypes[i].empty() || injectionTypes[i] == "NONE") 
	    error("Injection channel specified with invalid frame type");
	 if (injectionFactors[i] == 0.0) {
	    cout << "warning: zero factor on valid injection channel: "
		 << injectionNames[i] << endl;
	 }
      }
   }

   //-----------------------------------  Validate injection time shift count
   if (injectionTimeShifts.size() != nInject && !injectionTimeShifts.empty()) {
      error("number of injectionTimeShifts and injections are inconsistent");
   }

   // validate block duration
   if (blockDuration <= 0) {
      error("block duration must be non-zero and positive");
   } else if (blockDuration != double(int(blockDuration))) {
      error("block duration must be an integer");
   }

   if (applyClustering < 0) {
      error("applyClustering parameter not set");
   }

   if (writeClusters < 0) {
      error("writeClusters parameter not set");
   }

   if (triggerFormat != "txt" && triggerFormat != "xml") {
      error(string("Unrecognized trigger format: ") + triggerFormat);
   }


#ifndef NO_TARGETED
   // validate sky coordinate system and position
   if (skyCoordinateSystem != "equatorial" &&
       skyCoordinateSystem != "geocentric" &&
       skyCoordinateSystem != "galactic") {
      error("unknown sky coordinate system");
   }
#endif

#ifndef NO_FOLLOWUP
   // check prc injection file location
   if (bayesian) {
      if (nChan < 2) {
	 error("bayesian followup requires at least two channels");
      }
      if (coincidenceNumber < 2) {
	 error("bayesian followup requires coincidenceNumber > 1");
      }
      if (!prcInjectionFile.empty() && !exist(prcInjectionFile,"file") ) {
	 error(string("PRC injection file ")+prcInjectionFile+" not found.");
      }
   }
#endif
}

//======================================  Inline function to extend a string
//
//   Return a string with at least the specified length. If the initial text
//   is longer than the specified length, the returned string will be identical
//   to the initiali string.
inline std::string
straw(const std::string& in, size_t wid) {
   string col(in);
   if (wid > col.size()) col.resize(wid, ' ');
   return col;
}

//======================================  Display parameter values
static std::ostream&
disp_bvalue(std::ostream& out, const std::string& name, bool b) {
   out << straw(name + ":", 32) << boolstr_tf(b) << endl;
   return out;
}

static std::ostream&
disp_darray(std::ostream& out, const std::string& name, const dble_vect& d) {
   out << straw(name + ":", 32) << "[";
   for (size_t i=0; i<d.size(); i++) {
      if (i) out << " ";
      out << d[i];
   }
   out << "]" << endl;
   return out;
}

static std::ostream&
disp_dvalue(std::ostream& out, const std::string& name, double d) {
   out << straw(name + ":", 32) << d << endl;
   return out;
}

static std::ostream&
disp_sarray(std::ostream& out, const std::string& name, const str_vect& s) {
   out << straw(name + ":", 32) << "{";
   for (size_t i=0; i<s.size(); i++) {
      if (i) out << " ";
      out << "'" << s[i] << "'";
   }
   out << "}" << endl;
   return out;
}

static std::ostream&
disp_svalue(std::ostream& out, const std::string& name, const std::string& s) {
   out << straw(name + ":", 32) << "'" << s << "'" << endl;
   return out;
}

void
wparameters::display(std::ostream& out) const {
   //-----------------------------------  Input data.
   disp_sarray(out, "channelNames",         channelNames);
   disp_sarray(out, "frameTypes",           frameTypes);
   
   //-----------------------------------  Control parameters.
   disp_svalue(out, "analysisMode",        analysisMode);
   disp_dvalue(out, "sampleFrequency",     sampleFrequency);
   disp_darray(out, "qRange",              qRange);
   disp_darray(out, "frequencyRange",      frequencyRange);
   disp_dvalue(out, "maximumMismatch",     maximumMismatch);
   disp_dvalue(out, "falseEventRate",      falseEventRate);
   disp_dvalue(out, "eventThreshold",      eventThreshold);
   disp_dvalue(out, "blockDuration",       blockDuration);
   disp_dvalue(out, "conditionDuration",   conditionDuration);
   disp_darray(out, "dataFactors",         dataFactors);
   disp_darray(out, "timeShifts",          timeShifts);

   //-----------------------------------  State information parameters
   disp_sarray(out, "stateNames",          stateNames);
   disp_sarray(out, "stateTypes",          stateTypes);
   disp_darray(out, "stateMasks",          stateMasks);
   disp_bvalue(out, "errorOnStateError",   errorOnStateError);

   //-----------------------------------  Operating state conditions
   disp_svalue(out, "oscFile",             oscFile);
   disp_svalue(out, "oscFrameType",        oscFrameType);
   disp_sarray(out, "oscConditions",       oscConditions);
   disp_dvalue(out, "oscStride",           oscStride);
   
   //-----------------------------------  Injection parameters
   disp_sarray(out, "injectionNames",      injectionNames);
   disp_sarray(out, "injectionTypes",      injectionTypes);
   disp_darray(out, "injectionFactors",    injectionFactors);
   disp_darray(out, "injectionTimeShifts", injectionTimeShifts);
   disp_dvalue(out, "highPassCutoff",      highPassCutoff);
   disp_dvalue(out, "lowPassCutoff",       lowPassCutoff);
   disp_dvalue(out, "whiteningDuration",   whiteningDuration);
   disp_dvalue(out, "transientFactor",     transientFactor);
   disp_bvalue(out, "doubleWhiten",        doubleWhiten);
   disp_dvalue(out, "extraBlockOverlap",   extraBlockOverlap);
   disp_dvalue(out, "outlierFactor",       outlierFactor);
   disp_dvalue(out, "falseVetoRate",       falseVetoRate);
   disp_dvalue(out, "vetoThreshold",       vetoThreshold);
   disp_dvalue(out, "uncertaintyFactor",   uncertaintyFactor);
   disp_dvalue(out, "correlationFactor",   correlationFactor);
   disp_dvalue(out, "maximumSignificants", maximumSignificants);
   disp_dvalue(out, "maximumTriggers",     maximumTriggers);
   disp_dvalue(out, "durationInflation",   durationInflation);
   disp_dvalue(out, "bandwidthInflation",  bandwidthInflation);
   disp_dvalue(out, "minTriggerSNR",       minTriggerSNR);
   disp_sarray(out, "triggerFiles",        triggerTypes);
   disp_sarray(out, "triggerPrefix",       triggerFiles);
   disp_sarray(out, "triggerFields",       triggerFields);
   disp_svalue(out, "triggerFormat",       triggerFormat);
   disp_dvalue(out, "randomSeed",          randomSeed);
   disp_dvalue(out, "numberOfChannels",    numberOfChannels());
   disp_dvalue(out, "numberOfSites",       numberOfSites());
   disp_dvalue(out, "injectionChannels",   injectionChannels());


   //-----------------------------------  Clustering parameters.
   disp_dvalue(out, "applyClustering",     applyClustering);
   disp_svalue(out, "clusterMethod",       clusterMethod);
   disp_dvalue(out, "clusterRadius",       clusterRadius);
   disp_dvalue(out, "clusterDensity",      clusterDensity);
   disp_dvalue(out, "clusterSingles",      clusterSingles);
   disp_svalue(out, "clusterLinkage",      clusterLinkage);
   disp_svalue(out, "clusterCriterion",    clusterCriterion);
   disp_dvalue(out, "clusterThreshold",    clusterThreshold);
   disp_svalue(out, "distanceMetric",      distanceMetric);
   disp_dvalue(out, "writeClusters",       writeClusters);

   disp_dvalue(out, "coincidenceNumber",   coincidenceNumber);
   disp_dvalue(out, "maximumCoincidents",  maximumCoincidents);

#ifndef NO_TARGETED
   //-----------------------------------  Optional fields for targeted searches:
   disp_darray(out, "skyPosition",         skyPosition);
   disp_svalue(out, "skyCoordinateSystem", skyCoordinateSystem);
#endif // !defined (NO_TARGETED)

   //-----------------------------------  Fields for 'coherent' analysis
   disp_bvalue(out, "applyVeto",            applyVeto);
   disp_dvalue(out, "vetoDurationFactor",   vetoDurationFactor);
   disp_dvalue(out, "vetoBandwidthFactor",  vetoBandwidthFactor);
   disp_dvalue(out, "maximumConsistents",   maximumConsistents);

   // 'bayesian' position reconstruction:
   disp_bvalue(out, "bayesian",             bayesian);
#ifndef NO_FOLLOWUP
   //-----------------------------------  Parameters for followup analyses.
   disp_dvalue(out, "maxFollowTriggers",    maxFollowTriggers);
   disp_dvalue(out, "snrRatioCut",          snrRatioCut);
   disp_bvalue(out, "writeSkymap",          writeSkymap);
   disp_bvalue(out, "gzipSkymap",           gzipSkymap);
   disp_svalue(out, "prcInjectionFile",     prcInjectionFile);

   // xCoherentCheck null energy statistic:
   disp_bvalue(out, "xCoherentCheck",       xCoherentCheck);

   // MaxEnt waveform reconstruction:
   disp_bvalue(out, "maxEnt",               maxEnt);

   // Combined event statistic:
   disp_bvalue(out, "combinedStat",         combinedStat);
   disp_darray(out, "combinedParameters",   combinedParameters);
#endif // !defined (NO_FOLLOWUP)

   //-----------------------------------  Extra first character of each channel
   disp_svalue(out, "sites",                sites);
}

//======================================  Count injections.
bool
wparameters::inject(size_t inx) const {
   return injectionNames[inx] != "NONE" &&
      injectionTypes[inx] != "NONE" &&
      injectionFactors[inx] != 0.0;
}

//======================================  Count injections.
size_t
wparameters::injectionChannels(void) const {
   size_t nInj = 0;
   size_t N = numberOfChannels();
   for (size_t i=0; i<N; i++) {
      if (inject(i)) nInj++;
   }
   return nInj;
}
