/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "woutput.hh"
#include "scandir.hh"

#include <sys/stat.h>
#include <sstream>

using namespace wpipe;
using namespace std;

//=======================================  Parse a time string;
static void 
parse_time(const std::string& time, Time& t0, int& dt) {
   const char* ptr = 0;
   t0 = Time(strtol(time.c_str(), const_cast<char**>(&ptr), 0)); 
   dt = -1;
   if (*ptr == '-') dt = strtol(ptr+1, const_cast<char**>(&ptr), 0);
}

//======================================  Output type constructor
wouttype::wouttype(void) {
}

//======================================  Output type destructor
wouttype::~wouttype(void) {
}

//======================================  Output type constructor
wouttype::wouttype(const str_vect& channels, const Time& time, int dt,
		   const std::string& prefix, const std::string& dir, 
		   const std::string& fmt, const std::string& type)
   : pattern(dir, prefix, dt, fmt), trig_type(type)
{
   addChannels(channels, time);
}

void
wouttype::addChannels(const str_vect& channels, const string& time) {
   Time t0;
   int dt;
   parse_time(time, t0, dt);
   addChannels(channels, t0, dt);
}

void
wouttype::addChannels(const str_vect& channels, const Time& time, int dt) {
   size_t N = channels.size();
   if (N == 1 && channels[0] == "*") {
      channelNames.push_back(channels[0]);
      _paths.push_back(pattern.file_path(time, dt));
   }
   else {
      for (size_t i=0; i<N; i++) {
	 string chan = channels[i];
	 channelNames.push_back(chan);
	 _paths.push_back(gen_path(chan, time, dt));
      }
   }
}

//======================================  Display output type
std::ostream&
wouttype::display(std::ostream& out) const {
   out << "Output directory:     " << pattern.directory() << endl;
   out << "Format:               " << pattern.extension() << endl;
   out << "Trigger types:        " << trig_type << endl;
   out << "Channel List:         " << endl;
   for (size_t i=0; i<channelNames.size(); ++i) {
      out << channelNames[i] << "   " << _paths[i] << endl;
   }
   return out;
}

//======================================  Find channel number
size_t 
wouttype::find(const std::string& chan) const {
   size_t N = channelNames.size();
   if (N == 1 && channelNames[0] == "*") return 0;

   for (size_t i=0; i<N; i++) {
      if (chan == channelNames[i]) return i;
   }
   return N;
}

//======================================  Generate a path string.
std::string
wouttype::gen_path(const std::string& chan, const Time& time, int dt) const {
   return pattern.file_path(time, dt);
}

//======================================  Update all paths
void
wouttype::update(const Time& time, int dt) {
   pattern.make_dir(pattern.dir_name(time));
   size_t N = channelNames.size();
   if (N == 1 && channelNames[0] == "*") {
      _paths[0] = pattern.file_path(time, dt);
   }
   else {
      for (size_t i=0; i<N; i++) {
	 _paths[i] = gen_path(channelNames[i], time, dt);
      }
   }
}

//======================================  Empty woutput constructor
woutput::woutput(void) 
   : site_pfx("XX")
{
}

//======================================  woutput destructor
woutput::~woutput(void) {
}

//======================================  woutput constructor
woutput::woutput(const str_vect& chans, const string& time, const string& dir, 
		 const string& fmt, const str_vect& otyp, const str_vect& pfx) 
   : site_pfx("XX")
{
   addMany(chans, time,  dir, fmt, otyp, pfx);
}

//======================================  Add entries for specified types
void
woutput::addMany(const str_vect& chans, const string& time, const string& dir, 
		 const string& fmt, const str_vect& otyp, const str_vect& pfx) {
   Time t0;
   int dt;
   parse_time(time, t0, dt);
   if (otyp.empty()) {
      addtype(chans, t0, dt, dir, fmt, "DOWNSELECT", "");
      addtype(chans, t0, dt, dir, fmt, "CLUSTER", "");
      addtype(chans, t0, dt, dir, fmt, "COINCIDE", "");
   }

   else {
      size_t N=otyp.size();
      for (size_t i=0; i<N; i++) {
	 string prefix;
	 if (i < pfx.size()) prefix = pfx[i];
	 addtype(chans, t0, dt, dir, fmt, otyp[i], prefix);
      }
   }
}

//======================================  Add a type entry
void 
woutput::addtype(const str_vect& chans, const Time& time, int dt,
		 const string& dir, const string& fmt, 
		 const string& type, const string& prefix) {
   string pfx = prefix;
   if (prefix.empty()) {
      if (chans[0] == "*") {
	 pfx = site_pfx;
      } else {
	 pfx = chans[0].substr(0,2);
      }
      pfx += "-OMEGA_TRIGGERS_";
      pfx += type;
   }
   omap_iter itr = out_map.find(type);
   if (itr == out_map.end()) {
      out_map.insert(omap_value(type, wouttype(chans, time, dt, pfx, 
					       dir, fmt, type)));
   } else {
      itr->second.addChannels(chans, time);
   }
}

//======================================  Display the output file map.
std::ostream&
woutput::display(std::ostream& out) const {
   out << "Display of output map" << endl;
   out << "---------------------" << endl;
   for (const_omap_iter i=out_map.begin(); i != out_map.end(); i++) {
      out << "File definitions for "<< i->first << " data type." << endl;
      i->second.display(out);
      out << endl;
   }
   return out;
}

//======================================= Update time stamps in all files
void
woutput::update(const string& time, const str_vect& types) {
   Time t0;
   int dt;
   parse_time(time, t0, dt);
   update(types, t0, dt);
}

//======================================= Update time stamps in all files
void
woutput::update(const str_vect& types, const Time& t0, int dt) {
  
   if (types.empty()) {
      for (omap_iter itr=out_map.begin(); itr!=out_map.end(); ++itr) {
	 itr->second.update(t0, dt);
      }
   }

   else {
      size_t N=types.size();
      for (size_t i=0; i<N; i++) {
	 string otype = types[i];
	 omap_iter itr = out_map.find(otype);
	 if (itr == out_map.end()) continue;
	 itr->second.update(t0, dt);
      }
   }
}
