/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "weventlist.hh"
#include "wtile.hh"
#include "wtransform.hh"
#include "woutput.hh"
#include "matlab_fcs.hh"

#include "DVecType.hh"
#include "lcl_array.hh"
#include "TrigClient.hh"
#include "TrigWriter.hh"
#include "SBTrigger.hh"

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include <algorithm>

//======================================  Define utility stuff in wpipe::
namespace wpipe {
   const size_t null_cluster_id = ~0;

   //===================================  Utility class to measure t-F overlap
   class tfbox {
   public:
      tfbox(void);
      tfbox(const wevent& evt, double tInflation, double fInflation);
      bool overlap(const tfbox& box) const;
      void merge(const tfbox& box);
      static const double _dt;       // Omicron slop constant
      static const bool  _flextile;  // Omicron flexible tiles
   private:
      double _tmin;
      double _tmax;
      double _fmin;
      double _fmax;
   };

   const double tfbox::_dt = 0.0;        // Omicron slop constant
   const bool   tfbox::_flextile = false; // Omicron flexible tiles


   //====================================  tfBox constructor
   tfbox::tfbox(const wevent& evt, double tInflation, double fInflation) {
      double delta_t = tInflation * evt.duration;
      _tmin = evt.t_offset - 0.5*delta_t;
      _tmax = _tmin + delta_t;

      double delta_f = fInflation * evt.bandwidth;
      _fmin = evt.frequency - 0.5*delta_f;
      _fmax = _fmin + delta_f;
   }

   //====================================  tfbox test overlap
   bool 
   tfbox::overlap(const tfbox& box) const {
      return _tmin < (box._tmax + _dt) && box._tmin < (_tmax + _dt) && 
	 _fmin < box._fmax && box._fmin < _fmax;
   }

   //====================================  tfbox test overlap
   void
   tfbox::merge(const tfbox& box) {
       if (box._tmin < _tmin) _tmin = box._tmin;
       if (box._tmax > _tmax) _tmax = box._tmax;
       if (box._fmin < _fmin) _fmin = box._fmin;
       if (box._fmax > _fmax) _fmax = box._fmax;
   }

}  // namespace wpipe.

using namespace wpipe;
using namespace std;

//======================================  wevent methods.
wevent::wevent(void)
   : t_offset(0), frequency(0), q(0), duration(0), bandwidth(0),
     normalizedEnergy(0), amplitude(0), incoherentEnergy(0),
     cluster_size(0), cluster_id(null_cluster_id)
{}

//======================================  Display contents of an event
void
wevent::dump(std::ostream& out) const {
   out << "    event time offset: " << t_offset << endl;
   out << "    Mean frequency:    " << frequency << endl;
   out << "    Q value:           " << q << endl;
   out << "    Duration:          " << duration << endl;
   out << "    Bandwidth:         " << bandwidth << endl;
   out << "    Normalized energy: " << normalizedEnergy << endl;
   out << "    Amplitude:         " << amplitude << endl;
   out << "    cluster size:      " << cluster_size << endl;
   out << "    cluster ID:        ";
   if (cluster_id == null_cluster_id) out << "none" << endl;
   else                               out << cluster_id << endl;
};

//======================================  weventlist methods.
weventlist::weventlist(const std::string& event_class)
   : _list_class(event_class), _overflowFlag(0), _sorted(false)
{}

//======================================  Add an event to the list
void
weventlist::addEvent(const wevent& evt) {
   if (_events.empty()) {
      _sorted = true;
   } else if (_sorted) {
      _sorted = !(_events.back() < evt);
   }
   _events.push_back(evt);
}

//======================================  Compact display of all events.
void 
weventlist::display(std::ostream& out) const {
   out << "Events of class " << _list_class << " for channel " 
       << _channelName << " t0 " << _refTime.totalS() << endl;
   size_t N = _events.size();
   for (size_t i=0; i<N; i+=8) {
      size_t M = (i+8<N) ? i+8 : N;
      out << "time:";
      for (size_t j=i; j<M; j++) out << " " << setw(8) << right 
				     << _events[j].t_offset;
      out << endl << "freq:";
      for (size_t j=i; j<M; j++) out << " " << setw(8) << right 
				     << _events[j].frequency;
      out << endl << "Enrm:";
      for (size_t j=i; j<M; j++) out << " " << setw(8) << right 
				     << _events[j].normalizedEnergy;
      out << endl << endl;
   }
}

//======================================  Display all events in a list.
void
weventlist::dump(std::ostream& out) const {
   size_t N = _events.size();
   out << "Events for channel: " << _channelName << endl;
   for (size_t i=0; i<N; i++) {
      out << "  Event " << i << ":" << endl;
      _events[i].dump(out);
   }
}

//======================================  Limit number of events in a list.
void
weventlist::limit(size_t maxEvents) {
   if (_events.size() > maxEvents) {
      sort();
      _events.erase(_events.begin()+maxEvents, _events.end());
      _overflowFlag = 1;
   }
}

//======================================  Sort event list by decreasing normE
void
weventlist::sort(void) {
   if (!_events.empty() && !_sorted) {
      std::sort(_events.begin(), _events.end(), std::greater<wpipe::wevent>());
      _sorted = true;
   }
}

//======================================  Static compare function
static bool
tless_wevent(const wevent& a, const wevent& b) {
   return a.t_offset < b.t_offset;
}

//======================================  Sort event list by increasing time
void
weventlist::tsort(void) {
   if (!_events.empty() && !_sorted) {
      std::sort(_events.begin(), _events.end(), &tless_wevent);
      _sorted = false;
   }
}

//======================================  Select non-overlapping events.
void
weventlist::wselect(const weventlist& significants, double dtInflation,
		    double bwInflation, size_t maximumEvents, int debugLevel)
{  
   //-----------------------------------  Propogate channel name & overflow flag
   _refTime      = significants._refTime;
   _channelName  = significants._channelName;
   _overflowFlag = significants._overflowFlag;
   _list_class   = significants._list_class;

   //-----------------------------------  Get number of significant tiles
   size_t numberOfTiles = significants.size();
   if (numberOfTiles == 0) return;

   //-----------------------------------  Verify that tiles are sorted by E
   if (debugLevel > 2) {
      cerr << "weventlist::wselect: Checking event list order." << endl;
      double eLast = significants[0].normalizedEnergy;
      for (size_t i=1; i<numberOfTiles; i++) {
	 double erg_i = significants[i].normalizedEnergy;
	 if (erg_i > eLast) {
	    ostringstream msg;
	    msg << "wselect called with unsorted significants, erg[" << i 
		<< "] = " << erg_i << " prev = " << eLast;
	    throw logic_error(msg.str());
	 }
	 eLast = erg_i;
      }
   }

   // initialize event list
   std::vector<tfbox> eventBox;
   eventBox.reserve(512);
   eventBox.push_back(tfbox(significants._events[0], dtInflation, bwInflation));
   addEvent(significants._events[0]);

   // begin loop over significant tiles
   for (size_t tileIndex=1; tileIndex < numberOfTiles; tileIndex++) {

      //  Get the tile tf-box
      tfbox tfTile(significants[tileIndex], dtInflation, bwInflation);

      //// Loop over events
      size_t eventCount = _events.size();
      bool overlap = false;
      for (size_t eventIndex=0; eventIndex<eventCount; eventIndex++) {

	 //  test for overlap - bail of so
	 overlap = tfTile.overlap(eventBox[eventIndex]);
	 if (overlap) {
	    if (tfbox::_flextile) eventBox[eventIndex].merge(tfTile);
	    break;
	 }
      }

      // add non-overlapping tiles to event list.
      if (!overlap) {
	 addEvent(significants[tileIndex]);
	 eventBox.push_back(tfTile);
      }
   }

   ///////////////////////////////////////////////////////////////////////////
   //                  check for excessive number of events                 //
   ///////////////////////////////////////////////////////////////////////////

   // if maximum allowable number of significant tiles is exceeded
   if (size() > maximumEvents) {

      // issue warning
      wlog(debugLevel, 1, string("WARNING: ") + _channelName + 
	   ": maximum number of events exceeded.");

      // truncate lists of significant event properties
      limit(maximumEvents);
   }

   // otherwise return a sorted list
   else {
      sort();
   }
}

//======================================  Threshold an independent signal.
void
weventlist::wthreshold(const qTransform& qSignal, const wtile& tiling, 
		       double eventThreshold, 
		       const Time& refTime, const dble_vect& tRange, 
		       const dble_vect& fRange, const dble_vect& qRnge,
		       size_t maximumSignificants, 
		       double uncertaintyFactor, int debugLevel) 
{
   const double dInf(1.0/0.0);

   //-----------------------------------  Apply default arguments
   _refTime = refTime;

   dble_vect timeRange;
   if (tRange.empty()) {
      timeRange.resize(2);
      timeRange[0] = -0.5 * (tiling.duration() - 2*tiling.transientDuration());
      timeRange[1] = -timeRange[0];
   } else if (tRange.size() == 2) {
      timeRange = tRange;
   } else {
      error("Time range must be two component vector [tmin tmax].");
   }

   dble_vect frequencyRange;
   if (fRange.empty()) {
      frequencyRange.resize(2);
      frequencyRange[0] = -dInf;
      frequencyRange[1] = dInf;
   } else if (fRange.size() == 2) {
      frequencyRange = fRange;
      if (fRange[0] <= 0) frequencyRange[0] = -dInf;
      if (fRange[1] <= 0) frequencyRange[1] =  dInf;
   } else {
      error("Frequency range must be two component vector [fmin fmax].");
   }

   dble_vect qRange;
   if (qRnge.empty()) {
      qRange.resize(2);
      qRange[0] = -dInf;
      qRange[1] = dInf;
   } else if (qRnge.size() <= 2) {
      qRange = qRnge;
   } else {
      error("Q range must be scalar or two component vector [Qmin Qmax].");
   }

   // if (only a single Q is requested, find nearest Q plane
   if (qRange.size() == 1 ||  qRange[0] == qRange[1]) {
      qRange[0] = tiling.nearest_q(qRange[0]);
      qRange[1] = qRange[0];
   }

   ////////////////////////////////////////////////////////////////////////////
   //            initialize statistically significant event list             //
   ////////////////////////////////////////////////////////////////////////////

   // fill channel names
   _channelName = qSignal._channelName;

   //----------------------------------- Calculate valid time and duration.
   Time     t0 = _refTime     + timeRange[0];
   Interval dt = timeRange[1] - timeRange[0];

   if (debugLevel > 2) {
      cout << "Thresholding channel " << _channelName << " from t = "
	   << t0.totalS() << "-" << (t0+dt).totalS() << endl;
      cout << "  q    f  #significant" << endl;
  }

   ////////////////////////////////////////////////////////////////////////////
   //                           begin loop over Q planes                     //
   ////////////////////////////////////////////////////////////////////////////
   for (int plane=0; plane < tiling.numberOfPlanes(); plane++) {
      double qVal = tiling.planes(plane).q;

      //--------------------------------  Skip Q planes outside requested range
      if (qVal < qRange[0] || qVal > qRange[1]) {
	 continue;
      }

      /////////////////////////////////////////////////////////////////////////
      //                      begin loop over frequency rows                 //
      /////////////////////////////////////////////////////////////////////////
      for (int row=0; row < tiling.planes(plane).numberOfRows; row++) {
	 const qrow& tilePlaneRow(tiling.planes(plane).row(row));
	 double fVal = tilePlaneRow.frequency;

	 //-----------------------------  skip frequency rows outside of range
	 if ((fVal < frequencyRange[0]) || (fVal > frequencyRange[1])) {
	    continue;
	 }
	
	 ///////////////////////////////////////////////////////////////////////
	 //                    threshold on significance                      //
	 ///////////////////////////////////////////////////////////////////////
	 const trow& signalPlaneRow(qSignal.planes(plane).rows(row));
	 TSeries signal = signalPlaneRow.normalizedEnergies.extract(t0, dt);

	 ///////////////////////////////////////////////////////////////////////
	 //                    threshold on central time                      //
	 ///////////////////////////////////////////////////////////////////////

	 // number of statistically significant tiles in frequency row
	 size_t numberOfSignificants = signal.getNGreater(eventThreshold);
	 if (numberOfSignificants == 0) continue;
	 double meanEnergy = signalPlaneRow.meanEnergy;
	 double amplFact   = meanEnergy / signalPlaneRow.conditioningCoef;
	 //cout.unsetf(ios_base::floatfield);
	 //cout << "meanEnergy, ampl-fact = " << meanEnergy << ", " 
	 //     << amplFact << endl;
	 if (debugLevel > 2) {
	    cout << qVal << " " << fVal << " " << numberOfSignificants << endl;
	 }

	 ///////////////////////////////////////////////////////////////////////
	 //      append significant tile properties to event structure        //
	 ///////////////////////////////////////////////////////////////////////
	 event_vect evec;
	 evec.reserve(numberOfSignificants);

	 //  construct an event for this plane/row.
	 wevent evt;
	 evt.frequency = fVal;
	 evt.q         = qVal;
	 evt.duration  = tilePlaneRow.duration;
	 evt.bandwidth = tilePlaneRow.bandwidth;

	 //  loop over vectors.
	 const DVectD& sig_dvd=dynamic_cast<const DVectD&>(*signal.refDVect());
	 int sigLength = signal.getNSample();
	 for (int ibin=0; ibin < sigLength; ++ibin) {
	    if (sig_dvd[ibin] >= eventThreshold) {
	       evt.t_offset  = double(signal.getBinT(ibin) - _refTime);
	       evt.normalizedEnergy = sig_dvd[ibin];
	       evt.amplitude = sqrt((evt.normalizedEnergy - 1.0) * amplFact);
	       evec.push_back(evt);
	    }
	 }

	 ///////////////////////////////////////////////////////////////////////
	 //             prune excessive significants as we accumulate         //
	 ///////////////////////////////////////////////////////////////////////
 	 numberOfSignificants = _events.size() + evec.size();
	 if (debugLevel >= 2 && numberOfSignificants > maximumSignificants) {
	    cout << _channelName << ": triming excess significants to maximum ("
		 << maximumSignificants << ")" << endl; 
	 }
	 sorted_merge(evec, maximumSignificants);
      }	 // end loop over frequency rows
   }  // end loop over Q planes

   //-----------------------------------  Final sort.
   sort();
}

//======================================  Threshold a signal + reference
void
weventlist::wthreshold(const qTransform& qSignal, const qTransform& qRefer, 
		       const wtile& tiling, double eventThreshold, 
		       const Time& refTime, const dble_vect& tRange, 
		       const dble_vect& fRange, const dble_vect& qRnge, 
		       size_t maximumSignificants, double uncertaintyFactor, 
		       double correlationFactor, int debugLevel) {

   ////////////////////////////////////////////////////////////////////////////
   //                        process command line arguments                  //
   ////////////////////////////////////////////////////////////////////////////

   const double dInf(1.0/0.0);

   // apply default arguments
   _refTime = refTime;

   dble_vect timeRange;
   if (tRange.empty()) {
      timeRange.resize(2);
      timeRange[0] = -0.5 * (tiling.duration() - 2*tiling.transientDuration());
      timeRange[1] = -timeRange[0];
   } else {
      timeRange = tRange;
   }

   //-----------------------------------  Check the frequency range
   dble_vect frequencyRange = fRange;
   if (fRange.empty()) frequencyRange.resize(2, 0.0);
   if (frequencyRange[0] <= 0) frequencyRange[0] = -dInf;
   if (frequencyRange[1] <= 0) frequencyRange[1] =  dInf;

   //-----------------------------------  Check the Q range
   dble_vect qRange;
   if (qRnge.empty()) {
      qRange.resize(2);
      qRange[0] = -dInf;
      qRange[1] = dInf;
   } else {
      qRange = qRnge;
   }

   //-----------------------------------  find nearest plane to a single Q 
   if (qRange.size() == 1 || qRange[0] == qRange[1]) {
      qRange[0] = tiling.nearest_q(qRange[0]);
      qRange[1] = qRange[0];
   }

   /////////////////////////////////////////////////////////////////////////////
   //                       validate command line arguments                   //
   /////////////////////////////////////////////////////////////////////////////

   // Check for two component range vectors
   if (timeRange.size() != 2) {
      error("Time range must be two component vector [tmin tmax].");
   }

   if (frequencyRange.size() != 2) {
      error("Frequency range must be two component vector [fmin fmax].");
   }

   if (qRange.size() > 2) {
      error("Q range must be scalar or two component vector [Qmin Qmax].");
   }

   ////////////////////////////////////////////////////////////////////////////
   //                           begin loop over Q planes                     //
   ////////////////////////////////////////////////////////////////////////////
   for (int plane=0; plane < tiling.numberOfPlanes(); plane++) {

      //--------------------------------  Skip Q planes outside requested range
      if (tiling.planes(plane).q < qRange[0] || 
	  tiling.planes(plane).q > qRange[1]) {
	 continue;
      }

      /////////////////////////////////////////////////////////////////////////
      //                    begin loop over frequency rows                   //
      /////////////////////////////////////////////////////////////////////////
      for (int row=0; row < tiling.planes(plane).numberOfRows; row++) {
	 const qrow& tilePlaneRow(tiling.planes(plane).row(row));

	 //-----------------------------  Skip frequency rows outside of range
	 if ((tilePlaneRow.frequency < frequencyRange[0]) || 
	     (tilePlaneRow.frequency > frequencyRange[1])) {
	    continue;
	 }
	
	 //////////////////////////////////////////////////////////////////////
	 //                    threshold on significance                     //
	 //////////////////////////////////////////////////////////////////////

	 TSeries signal = qSignal.planes(plane).rows(row).normalizedEnergies
	    .extract(_refTime+timeRange[0],  timeRange[1]-timeRange[0]);
	 TSeries refer  = qRefer.planes(plane).rows(row).normalizedEnergies
	    .extract(_refTime+timeRange[0],  timeRange[1]-timeRange[0]);
	 TSeries select(signal);
	 TSeries temp(refer);
	 temp *= correlationFactor;
	 select -= temp;

	 //////////////////////////////////////////////////////////////////////
	 //                    threshold on central time                     //
	 //////////////////////////////////////////////////////////////////////

	 // number of statistically significant tiles in frequency row
	 size_t numberOfSignificants = select.getNGreater(eventThreshold);
	 if (numberOfSignificants == 0) continue;
	 double meanEnergy = qSignal.planes(plane).rows(row).meanEnergy;

	 //////////////////////////////////////////////////////////////////////
	 //      append significant tile properties to event structure       //
	 //////////////////////////////////////////////////////////////////////
	 event_vect evec;
	 evec.reserve(numberOfSignificants);

	 //  construct an event for this plane/row.
	 wevent evt;
	 evt.frequency = tilePlaneRow.frequency;
	 evt.q         = tiling.planes(plane).q;
	 evt.duration  = tilePlaneRow.duration;
	 evt.bandwidth = tilePlaneRow.bandwidth;

	 //-----------------------------  loop over vectors.
	 const DVectD& sig_dvd=dynamic_cast<const DVectD&>(*signal.refDVect());
	 const DVectD& sel_dvd=dynamic_cast<const DVectD&>(*select.refDVect());
	 const DVectD& ref_dvd=dynamic_cast<const DVectD&>(*refer.refDVect());
	 int sigLength = signal.getNSample();
	 for (int ibin=0; ibin < sigLength; ++ibin) {
	    if (sel_dvd[ibin] >= eventThreshold) {
	       evt.t_offset   = double(signal.getBinT(ibin) - _refTime);
	       evt.normalizedEnergy = sig_dvd[ibin];
	       evt.amplitude = sqrt((evt.normalizedEnergy - 1.0) * meanEnergy);
	       evt.incoherentEnergy = ref_dvd[ibin];
	       evec.push_back(evt);
	    }
	 }
      
	 //-----------------------------  Flag excess events.
	 numberOfSignificants = _events.size() + evec.size();
	 if (debugLevel >= 2 && numberOfSignificants > maximumSignificants) {
	    cout << _channelName << ": triming excess significants to maximum ("
		 << maximumSignificants << ")" << endl;
	 }

	 //-----------------------------  Merge, prune excess on the fly.
	 sorted_merge(evec, maximumSignificants);
      }	 // end loop over frequency rows
   }   // end loop over Q planes
   sort();
}

//======================================  Merge events
void
weventlist::sorted_merge(event_vect& ev, size_t maxLength) {
   size_t nAdd  = ev.size();
   if (!nAdd) return;
   size_t nList = _events.size();

   //-----------------------------------  Add events to the end
   if (nList + nAdd <= maxLength) {
      _events.insert(_events.end(), ev.begin(), ev.end());
      _sorted = false;
   }

   //-----------------------------------  See if no events yet.
   else if (_events.empty()) {
      _events.swap(ev);
      _sorted = false;
      limit(maxLength);
   }

   //-----------------------------------  Need to ignore some events.
   else {
      sort();
      std::sort(ev.begin(), ev.end(), std::greater<wpipe::wevent>());
      event_vect temp;
      temp.reserve(maxLength);
      event_iter in1=_events.begin();
      event_iter in2=ev.begin();
      for (size_t i=0; i<maxLength; ++i) {
	 if (in1 == _events.end()) {
	    temp.push_back(*in2++);
	 } else if (in2 == ev.end()) {
	    temp.push_back(*in1++);
	 } else if (*in1 < *in2) {
	    temp.push_back(*in2++);
	 } else {
	    temp.push_back(*in1++);
	 }
      }
      _events.swap(temp);
   }
}

//=====================================  swap contents of two lists
void 
weventlist::swap(weventlist& el) {
   _channelName.swap(el._channelName);
   _list_class.swap(el. _list_class);
   std::swap<int>(_overflowFlag, el._overflowFlag);
   std::swap<Time>(_refTime, el._refTime);
   std::swap<bool>(_sorted, el._sorted);
   _events.swap(el._events);
}

//======================================  Write all events
void
weventlist::writeEvent(const string& path, const str_vect& fields, 
		       const string& format) const {

   //-----------------------------------  Write a txt file.
   if (format == "txt") {
      writeEvent_txt(path, fields);
   }

   //------------------------------------  Write an xml file
   else if (format == "xml") {
      writeEvent_xml(path);
   }

   //------------------------------------  Unrecognized format
   else {
      error("Output format not recognized in weventlist::writeEvent");
   }
}

//======================================  Write all events to a txt file
void 
weventlist::writeEvent_txt(const string& path, const str_vect& flds) const {
   ofstream out(path.c_str(), ios::app);
   if (!out.is_open()) {
      error(string("Unable to open output file: ") + path);
   }
   writeEvent_txt(out, flds);
}

//======================================  Write all events to a specified
void 
weventlist::writeEvent_txt(ostream& out, const str_vect& fields) const {
   size_t M = fields.size();
   size_t N = _events.size();
   lcl_array<double> fld_ij(M*N);
   lcl_array<int>    prec_i(M);

   for (size_t inx=0; inx<M; inx++) prec_i[inx] = -16;

   for (size_t i=0; i<M; i++) {
      size_t inx = i;
      if (fields[i] == "time") {
	 prec_i[i] = 9;
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _refTime.totalS() + _events[j].t_offset;
	    inx += M;
	 }
      }
      else if (fields[i] == "frequency") {
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].frequency;
	    inx += M;
	 }
      }
      else if (fields[i] == "q") {
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].q;
	    inx += M;
	 }
      }
      else if (fields[i] == "duration") {
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].duration;
	    inx += M;
	 }
      }
      else if (fields[i] == "bandwidth") {
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].bandwidth;
	    inx += M;
	 }
      }
      else if (fields[i] == "normalizedEnergy" ||
	       fields[i] == "clusterNormalizedEnergy") {
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].normalizedEnergy;
	    inx += M;
	 }
      }
      else if (fields[i] == "clusterSize") {
	 prec_i[i] = 0;
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].cluster_size;
	    inx += M;
	 }
      }
      else if (fields[i] == "clusterNumber") {
         prec_i[i] = 0;
	 for (size_t j=0; j<N; j++) {
	    fld_ij[inx] = _events[j].cluster_id;
	    inx += M;
	 }
      }
      else {
	 error(string("Unrecognized field name: ") + fields[i]);
      }
   }

   //------------------------------------  Write output field names
   if (!out.tellp()) {
      for (size_t i=0; i<M; i++) {
	 out << "% " << fields[i] << endl;
      }
   }

   //------------------------------------  write a one-line ID
   out << "% - events for channel: " << _channelName << " t-reference: "
       << _refTime << " ofl: " << _overflowFlag << endl;

   //------------------------------------  loop over channels
   out << fixed << setprecision(9);
   int set_pr = 9;
   size_t ij = 0;
   for (size_t i=0; i<N; i++) {
      for (size_t j=0; j<M; j++) {
	 if (j) out << " ";
	 if (prec_i[j] != set_pr) {
	    set_pr = prec_i[j];
	    if (set_pr == 0)     out.unsetf(ios_base::floatfield);
	    else if (set_pr < 0) out << scientific << setprecision(-set_pr);
	    else                 out << fixed << setprecision(set_pr);
	 }
	 out << fld_ij[ij++];
      }
      out << endl;
   }
}

//======================================  Write all events to an xml file
void 
weventlist::writeEvent_xml(const string& path) const {
   TrigClient out(trig::TrigWriter::kSBTrig);
   out.setTableFile(path.c_str());
   writeEvent_xml(out);
}

//======================================  Write all events to a TrigClient
void 
weventlist::writeEvent_xml(TrigClient& out) const {
   string ifo = _channelName.substr(0,2);

   size_t N = _events.size();
   for (size_t i=0; i<N; i++) {
      const wevent& evi(_events[i]);
      Time t0 = _refTime + evi.t_offset;
      Interval dt = evi.duration;
      // The signal/noise ratio come from the normalized energy as follows:
      // The nomalized energy is the sum of the signal from two degrees of 
      // freedom (sin, cosine components), the noise from which add up to 1. 
      // Thus: snr = sqrt(2*NE - 1)
      double S2N = sqrt(2*evi.normalizedEnergy - 1);
      trig::SBTrigger wtrig("Omega_C", channelName(), t0 - dt*0.5, dt, 
			    evi.frequency, evi.bandwidth, S2N);
      wtrig.setPixelCount(evi.cluster_size);
      wtrig.peak_time(t0);
      wtrig.ifo(ifo);
      wtrig.amplitude(evi.amplitude);
      out.sendTrigger(wtrig);
   }
}

//======================================  Create and reserve space
weventstack::weventstack(size_t N) {
   _lists.reserve(N);
}

//======================================  Dump all events in a weventstack.
void
weventstack::dump(std::ostream& out) const {
   size_t N = _lists.size();
   for (size_t i=0; i<N; i++) {
      _lists[i].dump(out);
   }
}

//======================================  Dump all events in a weventstack.
void
weventstack::display(std::ostream& out) const {
   size_t N = _lists.size();
   for (size_t i=0; i<N; i++) {
      _lists[i].display(out);
   }
}

//======================================  Move a list to this stack
void
weventstack::moveLists(weventstack& el) {
   weventlist mtlist;
   size_t N = el.numberOfChannels();
   for (size_t i=0; i<N; i++) {
      _lists.push_back(mtlist);
      _lists.back().swap(el._lists[i]);
   }
}

//======================================  Print weventstack status.
void
weventstack::status(std::ostream& out) const {
   size_t N = _lists.size();
   for (size_t i=0; i<N; i++) {
      out << "    " << _lists[i].channelName() << ": " << _lists[i].size() 
	  << " " << _lists[i].eventClass() << "s." << endl;
   }
}

//======================================  Swap contents of two stacks
void
weventstack::swap(weventstack& wstak) {
   _lists.swap(wstak._lists);
}

//======================================  Total events in the stack;
size_t 
weventstack::totalEvents(void) const {
   size_t total = 0;
   size_t N = _lists.size();
   for (size_t i=0; i<N; i++) {
      total += _lists[i].size();
   }
   return total;
}

//======================================  Select event from an event-stack.
void 
weventstack::tsort(void) {

   // determine number of channels
   size_t N = _lists.size();
   for (size_t i=0; i<N; i++) {
      _lists[i].tsort();
   }
}

//======================================  Select event from an event-stack.
void 
weventstack::wselect(const weventstack& significants,  double dtInflation,
		     double bwInflation, size_t maxEvents, int debugLevel) {

   // determine number of channels
   size_t N = significants.numberOfChannels();
   _lists.resize(N, weventlist("wselect"));

   for (size_t i=0; i<N; i++) {
      _lists[i].wselect(significants[i], dtInflation, bwInflation, maxEvents,
			debugLevel);
   }
}

//====================================== Build an event stack from a 
//                                       transform stack
void 
weventstack::wthreshold(const wtransform& transforms, const wtile& tiling, 
			double eventThreshold, 
			const Time& referenceTime, const dble_vect& timeRange, 
			const dble_vect& fRange, const dble_vect& qRange, 
			size_t maximumSignificants,  const string& analysisMode,
			double vetoThreshold, double uncertaintyFactor, 
			double correlationFactor, int debugLevel) 
{
   size_t N = transforms.numberOfChannels();
   _lists.resize(N, weventlist("tile"));

   //------------------------------------  Coherent analysis
   if (analysisMode == "coherent") {

      for (size_t i=0; i<N; i+=2) {
	 //--------------------------------  signal channel
	 _lists[i].wthreshold(transforms[i], transforms[i+1], tiling, 
			      eventThreshold, referenceTime, timeRange, 
			      fRange, qRange,maximumSignificants, 
			      uncertaintyFactor, correlationFactor, debugLevel);

	 //--------------------------------  Null channel
	 _lists[i+1].wthreshold(transforms[i+1], tiling, 
				vetoThreshold, referenceTime, timeRange, 
				fRange, qRange, maximumSignificants, 
				uncertaintyFactor, debugLevel);
      }
   } 

   //------------------------------------  Independent channels
   else {
      for (size_t i=0; i<N; i++) {
	 _lists[i].wthreshold(transforms[i], tiling, eventThreshold, 
			      referenceTime, timeRange, fRange, qRange, 
			      maximumSignificants, uncertaintyFactor, 
			      debugLevel);
      }
   }
}

//======================================  Write all events
void
weventstack::writeEvents(const wouttype& wout, const str_vect& fields, 
			 const std::string& format) const {
   //-----------------------------------  Verify supported format.
   if (format != "xml" && format != "txt") {      
      error(string("Unknown trigger format: ") + format);
   }

   //------------------------------------  loop over channels
   size_t N = _lists.size();
   bool_vect done(N, false);
   for (size_t i=0; i<N; i++) {
      if (done[i]) continue;
      const string& path = wout.path(_lists[i].channelName()); 
      if (path.empty()) {
	 cerr << "No output path found for channel: " << _lists[i].channelName()
	      << endl;
	 continue;
      }

      //--------------------------------  Process a txt file
      if (format == "txt") {
	 //-----------------------------  Open the file.
	 ofstream out(path.c_str(), ios::app);
	 if (!out.is_open()) {
	    error(string("Unable to open output file: ") + path);
	 }

	 //-----------------------------  Add triggers for appropriate channels
	 for (size_t j=i; j<N; ++j) {
	    if (path == wout.path(_lists[j].channelName())) {
	       _lists[j].writeEvent_txt(out, fields);
	       done[j] = true;
	    }
	 }
      }

      //--------------------------------  Process an xml file
      else if (format == "xml") {
	 //-----------------------------  Construct a trigger client
	 TrigClient out(trig::TrigWriter::kSBTrig);
	 out.setTableFile(path.c_str());

	 //-----------------------------  Add triggers for appropriate channels
	 for (size_t j=i; j<N; ++j) {
	    if (path == wout.path(_lists[j].channelName())) {
	       _lists[j].writeEvent_xml(out);
	       done[j] = true;
	    }
	 }
      }
   }
}
