/***************************************************************************
    File        : RandomMersenne.h
    Description : Random number generator of type Mersenne twister.
                  Adaption of the code by: Uniform random number generators
                  by Agner Fog, sources randomc.zip at www.agner.org/random/
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/


/************************** MERSENNE.CPP ******************** AgF 2001-10-18 *
*  Random Number generator 'Mersenne Twister'                                *
*                                                                            *
*  This random number generator is described in the article by               *
*  M. Matsumoto & T. Nishimura, in:                                          *
*  ACM Transactions on Modeling and Computer Simulation,                     *
*  vol. 8, no. 1, 1998, pp. 3-30.                                            *
*  Details on the initialization scheme can be found at                      *
*  http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html                  *
*                                                                            *
*  Experts consider this an excellent random number generator.               *
*                                                                            *
*   2001 - 2004 A. Fog.                                                     *
*  GNU General Public License www.gnu.org/copyleft/gpl.html                  *
*****************************************************************************/


#ifndef __RANDOM_MERSENNE_H
#define __RANDOM_MERSENNE_H


/*!
 * \file RandomMersenne.h
 *  Random number generator of type Mersenne twister.
 *  Adaption of the code by: Uniform random number generators
 *  by Agner Fog, sources randomc.zip at www.agner.org/random/
 *   2001 - 2004 A. Fog.
 *  GNU General Public License www.gnu.org/copyleft/gpl.html
 *  \brief Implements a random number generator of type Mersenne twister.
 */



// System Libs
#include <ctime>
#include <vector>

// Defines size for MT19937:
#define MERS_N   624


namespace gwd {

  //! Implements a random number generator of type Mersenne twister.
  class RandomMersenne {
  public:
    /** Defaulf constructor. Uses time(NULL) to initialize the random number generator */
    RandomMersenne() { DetectComputerArchitecture();RandomInit((unsigned long int)time(NULL)); }

    /** Constructor takes a seed to initialize the random number generator */
    RandomMersenne(unsigned long int seed) {  DetectComputerArchitecture();RandomInit(seed); }

    virtual ~RandomMersenne() { }

    // Methods
    /** Re-initialize the random number generator by using a seed */
    void RandomInit(unsigned long int seed);    // re-seed

    /** Initialize with more the 32 bits. Use this function if you want to initialize
     *  with a seed with more than 32 bits. All bits in the seeds[] array will influence
     *  the sequence of random numbers generated.
     */
    void RandomInitByArray(std::vector<unsigned long int> seeds); // seed by more than 32 bits

    /** Returns an random integer in the interval  [min,max] */
    int IRandom(int min, int max);       // output random integer

    /** Returns a random double in the interval [0,1) */
    double Random();                     // output random float

    /** Generates 32 random bit */
    unsigned long int BRandom();         // output random bits

  private:
    unsigned long int mMT[MERS_N];                   // state vector
    int mMTIdx;
    enum Architecture {LITTLEENDIAN, BIGENDIAN, NONIEEE};
    Architecture mArchitecture;                  // conversion to float depends on computer architecture

    //! Determine littleendian, bigendian number representation
    void DetectComputerArchitecture();
  };


} // namespce

#endif // __RANDOM_MERSENNE_H
