/***************************************************************************
    File        : NotchFilter.h
    Description : Generates a Notch filter
 ---------------------------------------------------------------------------
    Begin       : Mon Sep 3 2001
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __NOTCHFILTER_H
#define __NOTCHFILTER_H

/*!
 * \file NotchFilter.h
 * A Notch filter generated with windowed filter design method. The frequency
 * bands are specified in frequency domain and tranformed to time domain.
 * The filter coefficients are multiplied in time domain with a window.
 * One of the common well known windows can be used. The default window is
 * the Hamming window.
 * \brief Notch filter generated with the windowed filter design method.
 */


// Libs
#include <cstdlib>
#include <vector>

// Project files
#include "Types.h"
#include "Fourier.h"
#include "Window.h"

namespace gwd {

  //! A Notch filter generated with the windowing filter design method.
  class NotchFilter {
  public:
   /** Default constructor. */
    NotchFilter()  {}

    /** Destructor */
    ~NotchFilter() {}

    // Methods
    /**
     * Computes the coefficients for the Notch filter. The frequency
     * bands are specified in frequency domain and tranformed to time domain.
     * The filter coefficients are multiplied in time domain with a window.
     * @param SamplingRate singal sample rate.
     * @param order filter order.
     * @param frq frequencies to be eliminated.
     * @param wdt band width around the frequencies.
     * @param filter filter coefficients.
     */
    void ComputeFilter(const double SamplingRate,const unsigned int order,
                       const Vector& frq,
                       const Vector& wdt,
                       Vector& filter);
  private:
    // Method
    /**
     * Specifyes the frequency bands in frequency domain.
     */
    void FrequencyResponse(const double SamplingRate,
                           const Vector& frq,
                           const Vector& wdt,
                           std::vector<Complex>& filter);
  };

} // namespace gwd
#endif // __NOTCHFILTER_H

