/***************************************************************************
    File        : Fourier.h
    Description : Discret Fourier Transform of real and complex data
                  This is a wraper class for the fftw 3.0.1 library
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __FOURIER_H
#define __FOURIER_H

/*! \file Fourier.h
 *  This is a wrapper class to the fftw 3.0.1.
 *  \brief This is a wrapper class to the fftw 3.0.1.
 */

// System Libs
#include <iostream>
#include <vector>
#include <complex>
#include <cmath>

// The fftw
#include "fftw3.h"

// Project files
#include "Types.h"
#include "Singleton.h"
#include "Numeric.h"


namespace gwd {

  const int DFT_FORWARD  = -1;
  const int DFT_BACKWARD =  1;

  //! This is a wrapper class to the fftw 3.0.1
  class Fourier {
  public:
    // Constructors
    /** Default constructor */
    Fourier ();

    /**
     * This constructor initializes the fftw for FFT of complex data.
     * The size of the input data is not necessary a power of two.
     * The vector data not necessary contains data.
     * @param NN size of input data.
     * @param isign indicates the direction of the transform.
     * @param data pointer to array where the trasform is performed in done in-place.
     */
    Fourier (const size_t NN,const int isign,std::vector<Complex>& data);

    /** This constructor initializes the fftw for a real to complex transform. */
    Fourier (const size_t NN,const Vector& inData,std::vector<Complex>& outData);

    /** This constructor initializes the fftw for a complex to real transfrom. */
    Fourier (const size_t NN,const std::vector<Complex>& inData,Vector&  outData);

    // Desctructors
    /** Destructor */
    virtual ~Fourier();

    /**
     * Initialize the DFT, i.e. runs the fftw3 plan, for an in-place complex DFT.
     * The parameter isign set the direction of the DFT.
     * @param NN size of data.
     * @param isign indicates the direction of the transform.
     * @param data pointer to array where the transform is performed in-place.
     * @return true in case of success, else false.
     */
    bool Init(const size_t NN,const int isign,std::vector<Complex>& data);

    /**
     * Initialize the DFT, i.e. runs the fftw3 plan, for an in-place real
     * to complex DFT.
     * The size of the input real vector is NN and the size of the output complex
     * vector is NN/2 + 1.
     * @param NN size of data.
     * @param inData real input data.
     * @param outData complex output data.
     * @return true in case of success, else false.
     */
    bool Init(const size_t NN,const Vector&  inData,std::vector<Complex>& outData);

    /**
     * Initialize the DFT, i.e. runs the fftw3 plan, for an in-place complex
     * to real DFT.
     * The size of the input complex vector is NN/2 + 1 and the size of output real
     * vector is NN.
     * @param NN size of data
     * @param inData complex input data.
     * @param outData real output data.
     * @return true in case of success, else false.
     */
    bool Init(const size_t NN,const std::vector<Complex>& inData,Vector&  outData);

    /**
     * Computes the DFT which was initialized by the Constructor or one of the Init methods
     * of the class.
     * The values of the real or complex input vector have to be set somewehre else
     * in the program.
     * @return true in case of success, else false.
     */
    bool dft();

    /**
     * Computes the DFT for a complex input. The Discrete Fourier Transform is done
     * in-place, i.e. the input data is overwritten. This method creates a local
     * fftw3 plan for a complex to complex DFT, carries out the DFT and then destroys
     * the plan.
     * @param isign indicates the direction of the transform.
     * @param data contains the input and is overwritte by the transform.
     * @return true in case of success, else false.
     */
    bool dft(const int isign, std::vector<Complex>& data);

    /**
     * Computes the DFT for a real input. The output is complex. The Discrete Fourier
     * This method creates a local fftw3 plan for a real to complex DFT, carries out
     * the DFT and then destroys the plan.
     * @param inData real input data.
     * @param outData complex output data.
     * @return true in case of success, else false.
     */
    bool dft(const Vector&  inData,std::vector<Complex>& outData);

    /**
     * Computes the DFT for a complex input. The output is real. The Discrete Fourier
     * This method creates a local fftw3 plan for a complex to real DFT, carries out
     * the DFT and then destroys the plan.
     * @param inData complex input data.
     * @param outData real output data.
     * @return true in case of success, else false.
     */
    bool dft(const std::vector<Complex>& inData,Vector&  outData);

    /**
     * The DFT is stored in-order or as double sided series in the output array,
     * with the zero frequency component in a position n = 0 and F(0) to Fmax in
     * bins 0 to (N/2-1) and Fmin to (F(0)-dF) in bins N/2 to N-1.
     * This method sorts a complex data into single sided series stored linearly
     * from F(0)-Fnyquist in increasing bins. The operation is done in-place.
     * @param inData data to be sorted in-place.
     * @return true in case of success, else false.
     */
    bool Sort(std::vector<Complex>& inData);

    /**
     * This method sorts complex data into single sided series stored linearly
     * from F(0)-Fnyquist in increasing bins. The result is written into the
     * output array.
     * @param inData data to be sorted.
     * @param outData sorted data.
     * @return true in case of success, else false.
     */
    bool Sort(const std::vector<Complex>& inData,std::vector<Complex>& sortedData);

    /**
     * This method sorts real data into single sided series stored linearly
     * from F(0)-Fnyquist in increasing bins. The sort is done in-place.
     * @param inData data to be sorted in-place.
     * @return true in case of success, else false.
     */
    bool Sort(Vector& inData);

    /**
     * This method sorts real data into single sided series stored linearly
     * from F(0)-Fnyquist in increasing bins. The result is written into the
     * output array.
     * @param inData data to be sorted.
     * @param outData sorted data.
     * @return true in case of success, else false.
     */
    bool Sort(const Vector& inData,Vector& sortedData);

  private:
    size_t        mSize;
    double*       mRealData;
    fftw_complex* mComplexData;
    fftw_plan     mPlanC2C;
    fftw_plan     mPlanR2C;
    fftw_plan     mPlanC2R;
  };


} // namespce

#endif // __FOURIER_H
