/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DMTServer.hh"
#include "MonServer.hh"
#include "TSeries.hh"
#include "Histogram1.hh"
#include "LSCSegFind.hh"
#include "TrigPar.hh"
#include <string>
#include <map>

/**  @name Cumulus
  *  Cumulus serves cumulative statistic time series to the dmtviewer.
  *
  *  \begin{center}{\large {\bf Running Cumulus}} \end{center}
  *
  *  The Cumulus command line syntax is as follows:
  *  
  *  \begin{verbatim}
  Cumulus -conf <cfile>
  \end{verbatim}
  *
  *  Where:
  *  \begin{tabular}{ll}
  *  -conf <cfile> & Specifies the configuration file name.
  *  \end{tabular}
  *
  *  \begin{center}{\large {\bf PSLmon Outputs}} \end{center}
  *
  *  {\bf Status Reports}
  *
  *  None defined yet.
  *
  *  \begin{center}{\large {\bf Configuring the Cumulus monitor}} \end{center}
  *
  *  A configuration file is used to specify how data are to be calculated.
  *
  *  The configuration file consists of an arbitrary number of lines.
  *  Each line consists of one or more fields separated by blank space.
  *  The line terminates at a new line or at a pound sign('\#').
  *  The first field of each line is a command which may be any of the 
  *  following:
  *  \begin{itemize} 
  *  \item #Display:# Generate the specified dmtviewer object.
  *  \item #Segment:# Define a segment.
  *  \end{itemize}
  *  The number of fields following the command differs among commands. 
  *  Optional fields are preceded by a hyphen and a keyword.
  *  The configuration command syntax is covered in the sections below 
  *  and the descriptions of the appropriate tools. 
  *
  *  {\bf Displays}
  *
  *  Displays are defined with the Display configuration command. 
  *  The syntax of the Display command is: \	\
  *  \begin{verbatim}
  Display <name> [-dt <t-step>] [-segment <segment>] [-start <gps-start>] 
  \end{verbatim}
  *  The currently defined parameters are: \	\
  *
  *  \begin{tabular}{|l|c|l|} \			\
  *  Parameter & Meaning \				\
  *  #<t-step># & time increment in the time series \	\
  *  #<gps-start># & start time of the time series \			\
  *  #<segment># & name of cumulative segment name to be displayed.
  *  \end{tabular}
  *
  *  {\bf Segment specification}
  *
  *  The segment specifier defines a segment to be displayed by Cumulus.
  *
  *  The syntax of the Display command is: \	\
  *  \begin{verbatim}
  Segment <name> [-ifo <ifo>] [-url <seg-url>] [-file <file-name>]	\
  [-server <-url-node>]
  \end{verbatim}
  *  The currently defined parameters are: \	\
  *
  *  \begin{tabular}{|l|c|l|} \			\
  *  Parameter & Meaning \				\
  *  #<t-step># & time increment in the time series \	\
  *  #<gps-start># & start time of the time series \			\
  *  #<segment># & name of cumulative segment name to be displayed.
  *  \end{tabular}
  *
  */
//@{
//@}

class Cumulus : public DMTServer {
public:
    typedef std::map<std::string, LSCSegFind> segmap;
    typedef segmap::iterator segmap_iter;
public:
    Cumulus(int argc, const char* argv[]);
    ~Cumulus(void);
    void ProcessData(const Time& t);
    void Attention(void);
    void readConfig(void);
    bool segExists(const std::string& name);

public:
    class segdsply {
    public:
	segdsply(const std::string& name, const std::string& seg, 
		 const Time& start, Interval dt);
	const char* getName(void) const {return _name.c_str();}
	void book(MonServer& svr);
	void fill(Cumulus::segmap& sm);
	const LockSegList& segEval(Cumulus::segmap& sm);
	void setDebug(int lvl);
    private:
	std::string _name;
	std::string _seg;
	Time        _start;
	Interval    _dt;
	TSeries     _ts;
	Histogram1  _hist;
	int         _debug;
    };
    typedef std::vector<segdsply> dspvect;
    typedef dspvect::iterator dsp_iter;
private:
    std::string mServerName;
    std::string mConfFile;
    segmap  mSegMap;
    dspvect mDspVect;
    TrigPar mDict;
};

//======================================  Inline functions.
inline bool
Cumulus::segExists(const std::string& name) {
    return mSegMap.find(name) != mSegMap.end();
}
