/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File: BitTest.hh
//
//    Bit Test monitor class definition.
//
#ifndef BITTEST_HH
#define BITTEST_HH

#include "DMTBase.hh"
#include "TrigClient.hh"
#include "AlarmClient.hh"
#include <list>
#include <iosfwd>
#include <string>

class ChanBit;

//======================================  Monitor class definition
class BitTest : public DMTBase, TrigClient {
  public:
    BitTest(int argc, const char *argv[]);
    ~BitTest();
  void ProcessFrame(DMTBase::frame_ptr_type frame);
    void Attention(void);
    bool ReadConfig(void);
    void Report(void);
    void Reset(void);

  private:
    AlarmClient   mAlarm;
    unsigned int  mReset;
    unsigned int  mSynch;
    unsigned int  mProcessed;
    std::string   mConfig;
    std::string   mOutput;
    std::list<ChanBit> mChannel;
    typedef std::list<ChanBit>::iterator channel_iter;
    bool          mGenTrig;
    Time          mLastProcessed;
};

//
//    ChanBit Class
//
//    The ChanBit class carries the status of each channel to be followed
//    by the bit tester.
//
class ChanBit {
  public:
     typedef unsigned int mask_t;
     typedef unsigned int counter_t;
     typedef unsigned int uint_t;
    //----------------------------------  Constructors, Destructors.
    ChanBit(const char *, mask_t ign = 0, counter_t maxct=1);
    ~ChanBit();

    //----------------------------------  Scan the data
    void Scan(uint_t nw, const short *data);
    void Scan(uint_t nw, const float *data);
    void Print(std::ostream& ostr, bool phdr=false) const;
    void Status(std::ostream& ostr, bool phdr=false) const;
    void incReadOutError(void) {mReadOutErr++;}
    void setDebug(int debug);
    void setReported(bool card);
    void Reset (void);
    bool BitError(void) const;
    bool RepeatError(void) const;
    bool OverflowError(void) const {return mOverFlow != 0;}

    /*  Test whether the channel has a flagable error. A flagable error 
     *  is considered to be a stuck bit, a repeat count error, a readout 
     *  error or an overflow error.
     *  @memo Test for a flagable error.
     */
    bool Error(void) const;
    bool isReported(void) const {return mReported;}

    //----------------------------------  Accessors
    mask_t      getStuckOn()   const;
    mask_t      getStuckOff()  const;
    mask_t      getStuckAdj()  const;
    const char* getChannel()   const {return mChanName.c_str();}
    counter_t   getNFrames()   const {return mNFrames;}
    counter_t   getNWords()    const {return counter_t(mNWords);}
    double      getAvgLen()    const;
    double      getAverage()   const;
    double      getSigma()     const;
    double      getMinimum()   const {return mMinimum;}
    double      getMaximum()   const {return mMaximum;}
    counter_t   getMaxRpt(void) const {return mMaxRpt;}
    counter_t   getLongRpt()   const {return mLongR;}
    mask_t      getLastValue()  const {return mValue;}
    mask_t      getHiBits() const;
    mask_t      getDBits()  const {return mDBits;}
    counter_t   getNOverFlow(void) const {return mOverFlow;}
    counter_t   getNReadOutError(void) const {return mReadOutErr;}


  private:
    std::string mChanName;
    counter_t mMaxRpt;
    counter_t mNFrames;
    mask_t    mSelect;
    counter_t mDBits;
    int       mDebug;
    double    mNWords;
    double    mAverage;
    double    mSigma;
    double    mMinimum;
    double    mMaximum;
    mask_t    mStuckOn;
    mask_t    mStuckOff;
    mask_t    mStuckAdj[4];
    mask_t    mValue;
    double    mFValue;
    counter_t mRepeat;
    counter_t mLongR;
    bool      mReported;
    bool      mInteger;
    counter_t mOverFlow;
    counter_t mReadOutErr;
};

//====================================  Inline ChanBit Methods
#ifndef __CINT__

//------------------------------------  Test if bits are stuck on/off
inline ChanBit::mask_t
ChanBit::getStuckOn()  const {
    return  (mStuckOn & mDBits);
}

inline ChanBit::mask_t
ChanBit::getStuckOff()  const {
    return  (~mStuckOff & mDBits);
}

//----------------------------------  Get the running average.
inline double 
ChanBit::getAverage() const {
    if (mNWords == 0) return 0.0;
    return mAverage/mNWords;
}

//----------------------------------  Get the running average length
inline double 
ChanBit::getAvgLen() const {
    if (mNFrames == 0) return 0.0;
    return mNWords/mNFrames;
}

//----------------------------------  Get the sigma
inline double 
ChanBit::getSigma() const {
    if (mNWords == 0) return 0.0;
    double Avg = mAverage/mNWords;
    double Var = mSigma/mNWords - Avg*Avg;
    if (Var <= 0) return 0.0;
    return ::sqrt(Var);
}

inline void 
ChanBit::setDebug(int debug) {
    mDebug = debug;
}

inline void 
ChanBit::setReported(bool card) {
    mReported = card;
}

#endif // __CINT__

#endif // BITTEST_HH
