/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_XWRITER_HH
#define XSIL_XWRITER_HH

#include <time.h>
#include <iostream>
#include <string>

namespace xsil {

/**  The %Xwriter class formats data for XML output. It provides checks
  *  of the XML syntax.
  *  @memo    XML writer class.
  *  @author  J. Zweizig
  *  @version 1.1; Modified 22-Feb-2000.
  *  @ingroup IO_xsil
  */
class Xwriter {
public:
    /**  Construct an XML file writer to format data to an output stream.
      *  @memo XML writer constructor.
      *  @param ostr Output stream to receive formatted XML.
      */
    Xwriter(std::ostream& ostr);

    /**  XML writer destructor.
      *  @memo destructor.
      */
    ~Xwriter(void);

    /**  Output specified XML text to the stream.
      *  @memo Write text string.
      *  @param data Text to be written to the output stream.
      */
    void text(const std::string& data);

    /**  End of an output line.
      */
    void endLine(void);

    /**  Get the current value of the indentation enable flag.
      *  @brief Get the indent enable flag.
      *  @return Indent enable flag value.
      */
    bool getIndentEnable(void) const;

    /**  Set the indentation enable flag to the specified value.
      *  @brief Set the indent enable flag.
      *  @param torf Indent enable flag value.
      */
    void setIndentEnable(bool torf);

    /**  Write an xsil tag to the output file. A list of attribute names 
      *  and associated values are written inside the tag block. Attribute
      *  names with null value pointers are not written. The tag name is
      *  added to the %Xwriter tag list for validation.
      *  @memo Write a tag.
      *  @param tagname Pointer to the tag name.
      *  @param nattr   Number of attributes to be included in the tag.
      *  @param attr    List of pointers to attribute names.
      *  @param value   List of pointers to attribute values.
      */
    void Tag(const char* tagname, int nattr=0, const char** attr=0, 
	     const char** value=0);

    /**  Write a dataless xsil tag to the output file. A list of attribute 
      *  names and associated values are written inside the tag block. 
      *  Attribute names with null value pointers are not written. The 
      *  \<tag/\> syntax is used and no end tag block will be generated.
      *  @memo Write a dataless tag.
      *  @param tagname Pointer to the tag name.
      *  @param nattr   Number of attributes to be included in the tag.
      *  @param attr    List of pointers to attribute names.
      *  @param value   List of pointers to attribute values.
      */
    void TagNData(const char* tagname, int nattr=0, const char** attr=0, 
	     const char** value=0);

    /**  Write an integer value to the output stream.
      *  @memo Write an integer.
      *  @param N value to be written.
      */
    void Integer(int N);

    /**  Write the \</name\> end tag to the output 
      *  stream. The specified tag name must match that of the most recent 
      *  unclosed tag.
      *  @memo write a close tag.
      *  @param tagname Name of tag to be closed.
      */
    void endTag(const char* tagname);

    /**  Synchronize the buffer to the output file by flushing buffered data
      *  to the output stream.
      *  @memo Flush output data.
      */
    void sync(void);

    /**  Set the document type text to a specified string.
      *  \brief Set the document type text.
      *  \param doc Document type string.
      */
    void setDocType(const std::string& doc);

    /**  Write the XML document header.
      *  @memo Write the document header.
      */
    void writeHeader(void);

    /**  Copy in the ligo_lw definition.
      *  @memo Copy in the ligo_lw definition.
      */
    void defineLigoLW(void);

private:
    void indent(void);

private:
    static const int MAXDEPTH=128;
    /**  Pointer to the output stream to receive the XML document.
      *  @memo Output Stream.
      */
    std::ostream*  mStream;

    /**  Current nesting depth of the output stream.
      *  Current nesting depth.
      */
    int       mDepth;

    /**  Number of characters written on the current line.
      *  @memo current line position.
      */
    int       mCursor;

    /**  List of tags for each nested structure.
      *  @memo Tag list.
      */
    const char* mNest[MAXDEPTH];

    /**  Document type text
      */
    std::string mDocType;

    /**  Document type text
      */
    bool mHeaderOK;

    /**  Enable indentation.
      */
    bool mIndentEbl;


};     // xsil::Xwriter

inline bool
xsil::Xwriter::getIndentEnable(void) const {
    return mIndentEbl;
}


}      // namespace xsil

#endif  //   XSIL_XWRITER_HH
