#include "PConfig.h"
#include "xsil/Xwriter.hh"
#include <cmath>
#include <cstring>

using namespace std;

//======================================  XML writer constructor
xsil::Xwriter::Xwriter(ostream& ostr) {
    mStream = &ostr;
    mDepth  = 0;
    mCursor = 0;
    //mDocType= "SYSTEM \"http://www.cacr.caltech.edu/ligo/XSIL/LIGO_LW.dtd\"";
    mDocType= "SYSTEM \"http://ldas-sw.ligo.caltech.edu/doc/ligolwAPI/html/ligolw_dtd.txt\"";
    mHeaderOK = false;
    mIndentEbl = true;
}

//======================================  XML writer destructor
xsil::Xwriter::~Xwriter() {
}

void
xsil::Xwriter::writeHeader(void) {
    if (!mHeaderOK) {
        (*mStream) << "<?xml version=\"1.0\"?>" << endl;
	(*mStream) << "<!DOCTYPE LIGO_LW " << mDocType << ">" << endl;
	mHeaderOK = true;
    }
}


//======================================  Write text
void 
xsil::Xwriter::text(const string& data) {
    unsigned int j(0), l(data.size());
    bool indentok = true;
    for (unsigned int i=0 ; i < l ; i++) {
        if (data[i] == '\n') {
	    if (indentok) indent();
	    (*mStream) << data.substr(j, i-j);
	    endLine();
	    indentok = true;
	    j=i+1;
	} else if (data[i] == '\\' && i+1 < l && data[i+1] == 'n') {
	    if (indentok) indent();
	    (*mStream) << data.substr(j, i-j);
	    endLine();
	    i++;
	    j=i+1;
	    indentok = false;
	}
    }
    if (j < l) {
        if (indentok) indent();
	(*mStream) << data.substr(j);
	mCursor += l - j;
    }

}

//======================================  Write text
void 
xsil::Xwriter::endLine(void) {
    (*mStream) << endl;
    mCursor = 0;
}

//======================================  Write begin tag
void 
xsil::Xwriter::Tag(const char* tagname, int nattr, const char* attr[], 
		   const char* value[]) {
    indent();
    (*mStream) << "<" << tagname;
    mCursor += 1 + strlen(tagname);

    //----------------------------------  Write attributes... xml requires
    //                                    that the attribute values be quoted.
    for (int i=0 ; i<nattr ; i++) {
        if (value[i] && strlen(value[i])) {
	    (*mStream) << " " << attr[i] << "=\"" << value[i] << "\"";
	    mCursor += 4 + strlen(attr[i]) + strlen(value[i]);
	}
    }
    (*mStream) << ">";
    mCursor++;
    mNest[mDepth++] = tagname;
}

//======================================  Write begin tag
void 
xsil::Xwriter::TagNData(const char* tagname, int nattr, 
			const char* attr[], const char* value[]) {
    indent();
    (*mStream) << "<" << tagname;
    for (int i=0 ; i<nattr ; i++) {
        if (*(value[i])) {
	    (*mStream) << " " << attr[i] << "=\"" << value[i] << "\"";
	}
    }
    (*mStream) << "/>" << endl;
    mCursor = 0;
}

//======================================  Write end tag
void 
xsil::Xwriter::endTag(const char* tagname) {
    if (!mDepth) {
        cerr << "Error extra end tag: </" << tagname << ">" << endl;
	return;
    }
    mDepth--;
    if (string(mNest[mDepth]) != string(tagname)) {
        cerr << "Error: end tag </" << tagname << "> doesn't match tag <" 
	     << mNest[mDepth] << ">." << endl;
    }
    indent();
    (*mStream) << "</" << tagname << ">" << endl;
    mCursor = 0;
}

//======================================  Write an integer.
void 
xsil::Xwriter::Integer(int N) {
    indent();
    (*mStream) << N;
    mCursor += 1 + int(log10(double(N)));
}

//======================================  Spew everything to the output file
void
xsil::Xwriter::setDocType(const string& x) {
    mDocType = x;
}
//======================================  Spew everything to the output file
void
xsil::Xwriter::sync(void) {
#ifdef __GNU_STDC_OLD
    mStream->rdbuf()->sync();
#endif
}

//======================================  Indent at the start of a line
void
xsil::Xwriter::indent(void) {
    writeHeader();
    if (!mCursor && mDepth && mIndentEbl) {
        for (int i=0 ; i<mDepth ; i++) (*mStream) << "  ";
	mCursor = 2 * mDepth;
    }
}

//======================================  Define the LIGO_LW document type
void
xsil::Xwriter::defineLigoLW(void) {
    string d;
    d  = "[\n";
    d += "<!ELEMENT LIGO_LW ((LIGO_LW|Comment|Param|Table|Array|Stream|IGWDFrame|AdcData|Time)*)>\n";
    d += "<!ATTLIST LIGO_LW\n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Comment (#PCDATA)>\n";
    d += "\n";
    d += "<!ELEMENT Param (#PCDATA|Comment)*>\n";
    d += "<!ATTLIST Param \n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type CDATA #IMPLIED\n";
    d += "          Unit CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Table (Comment?,Column*,Stream?)>\n";
    d += "<!ATTLIST Table \n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Column EMPTY>\n";
    d += "<!ATTLIST Column\n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type CDATA #IMPLIED\n";
    d += "          Unit CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Array (Dim*,Stream?)>\n";
    d += "<!ATTLIST Array \n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Dim (#PCDATA)>\n";
    d += "<!ATTLIST Dim \n";
    d += "          Name  CDATA #IMPLIED\n";
    d += "          Units CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Stream (#PCDATA)>\n";
    d += "<!ATTLIST Stream \n";
    d += "          Name      CDATA #IMPLIED\n";
    d += "          Type      (Remote|Local) \"Local\"\n";
    d += "          Delimiter CDATA \",\"\n";
    d += "          Encoding  CDATA #IMPLIED\n";
    d += "          Content   CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT IGWDFrame ((Comment|Param|Time|Detector|AdcData|LIGO_LW|Stream?|Array)*)>\n";
    d += "<!ATTLIST IGWDFrame \n";
    d += "          Name CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Detector ((Comment|Param|LIGO_LW)*)>\n";
    d += "<!ATTLIST Detector \n";
    d += "          Name CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT AdcData ((AdcData|Comment|Param|Time|LIGO_LW|Array)*)>\n";
    d += "<!ATTLIST AdcData \n";
    d += "          Name CDATA #IMPLIED>\n";
    d += "\n";
    d += "<!ELEMENT Time (#PCDATA)>\n";
    d += "<!ATTLIST Time \n";
    d += "          Name CDATA #IMPLIED\n";
    d += "          Type (GPS|Unix|ISO-8601) \"ISO-8601\">\n";
    d += "]";
    setDocType(d);
}

void
xsil::Xwriter::setIndentEnable(bool torf) {
    mIndentEbl = torf;
}
