#include "xsil/Xreader.hh"
#include "xsil/array.hh"
#include "xsil/ligolw.hh"
#include "xsil/param.hh"
#include "xsil/stream.hh"
#include "xsil/xsil_time.hh"
#include "xsil/table.hh"
#include <iostream>
#include <stdexcept>
#include <memory>
#include <cstdlib>

using namespace std;

namespace xsil {

  //===================================  Construct a reader
  Xreader::Xreader(std::istream& istr) 
    : mStream(istr), mLexr(stNStates), mDebug(0)
  {
      lxSetup();
      readHeader();
  }

  //====================================  Xml reader destructor
  Xreader::~Xreader(void) {
  }

  //====================================  Set up the lxr tables
  void
  Xreader::lxSetup(void) {
    //----------------------------------  Define group codes.
    Translate<char> TrTable(kInvChar);
    TrTable.set_alpha(kAlpha);
    TrTable.set_numer(kNumer);
    TrTable.set_type(" \t\n\r", kSpace);
    mLexr.setTable(TrTable);

    int Ret  = 1<<lxr::kReturn;
    int NoSv = 1<<lxr::kNoSave;
    int RUG  = Ret + NoSv + (1<<lxr::kUnget);

    //----------------------------------  Start state
    mLexr.addTransition(stStart, lxr::kChar,    '<',    0,    stTag);
    mLexr.addTransition(stStart, lxr::kChar,    '>',    Ret,  tkEndTag);
    mLexr.addTransition(stStart, lxr::kChar,    '=',    Ret,  tkEqual);
    mLexr.addTransition(stStart, lxr::kChar,    '&',    0,    stEscape);
    mLexr.addTransition(stStart, lxr::kChar,    '?',    0,    stQMark);
    mLexr.addTransition(stStart, lxr::kChar,    '"',    NoSv, stInString);
    mLexr.addTransition(stStart, lxr::kChar,    '/',    0,    stETag);
    mLexr.addTransition(stStart, lxr::kChar,    '.',    0,    stNumFix);
    mLexr.addTransition(stStart, lxr::kChar,    '-',    0,    stNumInt);
    mLexr.addTransition(stStart, lxr::kGroup,   kNumer, 0,    stNumInt);
    mLexr.addTransition(stStart, lxr::kGroup,   kSpace, NoSv, stStart);
    mLexr.addTransition(stStart, lxr::kDefault, 0,      0,    stInWord);
    mLexr.addTransition(stStart, lxr::kEOF,     0,      Ret+NoSv, tkEnd);

    //----------------------------------  In word
    mLexr.addTransition(stInWord, lxr::kGroup,   kSpace, Ret+NoSv, tkWord);
    mLexr.addTransition(stInWord, lxr::kChar,    '=',    RUG,      tkWord);
    mLexr.addTransition(stInWord, lxr::kChar,    '<',    RUG,      tkWord);
    mLexr.addTransition(stInWord, lxr::kChar,    '>',    RUG,      tkWord);
    mLexr.addTransition(stInWord, lxr::kChar,    '?',    RUG,      tkWord);
    mLexr.addTransition(stInWord, lxr::kChar,    '"',    RUG,      tkWord);
    //mLexr.addTransition(stInWord, lxr::kChar,    '/',    RUG,      tkWord);
    mLexr.addTransition(stInWord, lxr::kDefault, 0,      0,        stInWord);
    mLexr.addTransition(stInWord, lxr::kEOF,     0,      Ret+NoSv, tkWord);
    
    //----------------------------------  In a quoted string
    mLexr.addTransition(stInString, lxr::kChar,    '"', Ret+NoSv, tkString);
    mLexr.addTransition(stInString, lxr::kDefault, 0,   0,        stInString);

    //----------------------------------  In an integer number.
    mLexr.addTransition(stNumInt,   lxr::kGroup,   kNumer, 0,        stNumInt);
    mLexr.addTransition(stNumInt,   lxr::kChar,    '.',    0,        stNumFix);
    mLexr.addTransition(stNumInt,   lxr::kChar,    'e',    0,        stNumExp);
    mLexr.addTransition(stNumInt,   lxr::kDefault, 0,      RUG,      tkNumber);
    mLexr.addTransition(stNumInt,   lxr::kEOF,     0,      Ret+NoSv, tkNumber);

    //----------------------------------  In a fixed point number
    mLexr.addTransition(stNumFix,   lxr::kGroup,   kNumer, 0,        stNumFix);
    mLexr.addTransition(stNumFix,   lxr::kChar,    'e',    0,        stNumExp);
    mLexr.addTransition(stNumFix,   lxr::kDefault, 0,      RUG,      tkNumber);
    mLexr.addTransition(stNumFix,   lxr::kEOF,     0,      Ret+NoSv, tkNumber);

    //----------------------------------  In a scientific notation number
    mLexr.addTransition(stNumExp,   lxr::kGroup,   kNumer, 0, stNumSExp);
    mLexr.addTransition(stNumExp,   lxr::kChar,    '+',    0, stNumSExp);
    mLexr.addTransition(stNumExp,   lxr::kChar,    '-',    0, stNumSExp);

    //----------------------------------  In a scientific notation number
    mLexr.addTransition(stNumSExp, lxr::kGroup,   kNumer, 0, stNumSExp);
    mLexr.addTransition(stNumSExp, lxr::kDefault, 0,      RUG,      tkNumber);
    mLexr.addTransition(stNumSExp, lxr::kEOF,     0,      Ret+NoSv, tkNumber);

    //----------------------------------  Found a '?'
    mLexr.addTransition(stQMark, lxr::kChar,    '>', Ret, tkCloseHead);
    mLexr.addTransition(stQMark,  lxr::kDefault, 0,   RUG, tkQMark);

    //----------------------------------  Found a '/'
    mLexr.addTransition(stETag, lxr::kChar,    '>', Ret, tkEndSTag);
    mLexr.addTransition(stETag, lxr::kDefault, 0,   0, stInWord);

    //----------------------------------  In an open bracket
    mLexr.addTransition(stTag, lxr::kChar,    '/', Ret, tkOpenETag);
    mLexr.addTransition(stTag, lxr::kChar,    '!', Ret, tkOpenCTag);
    mLexr.addTransition(stTag, lxr::kChar,    '?', Ret, tkOpenHead);
    mLexr.addTransition(stTag, lxr::kDefault, 0,   RUG, tkOpenTag);

    //----------------------------------  In an escape sequence (&xxx.)
    mLexr.addTransition(stEscape, lxr::kChar,    ';', Ret, tkEscape);
    mLexr.addTransition(stEscape, lxr::kDefault, 0,   0,   stEscape);
    mLexr.check(false);
  }

  //====================================  Read and check the xml header
  void
  Xreader::readHeader(void) {
    if (mDebug) cout << "Read header... ";
    //----------------------------------  Read the xml header
    string TknString;
    LxrTokens tkn = getToken(TknString);
    if (tkn != tkOpenHead) throw runtime_error("File isn't xml");
    tkn = getToken(TknString);
    if (tkn!=tkWord || TknString!="xml") throw runtime_error("File isn't xml");
    while (tkn != tkEnd && tkn != tkCloseHead) {
        tkn = getToken(TknString);
    }
    if (mDebug) cout << "Done" << endl;
  }

  //=====================================  Read a complete document
  xobj*
  Xreader::readDoc(void) {
      if (mDebug) cout << "Read Document" << endl;
      xobj* rc(0);
      bool indoc = true;
      while (indoc) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkEnd:
	      indoc = false;
	      break;
	  case tkOpenCTag:
	    {
	      bool incmt(true);
	      while (incmt) {
		  tkn = getToken(token);
		  switch (tkn) {
		  case tkEndTag:
		      incmt = false;
		      break;
		  default:
		      break;
		  }
	      }
	      break;
	    }
	  case tkOpenTag:
	      rc = readObject();
	      break;
	  default:
	      throw runtime_error("Xreader: Unexpected token");
	  }
      }
      if (mDebug) cout << "Finished document" << endl;
      return rc;
  }

  //=====================================  lower case a string
  static void
  to_lower(string& s) {
      long N=s.size();
      for (long i=0; i<N; i++) s[i] = tolower(s[i]);
  }

  //=====================================  Read a Tag
  void 
  Xreader::readEndTag(const string& name) {
      string token;
      LxrTokens tkn = getToken(token);
      if (tkn != tkWord && tkn != tkString) {
	  throw runtime_error("readEndTag: Invalid tag type.");
      }
      to_lower(token);
      if (name != token) throw runtime_error("readEndTag: End tag mismatch.");
      tkn = getToken(token);
      if (tkn != tkEndTag) {
	  throw runtime_error("readEndTag: Invalid tag type.");
      }

  }

  //=====================================  Read an xml object
  xobj*
  Xreader::readObject(void) {

      //---------------------------------  Get the open tag
      string tagtype;
      attr_list attrs;
      int tagonly = readTag(tagtype, attrs);
      if (mDebug) {
	  cout << "Xreader: Process tag <" << tagtype;
	  for (const_attr_iter i=attrs.begin(); i != attrs.end(); ++i) {
	      cout << " " << i->first << "=\"" << i->second << "\"";
	  }
	  cout << "> ... ";
      }

      //---------------------------------  Go figure
      xobj* robj(0);
      if (tagtype == "array") {
	  robj = readArray(attrs, tagonly);
      } else if (tagtype == "ligo_lw") {
	  robj = readLigoLW(attrs, tagonly);
      } else if (tagtype == "param") {
	  robj = readParam(attrs, tagonly);
      } else if (tagtype == "stream") {
	  robj = readStream(attrs, tagonly);
      } else if (tagtype == "table") {
	  robj = readTable(attrs, tagonly);
      } else if (tagtype == "time") {
	  robj = readTime(attrs, tagonly);
      } else if (tagtype == "xsil") {
      } else {
	  throw runtime_error("Xreader::readObject: Unrecognized tag");
      }

      //--------------------------------  Check the end tag
      if (!tagonly) readEndTag(tagtype);
      return robj;
  }

  //====================================  get a pointer to an attribute
  static const char*
  getAttr(const attr_list& a, const string& n) {
      const_attr_iter p=a.find(n);
      if (p == a.end()) return "";
      return p->second.c_str();
  }

  //====================================  Read an array
  array*
  Xreader::readArray(const attr_list& attrs, int tagonly) {
     unique_ptr<array> r;
     r.reset(new array(getAttr(attrs, "name"), getAttr(attrs, "type")));

     bool inblock=true;
     while (inblock) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkOpenTag:
	    {
	      attr_list nxattr;
	      string tagname;
	      int tonly = readTag(tagname, nxattr);
	      if (tagname == "stream") {
		  unique_ptr<Stream> s;
		  s.reset(readStream(nxattr, tonly));
		  r->setStream(*s);
		  readEndTag(tagname);
	      } else if (tagname == "dim") {
		  string dtoken;
		  tkn = getToken(dtoken);
		  if (tkn != tkNumber) {
		      throw runtime_error("readArray: Numeric token expected");
		  }
		  int nDim = strtol(dtoken.c_str(), 0, 0);
		  tkn = getToken(dtoken);
		  if (tkn != tkOpenETag) {
		      throw runtime_error("readArray: Unexpected token");
		  }
		  readEndTag(tagname);
		  r->addDim(nxattr["name"].c_str(), nDim);
	      } else {
		  throw runtime_error("Xreader::readArray: Unexpected tag");
	      }
	    }
	    break;
	  case tkOpenETag:
	      inblock = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readArray: Unexpected token");
	  }
      }
      return r.release();
  }

  //====================================  Read an array
  param*
  Xreader::readParam(const attr_list& attrs, int tagonly) {
      unique_ptr<param> r;
      r.reset(new param(getAttr(attrs, "name"), getAttr(attrs, "type")));
      bool inpar(true);
      string par;
      while (inpar) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkWord:
	  case tkNumber:
	      if (!par.empty()) par += " ";
	      par += token;
	      break;
	  case tkString:
	      if (!par.empty()) par += " ";
	      par += "\"";
	      par += token;
	      par += "\"";
	      break;
	  case tkOpenETag:
	      inpar = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readParam: Unexpected token");
	  }
      }
      r->setUnit(getAttr(attrs, "unit"));
      r->setValue(par);
      return r.release();
  }

  //====================================  Read a stream
  ligolw*
  Xreader::readLigoLW(const attr_list& attrs, int tagonly) {
      unique_ptr<ligolw> obj(new ligolw(getAttr(attrs, "name"), 
				      getAttr(attrs, "type")));
      bool inblock = true;
      while (inblock) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkOpenTag:
	      obj->addObject(readObject());
	      break;
	  case tkOpenETag:
	      inblock = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readLigoLW: Unexpected token");
	  }
      }
      return obj.release();
  }

  //====================================  Read a stream
  Stream*
  Xreader::readStream(const attr_list& attrs, int tagonly) {
      unique_ptr<Stream> obj(new Stream(getAttr(attrs, "name"), 
				      getAttr(attrs, "type")));
      bool instream = true;
      while (instream) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkWord:
	  case tkString:
	      obj->Add(token);
	      break;
	  case tkNumber:
	      obj->append(token);
	      break;
	  case tkOpenETag:
	      instream = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readArray: Unexpected token");
	  }
      }
      return obj.release();
  }

  //====================================  Read a table
  table*
  Xreader::readTable(const attr_list& attrs, int tagonly) {
     unique_ptr<table> r;
     r.reset(new table(getAttr(attrs, "name"), getAttr(attrs, "type")));

     bool inblock=true;
     while (inblock) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkOpenTag:
	    {
	      attr_list nxattr;
	      string tagname;
	      int tonly = readTag(tagname, nxattr);
	      if (tagname == "stream") {
		  unique_ptr<Stream> s;
		  s.reset(readStream(nxattr, tonly));
		  r->setStream(*s);
		  readEndTag(tagname);
	      } else if (tagname == "column") {
		  if (!tonly) throw runtime_error("Column has data");
		  r->addColumn(nxattr["name"].c_str(), 
			       nxattr["type"].c_str(), 
			       nxattr["unit"].c_str());
	      } else {
		  throw runtime_error("Xreader::readTable: Unexpected tag");
	      }
	    }
	    break;
	  case tkOpenETag:
	      inblock = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readTable: Unexpected token");
	  }
      }
      return r.release();
  }

  //====================================  Read an array
  xsil_time*
  Xreader::readTime(const attr_list& attrs, int tagonly) {
      unique_ptr<xsil_time> r;
      r.reset(new xsil_time(getAttr(attrs, "name"), getAttr(attrs, "type")));
      const char* pDim = getAttr(attrs, "Dim");
      int nDim(1);
      if (*pDim) nDim = strtol(pDim, 0, 0);
      bool inpar(true);
      int wdCount(0);
      string par;
      while (inpar) {
	  string token;
	  LxrTokens tkn = getToken(token);
	  switch (tkn) {
	  case tkWord:
	  case tkString:
	  case tkNumber:
	      if (!par.empty()) par += " ";
	      par += token;
	      wdCount++;
	      break;
	  case tkOpenETag:
	      inpar = false;
	      break;
	  default:
	      throw runtime_error("Xreader::readTime: Unexpected token");
	  }
      }
      if (wdCount != nDim) throw runtime_error("readParam: wordcount != nDim");
      r->setValue(par);
      return r.release();
  }

  //=====================================  Read a Tag
  int 
  Xreader::readTag(string& name, attr_list& attrs) {
      string token;
      LxrTokens tkn = getToken(token);
      if (tkn != tkWord && tkn != tkString) throw runtime_error("Invalid tag type");
      to_lower(token);
      name = token;
      while (1) {
	  string nxattr, nxvalue;
	  tkn = getToken(nxattr);
	  if (tkn == tkEndTag || tkn == tkEndSTag) break;
	  if (tkn == tkEnd) throw runtime_error("Xreader: Unexpected EOF");
	  if (tkn != tkWord && tkn != tkString ) {
	      throw runtime_error("Xreader: Unexpected token");
	  }

	  tkn = getToken(nxvalue);
	  if (tkn != tkEqual) throw runtime_error("Xreader: Unexpected token");

	  tkn = getToken(nxvalue);
	  if (tkn != tkWord && tkn != tkString ) {
	      throw runtime_error("Xreader: Unexpected token");
	  }
	  to_lower(nxattr);
	  attrs[nxattr] = nxvalue;
      }
      if (tkn == tkEndSTag) return 1;
      return 0;
  }

  //=====================================  Get a token.
  Xreader::LxrTokens 
  Xreader::getToken(std::string& token) {
      LxrTokens rc = LxrTokens(mLexr.token(mStream, token));
      if (mDebug > 2) cout << "Xreader: token |" << token << "| type =" 
			   << rc << endl;
      return rc;
  }

  void
  Xreader::setDebug(int lvl) {
      mDebug = lvl;
  }

}      // namespace xsil
