/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef ISMBUF_HH_
#define ISMBUF_HH_
//
//   Shared memory stream input buffer class
//
#include "PConfig.h"
// #define ISMBUF_TRACE

#if defined(ISMBUF_TRACE)
  #include <string>
  #include <iostream>
#endif
#if defined(__GNU_STDC_OLD)
   #include <streambuf.h>
#else
   #include <iostream>
   #include <streambuf>
#endif

class LSMP_CON;

/**  This class interfaces the shared memory buffer manager to a standard
  *  data stream. iSMbuf is based on the standard stream buffer (streambuf)
  *  class and implements the virtual buffering methods of that class. In
  *  addition to the standard methods, the iSMbuf class contains waitBuf
  *  and setBCount methods to wait for asynchronous data and to set a 
  *  reserved buffer count, respectively. The buffer functions ignore the
  *  arrival of signals, except in the waitBuf method.
  *  @memo Shared memory stream input buffer class.
  *  @author John Zweizig
  *  @version 1.1; last modified March 6, 2008
  *  @ingroup IO_lsmp
  */
class iSMbuf : public std::streambuf {
public:

#ifdef __GNU_STDC_OLD
    typedef std::ios::open_mode openmode_type;
    typedef std::ios::seek_dir  seekdir_type;
#else
    /** STL IOS open mode ennumerator.
      */
    typedef std::ios::openmode  openmode_type;

    /** STL IOS seek direction ennumerator.
      */
    typedef std::ios::seekdir   seekdir_type;
#endif

    /**  Get the offset type from the STL streambuf.
      */
    typedef std::streamoff off_type;

    /**  Get the position type from the STL streambuf.
      */
    typedef std::streampos pos_type;

public:
    /**  Default constructor.
      *  The default constructor builds an unopened shared memory stream 
      *  buffer.
      */
    iSMbuf();

    /**  Construct an opened shared memory stream buffer.
      *  A shared memory stream buffer is created and attached (if possible) 
      *  to the specified shared memory partition. The partition to be 
      *  attached is specified by 'partition'. The mode flag is kept for
      *  compatibility with the file stream and must be set to ios::in.
      */
    iSMbuf(const char *partition, openmode_type mode=std::ios::in);

    /**  Shared memory stream buffer destructor.
      */
    ~iSMbuf();

    /**  Get the current buffer event id.
      */
    int eventid(void) const;

    /**  Test for latest buffer in partition.
     */
    bool latest(void) const;

    /**  Attach a shared memory partition to a stream buffer.
      *  The named partition is attached to the stream buffer.
      */
    iSMbuf* open(const char *partition, openmode_type mode=std::ios::in);

    /**  This parameter controls what fraction of the data read into 
      *  the shared memory will be copied to the data stream and what
      *  part of the shared memory resources can be held allocated the 
      *  stream. 'nbuf' == 0 indicates data is taken as requested.
      *  'nbuf' == -1 indicates that all data written to the shared 
      *  memory must be copied to the data stream. 'nbuf'==N indicates
      *  that up to N buffers (frames) of data should be held until the 
      *  stream is ready for them.
      *  @memo Set the number of segments to be reserved by this consumer.
      */
    void setBCount(int nbuf);

    /**  Test whether data are available. If data are available, timedWait
      *  returns true. If data are not immediately available and the timeout
      *  argument is zero, waitData returns false. Otherwise waitBuf waits
      *  data to arrive or the semaphore wait to be interrupted by a signal.
      *  If the timeout is negative, the wait lasts indefinitely.
      *  @memo Wait for data.
      *  @param timeout Maximum wait time. If timeout<0 the wait is indefinite.
      *  @return true if data available.
      */
    bool timedWait(double timeout);

    /**  Test whether data are available. If data are available, waitBuf
      *  returns true. If data are not immediately available and the nowait
      *  argument is true, waitData returns false. Otherwise waitBuf waits
      *  for data to arrive or the semaphore wait to be interrupted by a 
      *  signal.
      *  @memo Wait for data.
      *  @param nowait If true, waitBuf returns immediately.
      *  @return true if data available.
      */
    bool waitBuf(bool nowait=false);

    /**  Release a shared memory buffer.
      *  @memo Release shared memory buffer.
      */
    void relse(void);

    /**  Standard streambuf allocation function.
      */
    int doallocate(void);

    /**  Standard streambuf overflow function.
      */
    int overflow(int c) {return std::streambuf::traits_type::eof();}

    /**  Standard streambuf read function.
      */
    int underflow(void);


    /**  Standard streambuf seek offset function.
      */
    pos_type seekoff(off_type off, seekdir_type dir, openmode_type mode);

    /**  Standard streambuf seek position function.
      */
    pos_type seekpos(pos_type off, openmode_type mode);

    /**  Standard streambuf synchronization function.
      */
    int sync(){return 0;}

  private:
    void setptrs(int ix=0);
#ifndef __GNUG__
  void setb(char* b, char*eb, int a=0) {}
#endif


  private:
    ///                  Pointer to the shared memory consumer object.
    LSMP_CON*   mConsumer;
    ///                  Pointer to the current shared memory segment.
    const char* mBuffer;
    ///                  Data length in the current shared memory segment.
    int         mLength;
#ifdef ISMBUF_TRACE
  public:
    void pTrace(void) const {mTrace.pTrace();}
  private:
    class trace {
    #define maxTrace 8
    public:
        trace() : nTrace(0) {}
        void tracepoint(const std::string& funct, int arg1=0, int arg2=0);
        void pTrace(void) const;
        void clear(void);
    private:
        int nTrace;
        std::string mFuncName[maxTrace];
        int mArg1[maxTrace];
        int mArg2[maxTrace];
    } mTrace;
#endif
};

//======================================  Inline implementation of seekoff
#ifndef __GNU_STDC_OLD
inline iSMbuf::pos_type
iSMbuf::seekpos(pos_type off, openmode_type mode) {
    return seekoff(off_type(off), std::ios::beg, mode);
}
#endif

//======================================  Trace point function
#ifdef ISMBUF_TRACE
inline void 
iSMbuf::trace::tracepoint(const std::string& funct, int arg1, int arg2) {
    int inx = nTrace % maxTrace;
    mFuncName[inx] = funct;
    mArg1[inx] = arg1;
    mArg2[inx] = arg2;
    nTrace++;
}

inline void 
iSMbuf::trace::clear(void) {
    nTrace = 0;
}
#endif

#endif  //  ISMBUF_HH_
