/* -*- mode: c++; c-basic-offset: 4; -*- */
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif

#include "PConfig.h"
#include "lsmp.hh"
#include "lsmp_int.hh"
#include "gds_atomic.h"
#include <iostream>
#include <iomanip>
#include <cstdio>
#include <cstring>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <unistd.h>
#include "SysError.hh"

//======================================  PID source
#ifndef P__Linux
#define GET_PID_FROM_PART 1
#define MAX_VIRTUAL_PID 100
#endif

class lsmp_registry {
public:
    lsmp_registry(void);
    ~lsmp_registry(void);
    void add(LSMP* part);
    void remove(LSMP* part);
private:
    static const size_t maxpart=16;
    LSMP* list[maxpart];
};

using namespace std;

static lsmp_registry partition_registry;

#define SEMPROT 0666

//
//    Methods for the LSMP object class.
//
//    Default constructor
//
LSMP::LSMP(void)
  :  error(NoError), accessed(false), kpflg(false), 
     NByte(0), NBuff(0), mLockCount(0) 
{
    partition_registry.add(this);
}

//
//    Copy constructor
//
LSMP::LSMP(LSMP &copy)
  :  error(NoError), accessed(false), kpflg(false), 
     NByte(0), NBuff(0), mLockCount(0) 
{
    partition_registry.add(this);
    _partition = copy._partition;
    if (_partition.is_attached()) {
        pointer = copy.pointer;
	conptr  = copy.conptr;
	bufptr  = copy.bufptr;
	if (!access()) {
	    release();
	    return;
	}
    }
    mLockCount = copy.mLockCount;
}

//
//    Access an existing partition.
//
LSMP::LSMP(const char *name) 
  : error(NoError), accessed(false), kpflg(false), NByte(0), NBuff(0), 
    mLockCount(0)
{
    partition_registry.add(this);
    if (!find(name)) access();
    if (!accessed) release();
}

//
//    Create a partition
//
LSMP::LSMP(const char *name, int nbuf, int size) 
  : error(NoError), accessed(false), kpflg(false), NByte(0), NBuff(0), 
    mLockCount(0) 
{
    partition_registry.add(this);
    if (!find(name)) {
        access();
	if (!accessed) {
	    release();
	} else if (NBuff < nbuf || NByte < size) {
	    release();
	    error = IncompatiblePartition;
	}
    } else if (!make(name, nbuf, size)) {
	access();
    } else {
	release();
    }
}

//
//    Release/Delete a partition
//
LSMP::~LSMP(void) {

    if (accessed) deaccess();
    if (attached()) release();

    //------------------------------  Remove it from the partition registry.
    partition_registry.remove(this);
}

//--------------------------------------------------------------------------
//
//    void LSMP::rqsynch(bool on)
//
//    Enable or disable request synchronization. When request synchronization
//    is enabled, LSMP will not allocate a free buffer to any producer unless
//    one or more consumers are waiting for data.
//
//    Parameters:
//        on    Request synchronization is enabled if on==true and disabled
//              if on==false.
//
//    Returns
//        none
//
//--------------------------------------------------------------------------
void
LSMP::rqsynch(bool on) {
    if (on) pointer->status |= RQSYNCH;
    else    pointer->status &= ~RQSYNCH;
}

//--------------------------------------------------------------------------
//
//    void LSMP::bufmode(int mode)
//
//    Set the buffer release strategy according to mode. 
//
//    Parameter:
//      mode 0  A buffer is released (moved to the free list) any time
//              is not reserved and not in use.
//           1  A buffer is released after it has been seen by at least
//              one consumer and is no longer reserved or in use.
//           2  A buffer is not released except if it is unreserved, 
//              not in use and needed to fill a producer request.
//           3  All incoming buffers are reserved for all consumers.
//              The buffer is entered into the free list after being
//              released by all consumers.
//
//    Note that this function is meant to be used only at startup time. 
//    Using this function to turn off buffer scavaging (i.e. to change 
//    from mode 2) after buffers have been used can cause the partition 
//    lose some or all of the would-be available buffers.
//
//--------------------------------------------------------------------------
void
LSMP::bufmode(int mode) {
    switch(mode) {
    case 0:
        pointer->status &= ~(SCAVAGE | RELBUF | EXPOSE);
        pointer->status |= RELBUF;
	break;
    case 1:
        pointer->status &= ~(SCAVAGE | RELBUF | EXPOSE);
	break;
    case 2:
        pointer->status &= ~(SCAVAGE | RELBUF | EXPOSE);
        pointer->status |= SCAVAGE;
	break;
    case 3:
        pointer->status &= ~(SCAVAGE | RELBUF | EXPOSE);
        pointer->status |= EXPOSE | RELBUF;
	break;
    default:
        cerr << "LSMP::bufmode: Invalid mode (" << mode << ") request" << endl;
    }
}

//--------------------------------------------------------------------------
//
//    bool LSMP::keep(bool flag);
//
//    Don't delete the partition on release
//
//    Parameters
//      flag   If true the partition will be kept on release.
//
//    Returns:
//      none
//
//--------------------------------------------------------------------------
void 
LSMP::keep(bool flag) {
    kpflg = flag;
}

//--------------------------------------------------------------------------
//
//    bool LSMP::lock(bool flag);
//
//    Lock the shared memory partition into memory. On sun platforms, 
//    the function will attempt to change the effective user id to root 
//    while performing this operation.
//
//    Parameters
//      flag   If true the partition will be lock in memory. If false,
//             the partition will be unlocked.
//
//    Returns:
//      true on error
//
//--------------------------------------------------------------------------
bool
LSMP::lock(bool flag) {
    error = NoError;
    if (!_partition.lock(flag)) {
	error = NoPriviledge;
	return true;
    }
    return false;
}

//--------------------------------------------------------------------------
//
//    const char* LSMP::Error(void);
//
//    Get a text string describing the last error
//
//    Parameters
//      none
//
//    Returns:
//      A pointer to a constant error message string.
//
//--------------------------------------------------------------------------
const char*
LSMP::Error(void) const {
  switch (error) {
  case NoError:
      return "No error";
  case NotAttached:
      return "Partition not attached";
  case IncompatiblePartition:
      return "Existing partition is incompatible with request";
  case CantGetSem:
      return "Semaphore doesn't exist";
  case NoFreeID:
      return "No free partition IDs are available";
  case CantAttach:
      return "Unable to attach partition";
  case CantCreate:
      return "Unable to create a new partition";
  case CantCreateSem:
      return "Unable to create a new semaphore";
  case NoPriviledge:
      return "No Priviledge for attempted operation";
  case BadVersion:
      return "Incompatible version number";
  case NoPartition:
      return "Named partition not found";
}
  return "Unidentifier error";
}

//--------------------------------------------------------------------------
//
//    std::string LSMP::name(void);
//
//    Get the name of the attached partition.
//
//--------------------------------------------------------------------------
std::string
LSMP::name(void) const {
    if (!attached()) return "";
    string::size_type l = strlen(pointer->name);
    if (l >= LSMP_LNAME) return string(pointer->name, LSMP_LNAME);
    return pointer->name;
}

//--------------------------------------------------------------------------
//
//    bool LSMP::find(const std::string& name);
//
//    Find and attach an existing shared memory partition. Set the LSMP 
//    pointers to indicate where it can be found.
//
//    Parameters
//      name   String used to identify the parition to be attached.
//
//    Returns:
//      true   An error occurred while finding or attaching the partition.
//      false  The requested operation completed.
//
//--------------------------------------------------------------------------
bool 
LSMP::find(const std::string& tname) {

    //---------------------------------  Don't build partition a second time
    if (attached() || tname.empty()) return true;

   //---------------------------------  Loop over valid IDs
    for (int id=0; id < LSMP_MAXSHM; id++) {
	if (_partition.attach(LSMP_SHMBASE + id)) {
	    pointer = reinterpret_cast<LSMP_global*>(_partition.ref());
	    if (name() == tname) return false;
	    _partition.release();
	}
    }
    error = NoPartition;
    return true;
}

//-------------------------------------------------------------------------
//
//    bool LSMP::access(void)
//
//    Bump the use count on a partition
//
//-------------------------------------------------------------------------
bool 
LSMP::access() {
    if (!attached() || accessed) return false;
    if (pointer->version != LSMP_VERSION) {
        error = BadVersion;
	return false;
    }
#ifdef USE_BUILTIN_ATOMICS
    ATOMIC_POSTINC(pointer->gbl_count);
#else
    while (!gate(true));
    pointer->gbl_count++;
    gate(false);
#endif
    conptr = pointer->refCons(0);
    bufptr = pointer->refBuffer(0);
    NBuff  = pointer->nbuf;
    NByte  = pointer->lbuf;
    accessed = true;
    return true;
}

//-------------------------------------------------------------------------
//
//    void deaccess()
//
//    Decrement the use count of an attached partition and mark the handle
//    as not in use.
//
//    Parameters:
//      none
//
//-------------------------------------------------------------------------
void 
LSMP::deaccess() {
    if (accessed && pointer->gbl_count) {
#ifdef USE_BUILTIN_ATOMICS
	ATOMIC_PREDEC(pointer->gbl_count);
#else
	while(!gate(true));
	pointer->gbl_count--;
	gate(false);
#endif
    }
    accessed = false;
    return;
}

//-------------------------------------------------------------------------
//
//    bool make(const char *name, int nbuf, int size)
//
//    A shared memory partition is created and initialized.
//
//    Parameters:
//        name    Shared memory partition name field.
//        nbuf    number of buffers to be created in the shared memory 
//                partition.
//        size    Lenght of the shared memory buffers.
//
//    Return values:
//        true    An error occurred while attempting to create the partition.
//        false   The partition was created as requesteed.
//
//-------------------------------------------------------------------------
bool 
LSMP::make(const char *name, int nbuf, int size) {
    int id, idmax, minsiz;
    union semun setv;

    //---------------------------------  Find a free external ID
    error = CantAttach;
    if (attached()) return true;
    error = NoError;
    minsiz = sizeof(LSMP_global) + LSMP_MAXCONS*sizeof(LSMP_consbk);
    idmax  = LSMP_SHMBASE + LSMP_MAXSHM;
    for (id = LSMP_SHMBASE ; id < idmax ; id++) {
        if (!_partition.exists(id, minsiz)) break;
    }

    if (id >= idmax) {
        error = NoFreeID;
	return true;
    }

    //---------------------------------  Create the partition.
    minsiz += nbuf * (size + sizeof(LSMP_buffer));
    if (!_partition.create(id, minsiz) ) {
	perror("LSMP_make gds_shm::create() failed");
        error = CantCreate;
	return true;
    }

    //---------------------------------  Initialize
    pointer = reinterpret_cast<LSMP_global*>(_partition.ref());
    pointer->gbl_count = 0;
    pointer->nbuf = nbuf;
    pointer->lbuf = size;
    for (int i=0; i < LSMP_LMASK; i++) {
	pointer->con_semid[i] = -1;
    }

    //---------------------------------  Fill in partition fields
    strncpy(pointer->name, name, LSMP_LNAME);
    pointer->name[LSMP_LNAME-1] = 0;
    pointer->gbl_semid = semget(IPC_PRIVATE, gbl_nsems, IPC_CREAT + SEMPROT);
    if (pointer->gbl_semid < 0) {
	perror("LSMP_make - error allocating global sems");
	error = CantCreateSem;
	return true;
    }
    pointer->version = LSMP_VERSION;
    pointer->status  = SCAVAGE | RELBUF;

    setv.val = 0;
    semctl(pointer->gbl_semid, gbl_full, SETVAL, setv);
    setv.val = nbuf;
    semctl(pointer->gbl_semid, gbl_empty, SETVAL, setv);

    pointer->ncons = 0;
    pointer->conmask.zero();
    pointer->conresrv.zero();
    for (int i=0; i < LSMP_LMASK; i++) {
	pointer->con_semid[i] = semget(IPC_PRIVATE, LSMP_CONSPWD, 
				       IPC_CREAT+SEMPROT);
	if (pointer->con_semid[i] < 0) {
	    perror("LSMP_make - error allocating consumer sems");
	    error = CantGetSem;
	    return true;
	}
    }
    setv.val = 1;
    semctl(pointer->gbl_semid, gbl_gate, SETVAL, setv);
    pointer->free.init();
    pointer->full.init();

    //-------------------------------------  Initialize the consumer blocks
    LSMP_consbk* pcon = pointer->refCons(0);
    for (int i=0 ; i < LSMP_MAXCONS; i++) {
	setv.val = 0;
	semctl(pointer->con_semid[CON_WORD(i)], CON_BIT(i), SETVAL, setv);
	pcon->mxbuf = 0;
	pcon++;
    }

    //----------------------------------------- Set up the buffer area
    LSMP_buffer* pbuf = pointer->refBuffer(0);
    long boff = long(pointer->refBuffer(nbuf)) - long(pointer);
    for (int i=0; i < nbuf; i++) {
	pbuf[i].use_count = 0;
	pbuf[i].boff   = boff;
	boff += pointer->lbuf;
	pbuf[i].reserve_mask.zero();
	pbuf[i].seen_mask.zero();
	pbuf[i].ldata     = 0;
	pbuf[i].trig      = 0;
	pbuf[i].evt_count = 0;
	pointer->free.link(pbuf,i);
    }
    return false;
}

//---------------------------------------------------------------------------
//
//    LSMP::release()
//
//    Release a shared memory partition. 
//
//    Parameters:
//      none
//
//    Side effects
//      The partition is released. The share memory will be deleted if it 
//      has not been attached by any other process.
//
//---------------------------------------------------------------------------
void 
LSMP::release(void) {
    union semun arg4;

    //----------------------------------  See if partition is in use by others
    if (!attached()) return;
    if (accessed) deaccess();
    if (!pointer->gbl_count && !kpflg) {

        //------------------------------  Remove the semaphores
        arg4.val = 0;
        if (pointer->gbl_semid >= 0) {
	    semctl(pointer->gbl_semid, 0, IPC_RMID, arg4);
	}
	for (int i=0 ; i<LSMP_LMASK ; i++) {
	    int sem_i = pointer->con_semid[i];
	    if (sem_i >= 0) semctl(sem_i, 0, IPC_RMID, arg4);
	}

	//------------------------------  Release the global partition
	_partition.release(true);
    } else {
        _partition.release();
    }
}

//---------------------------------------------------------------------------
//
//    int free_consumer()
//
//        Release an allocated consumer ID. All buffers reserved for the 
//        consumer are released before deleting the consumer.
//
//    Parameter:
//        none
//
//    Returns
//        0       Specified consumer was released as requested
//        -1      Error occurred while releasing consumer.
//
//---------------------------------------------------------------------------
int 
LSMP::free_consumer(int icon) {

    //----------------------------------  Get access to the global region.
    //
    //  Note that the gate must be closed even when using atomic operators
    //  in case this function interrupts another process accessing the 
    //  partition.
    while (!gate(true));

    //----------------------------------  Mark the consumer not used
    if (pointer->conmask.tclr(icon)) pointer->ncons--;

    //----------------------------------  Clean up full buffer list
    int next = -1;
    for (int i=pointer->full.head ; i>=0 ; i=next) {
        LSMP_buffer* pB = bufptr+i;
	next = pB->link;

        //------------------------------  Release buffers reserved by consumer
        if (pB->reserve_mask[icon]) {
	    pB->reserve_mask.clear(icon);
	    if (pB->seen_mask[icon]) pB->use_count--;
	}

	//------------------------------  Free unused buffers
	if (!pB->inUse()) {

	    //--------------------------  Scavage mode. Wake up the producer
	    struct sembuf sbuf;
	    if (testFlag(SCAVAGE)) {
	        sbuf.sem_num = gbl_empty;
		sbuf.sem_flg = 0;
		sbuf.sem_op  = 1;
		semop(pointer->gbl_semid, &sbuf, 1);

	    //---------------------------  Buffer seen or release mode. 
	    } else if (pB->seen() || testFlag(RELBUF)) {
	        pointer->full.remove(bufptr, i);
		pointer->free.link(bufptr, i);
		sbuf.sem_num   = gbl_empty;
		sbuf.sem_flg   = 0;
		sbuf.sem_op    = 1;
		semop(pointer->gbl_semid, &sbuf, 1);
	    }
	}
    }
    pointer->conresrv.clear(icon);

    //----------------------------------  Release the global region.
    gate(false);
    return 0;
}

//---------------------------------------------------------------------------
//
//    bool gate(bool close)
//
//    Allocate / Free the global partition control area for exclusive 
//    access.
//
//    Parameter:
//        close != 0 indicates that exclusive access to the global 
//                   partition is requested. Exclusive access is 
//                   ceded if close == 0;
//
//    Returns
//        true       successful completion.
//        false      An interrupt occurred while waiting for exclusive
//                   access.
//
//    Exceptions
//        SysError   A system error occurred whilein gate()
//
//---------------------------------------------------------------------------
bool 
LSMP::gate(bool close) const {
    struct sembuf sbuf;

    sbuf.sem_num = gbl_gate;
    if (close) sbuf.sem_op  = -1;
    else       sbuf.sem_op  =  1;
    sbuf.sem_flg = SEM_UNDO | IPC_NOWAIT;
    if (semop(pointer->gbl_semid, &sbuf, 1) < 0) {
        if (errno == EINTR)       return false;
	else if (errno != EAGAIN) throw SysError("LSMP::gate semop failed");
	mLockCount++;
	sbuf.sem_flg &= ~IPC_NOWAIT;
	if (semop(pointer->gbl_semid, &sbuf, 1) < 0) {
	    if (errno == EINTR) return false;
	    else                throw SysError("LSMP::gate semop failed");
	}
    }
    return true;
}

//---------------------------------------------------------------------------
//
//    bool dump()
//
//    Print out the status of an attached/accessed partition.
//
//---------------------------------------------------------------------------
void 
LSMP::dump(int flg) const {
    dump(cout, flg);
}


inline void
addWord(string& s, const char* c) {
    if (!s.empty()) s += ",";
    s += c;
}

//---------------------------------------------------------------------------
//
//    bool dump()
//
//    Print out the status of an attached/accessed partition.
//
//---------------------------------------------------------------------------
void 
LSMP::dump(ostream& Out, int flg) const {
    bool noheader = (flg & 0x0200) != 0;

    //-------------------------------- Check that we have accessed partition
    if (!accessed) {
        Out << "LSMP::dump() Global partition not accessed" << endl;
	return;
    }
    LSMP_global* pgbl = pointer;

    //-------------------------------- Print a header
    if ((flg & 1) != 0) {
        Out << endl;
	Out << "            Event Manager Partition - " << pgbl->name <<endl;
	Out << endl;
    }

    //-------------------------------- Print global control data
    if ((flg & 2) != 0) {
	if (!noheader) {
	    Out << "Global Control Data" << endl;
	    Out << "-------------------" << endl;
	}
	Out << "  version:           " << pgbl->version/100 << "."
	    << setfill('0') << setw(2) << pgbl->version%100 
	    << setfill(' ') << endl;
	Out << "  status:            " << getStat("pflags") << endl;
	Out << "  global use count:  " << pgbl->gbl_count << endl;
	Out << "  global sem id:     " << pgbl->gbl_semid << endl;
	Out << "  # buffers:         " << pgbl->nbuf << endl;
	Out << "  buffer length:     " << pgbl->lbuf << endl;
	Out << "  first full buffer: " << pgbl->full.head << endl;
	Out << "  last full buffer:  " << pgbl->full.tail << endl;
	Out << "  first free buffer: " << pgbl->free.head << endl;
	Out << "  #consumers:        " << pgbl->ncons << endl;
	Out << "  consumer sem id:";
	for (int i=0 ; i<LSMP_LMASK ; i++) Out << "   " << pgbl->con_semid[i];
	Out << endl;
	Out << "  consumer mask:     " << hex << pgbl->conmask.collect() 
	    << endl;
	Out << endl;
    }

    //----------------------------------  Print buffer status
    if ((flg & 4) != 0) {    
	if (!noheader) {
	    Out << "Buffer Status" << endl;
	    Out << "-------------" << endl;
	    Out <<" id status reserved seen  use   ldata   link  t-mask   count"
		<< endl;
	}
	LSMP_buffer* pbuf = bufptr;
	for (int i=0 ; i<pgbl->nbuf ; i++) {
	    Out << setw(3) << i 
		<< hex 
		<< setw(5) << pbuf->status
		<< setw(9) << pbuf->reserve_mask.collect()
		<< setw(7) << pbuf->seen_mask.collect()
		<< dec 
		<< setw(5) << pbuf->use_count
		<< setw(9) << pbuf->ldata 
		<< setw(5) << pbuf->link
		<< setw(9) << pbuf->trig
		<< setw(8) << pbuf->evt_count << endl;
	    pbuf++;
	}
        Out << endl;
    }

    //----------------------------------  Print consumer status
    if ((flg & 8) != 0) {
        if (pgbl->ncons) {
	    if (!noheader) {
		Out << "Consumer Status" << endl;
		Out << "---------------" << endl;
		Out << " id  pid  max-bf   trig   #segs   read  flags" << endl;
	    }
	    LSMP_consbk* pcon = conptr;
	    for (int i=0 ; i < LSMP_MAXCONS ; i++) {
	        if (pgbl->conmask.test(i)) {
		    string txtstr;
		    if (pcon->flags & READALL) addWord(txtstr,"readall");
		    if (pcon->flags & EVWAIT)  addWord(txtstr,"wait");
		    if (txtstr.empty()) txtstr = " - ";
		    Out << setw( 3) << i 
			<< setw( 6) << (int) pcon->pid
			<< setw( 5) << pcon->mxbuf 
			<< hex 
			<< setw(10) << pcon->trig_mask
			<< dec 
			<< setw( 6) << pcon->seg_ctr
			<< setw( 9) << pcon->seg_tot
			<< " " << txtstr << endl;
		    if ((flg & 0x100) != 0) {
			Out << "    min_time " << pcon->min_time
			    << " min_sep " << pcon->min_sep
			    << " time_ctr " << pcon->time_ctr
			    << " skip_ctr " << pcon->skip_ctr << endl;
		    }
		}
		pcon++;
	    }
	} else if (!noheader) {
	    Out << "No Consumers" << endl 
		<< "------------" << endl;
	}
    }
    return;
}

//---------------------------------------------------------------------------
//
//    int buffer_addr(int ibuf)
//
//    Get the address of buffer ibuf
//
//    Parameters:
//        ibuf    id of buffer to be accessed.
//
//    Returns:
//        0       Buffer doesn't exist
//        >0      Address of requested buffer.
//
//---------------------------------------------------------------------------
char*
LSMP::buffer_addr(int ibuf) const {
    if (!accessed)            return (char*) 0;
    if (ibuf<0 || ibuf>NBuff) return (char*) 0;
    return (char*) pointer + bufptr[ibuf].boff;
}

//---------------------------------------------------------------------------
//
//    int buffer_length(int ibuf)
//
//    Get the length of the data in a specified buffer
//
//    Parameters:
//        ibuf    id of buffer to be accessed.
//
//    Returns:
//        >=0     Data length.
//
//---------------------------------------------------------------------------
int
LSMP::buffer_length(int ibuf) const {
    if (!accessed)            return 0;
    if (ibuf<0 || ibuf>NBuff) return 0;
    return bufptr[ibuf].ldata;
}

//---------------------------------------------------------------------------
//
//    string getStat(const string& par)
//
//    Get a partition parameter
//
//    Parameters:
//        par     parameter name
//
//    Returns:
//        parameter value as a string
//
//---------------------------------------------------------------------------
static string 
puti(int n) {
    const char* num = "0123456789";
    if (n < 0)     return string("-") += puti(-n);
    if (n/10 == 0) return string(num + n, 1);
    return puti(n/10) += string(num + (n%10), 1);
}

bool
LSMP::testFlag(int x) const {
    return pointer && (pointer->status & x) != 0;
}

string
LSMP::getStat(const string& par) const {
    string r;
    if (!attached()) {
        r = "0";
    } else if (par == "exists") {
        r = "1";
    } else if (par == "name") {
        r = pointer->name;
    } else if (par == "pflags") {
        if (testFlag(RELBUF))  addWord(r, "release");
	if (testFlag(SCAVAGE)) addWord(r, "scavage");
	if (testFlag(RQSYNCH)) addWord(r, "synch");
	if (testFlag(EXPOSE))  addWord(r, "expose");
	if (r.empty()) r = "-";
    } else if (par == "shmid") {
        r = puti(_partition.ident());
    } else if (par == "version") {
        r = puti(pointer->version);
    } else if (par == "nbuf") {
        r = puti(pointer->nbuf);
    } else if (par == "lbuf") {
        r = puti(pointer->lbuf);
    } else if (par == "ncons") {
        r = puti(pointer->ncons);
    } else if (par == "use_count") {
        r = puti(pointer->gbl_count);
    } else if (par == "buffer_tot") {
        int btot = 0;
	for (int i=0 ; i<pointer->nbuf ; i++) btot += bufptr[i].evt_count;
        r = puti(btot);
    } else if (par == "maxcons") {
        r = puti(LSMP_MAXCONS);
    } else if (par == "semglobal") {
        r = puti(pointer->gbl_semid);
    } else if (par == "fullbuf") {
        gate(true);
        r = puti(pointer->full.length(bufptr));
        gate(false);
    } else if (par == "freebuf") {
        gate(true);
        r = puti(pointer->free.length(bufptr));
        gate(false);
    } else if (par == "usedbuf") {
        gate(true);
        int N(0);
	for (int i=pointer->full.head ; i>=0 ; i=bufptr[i].link) {
	    if (bufptr[i].inUse()) N++;
	}
        r = puti(N);
        gate(false);
    } else if (par == "last_ID") {
        if (pointer->full.empty()) r = "-0";
        else r = puti(bufptr[pointer->full.tail].data_ID);
    } else if (par == "vstart") {
        if (pointer->full.empty()) {
	    r = "-0";
        } else {
	    int vstart = bufptr[pointer->full.tail].data_ID;
	    for (int i=0; i<pointer->nbuf; i++) vstart -= bufptr[i].evt_count;
	    r = puti(vstart);
	}
    } else {
        r = "Undefined parameter";
    }
    return r;
}

//======================================  Zero use count to zap partition.
pid_t
LSMP::my_procid(void) const {
    pid_t pid = getpid();
#ifdef GET_PID_FROM_PART
    if (pid < MAX_VIRTUAL_PID && attached()) {
	pid = _partition.my_pid();
    }
#endif
    return pid;
}

//======================================  Zero use count to zap partition.
void
LSMP::Zeuss(const char* name) {
    if (attached()) {
        cerr << "Can't zero user count - partition is attached." << endl;
    } else if (find(name)) {
        cerr << "Can't find partition: " << name << endl;
    } else if (! _partition.owner()) {
        cerr << "Not owner of partition: " << name << endl;
    } else {
        pointer->gbl_count = 0;
	kpflg              = false;
    }
}

//======================================  LSMP registry constructor
lsmp_registry::lsmp_registry(void) {
    for (size_t i=0; i<maxpart; ++i) list[i] = NULL;
}

//======================================  LSMP registry destructor
lsmp_registry::~lsmp_registry(void) {
    for (size_t i=0; i<maxpart; ++i) {
	if (list[i]) {
	    list[i]->release();
	    list[i] = NULL;
	}
    }
}

//======================================  Add LSMP to registry
void
lsmp_registry::add(LSMP* new_ptr) {
    for (size_t i=0; i<maxpart; ++i) {
	if ( __sync_bool_compare_and_swap(list+i, NULL, new_ptr) ) break;
    }
}

//======================================  Remove LSMP from registry
void
lsmp_registry::remove(LSMP* old_ptr) {
    for (size_t i=0; i<maxpart; ++i) {
	if ( __sync_bool_compare_and_swap(list+i, old_ptr, NULL) ) break;
    }
}
