/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_TRANSLATOR_HH
#define LMSG_TRANSLATOR_HH
#include "lmsg/MsgTypes.hh"

namespace lmsg {
    class Buffer;
    class MsgAddr;
    class Grinder;

    /**  Translator is a table driven translator for the Ligo DMT message 
      *  facility. The translation table is defined by a format string.
      *  Structures are translated to or from message buffers using the
      *  import and export methods. A single extended structure may be 
      *  translated to/from multiple message by setting the multibuffer 
      *  flag before translation.
      *  @brief Message data translator.
      *  @author John Zweizig
      *  @version 1.1; last modified March 4, 2008
      *  @ingroup IO_lmsg
      */
    class Translator {
    public:
	/**  Construct a translator and optionally set the translation table 
	  *  from the format argument. The format is specified as described 
	  *  for the \c Define() method.
	  *  @brief Constructor.
	  *  @param fmt Message data descriptor.
	  */
	Translator(const char* fmt=0);

	/**  Destroy a translator object.
	  *  @brief Destructor.
	  */
	~Translator(void);

	/**  Parse a description of the structure to be translated to or 
	  *  from a message. The structure is described by a null-terminated
	  *  character string. The format characters are:
	  *  <table>
	  *    <tr><td>Code</td><td>Data Type</td></tr>
	  *    <tr><td>A</td>  <td>char</td></tr>
	  *    <tr><td>B</td>  <td>char</td></tr>
	  *    <tr><td>D</td>  <td>double</td></tr>
	  *    <tr><td>F</td>  <td>float</td></tr>
	  *    <tr><td>I</td>  <td>int</td></tr>
	  *    <tr><td>NI</td> <td>network int</td></tr>
	  *    <tr><td>NS</td> <td>network short</td></tr>
	  *    <tr><td>R</td>  <td>MsgAddr</td></tr>
	  *    <tr><td>S</td>  <td>short</td></tr>
	  *    <tr><td>UB</td> <td>unswapped byte</td></tr>
	  *    <tr><td>UI</td> <td>unswapped int</td></tr>
	  *    <tr><td>US</td> <td>unswapped short</td></tr>
	  *  </table>
	  *  @brief Define the message format.
	  *  @param fmt Structure format description.
	  */
	error_type Define(const char* fmt);

	/**  Copy and translate a structure to the specified message 
	  *  buffer.
	  *  @brief Export structure to message.
	  *  @param in     Pointer to structure to be translated
	  *  @param length Length of structure.
	  *  @param b      Reference to output buffer.
	  *  @return lmsg status return
	  */
	error_type Export(const void* in, size_type length, Buffer& b);

	/**  Copy and translate a structure from the specified message 
	  *  buffer.
	  *  @brief Export structure to message.
	  *  @param b      Reference to constant input buffer.
	  *  @param out    Pointer to structure to receive data.
	  *  @param maxlen Length of structure.
	  *  @param len    Pointer to variable to receive length of copied data.
	  *  @return lmsg stats return
	  */
	error_type Import(const Buffer& b, void* out, size_type maxlen, 
			  size_type* len);

	/**  Reset status flags.
	  *  @brief reset translator.
	  */
	void Reset(void);

	/**  Test the Header OK flag.
	  *  @brief test header OK.
	  */
	bool isHeaderOK(void) const;

	/**  Test the multi-buffer flag.
	  */
	bool isMultiBuffer(void) const;

	/**  Set the header OK flag.
	 */
	void setHeaderOK(bool yesno=true);

	/**  Set the multibuffer flag.
	 */
	void setMultiBuffer(bool yesno=true);

	/**  Set the "keep variable names" flag. This uses more space, but
	  *  allows formatted message dumps.
	  *  @brief Set keep name flag.
	  *  @param yesno New flag value.
	  */
	void setKeepNames(bool yesno=true);

    private:
	enum tblop {
	    o_AFmt=1, o_BFmt, o_DFmt, o_FFmt, o_IFmt, o_NIFmt, o_NSFmt, o_RFmt,
	    o_SFmt, o_UBFmt, o_UIFmt, o_USFmt,
	    o_Name, o_RepB, o_RepS, o_RepI, o_RepD,
	    o_EndB, o_EndS, o_EndI, o_EndD, o_End, o_LTable};
	size_type fmtcd(const char* name, size_type len);
	size_type syntax(const char* fmt, size_type inx);

    private:
#define HEADEROKFLAG    1
#define MULTIBUFFERFLAG 2
	size_type mFlags;
	size_type mOffset;
	size_type mOpCd;
	size_type mLevel;
#define STACKSIZE 8
	size_type mLvPtr[STACKSIZE];
	size_type mLvRpt[STACKSIZE];
#define NO_TRANS ((short*)-1)
	short* mFormat;
	bool   mKeepNames;

    private:
	static bool   sTabSet;
	static size_type sTabLength[o_LTable];
	static size_type sTabAlign[o_LTable]; 
    };

} //  namespace lmsg

//======================================  Inline methods
inline bool
lmsg::Translator::isHeaderOK(void) const {
    return (mFlags & HEADEROKFLAG) != 0;
}

inline void 
lmsg::Translator::setHeaderOK(bool yesno) {
    if (yesno) mFlags |=  HEADEROKFLAG;
    else       mFlags &= ~HEADEROKFLAG;
}

inline bool
lmsg::Translator::isMultiBuffer(void) const {
    return (mFlags & MULTIBUFFERFLAG) != 0;
}

inline void 
lmsg::Translator::setMultiBuffer(bool yesno) {
    if (yesno) mFlags |=  MULTIBUFFERFLAG;
    else       mFlags &= ~MULTIBUFFERFLAG;
}

inline void 
lmsg::Translator::setKeepNames(bool yesno) {
    mKeepNames = yesno;
}

#endif   // LMSG_TRANSLATOR_HH
