/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "lmsg/Buffer.hh"
#include "lmsg/Grinder.hh"
#include "lmsg/MsgAddr.hh"
#include "lmsg/Translator.hh"
#include "lmsg/ErrorList.hh"
#include <iostream>

using namespace lmsg;
using namespace std;

//--------------------------------------  Aligndet local class
template<class T>
class AlignDet {
public:
    typedef unsigned long mask_t;
    AlignDet(void) {}
    ~AlignDet(void) {}
    mask_type getAlign(void) const;
private:
    double start;
    char   a;
    T      b;
};

template<class T>
inline lmsg::mask_type
AlignDet<T>::getAlign(void) const {
    return mask_t(&b) - sizeof(a) - mask_t(&a);
}


//--------------------------------------  Translator static tables
bool Translator::sTabSet(false);
size_type Translator::sTabLength[o_LTable];
size_type Translator::sTabAlign[o_LTable]; 

//--------------------------------------  Byte ordering/alignment detection
static Grinder g;

//--------------------------------------  Translator member methods
Translator::Translator(const char* fmt) 
  : mFlags(0), mLevel(0), mFormat(0)
{
    if (!sTabSet) {
        for (int i=0 ; i<o_LTable ; i++) {
	    sTabLength[i] = 0;
	    switch(i) {

	    //--------------------------  Byte data items
	    case o_AFmt:
	    case o_BFmt:
	    case o_UBFmt:
	        sTabLength[i] = sizeof(char);
	    case o_RepB:
	    case o_EndB:
	        sTabAlign[i] = AlignDet<char>().getAlign();
		break;

	    //--------------------------  Double-word (8-byte) data items
	    case o_DFmt:
	        sTabLength[i] = sizeof(double);
	    case o_RepD:
	    case o_EndD:
	        sTabAlign[i] = AlignDet<double>().getAlign();
		break;

	    //--------------------------  Word (4-byte) data items
	    case o_FFmt:
	    case o_IFmt:
	    case o_UIFmt:
	    case o_NIFmt:
	        sTabLength[i] = sizeof(int);
	    case o_RepI:
	    case o_EndI:
	        sTabAlign[i] = AlignDet<int>().getAlign();
		break;

	    //--------------------------  MsgAddr data
	    case o_RFmt:
	        sTabLength[i] = sizeof(MsgAddr);
		sTabAlign[i] = AlignDet<int>().getAlign();
		break;

	    //--------------------------  Short (2-byte) data items
	    case o_NSFmt:
	    case o_SFmt:
	    case o_USFmt:
	        sTabLength[i] = sizeof(short);
	    case o_RepS:
	    case o_EndS:
	        sTabAlign[i] = AlignDet<short>().getAlign();
		break;

	    //--------------------------  Other fields
	    default:
	        sTabAlign[i] = 0;
	    }
	}
        sTabSet = true;
    }
    if (fmt) Define(fmt);
}

Translator::~Translator(void) {
    if (mFormat) delete[] mFormat;
}

size_type 
Translator::Define(const char* fmt) {
    char name[64], *ptmp;
    int i;
    short temp[512], *ptlvl[STACKSIZE];
    size_type lvtyp[STACKSIZE];

/*--------------------------------------- loop over format characters.    */
    size_type level = 0;
    short* ptbl  = temp;
    ptlvl[0] = ptbl;
    lvtyp[0] = 0;
    size_type lname = 0;
    long   rept  = -1;

    for (size_type inx=0 ; fmt[inx] ;) {
        //------------------------------  Copy a name string
        char xch = fmt[inx];
        if ((xch>='a' && xch<='z') || (xch>='A' && xch<='Z')) {
	    while ((fmt[inx]>='a' && fmt[inx]<='z') || 
		   (fmt[inx]>='A' && fmt[inx]<='Z') ||
		   (fmt[inx]>='0' && fmt[inx]<='9') || 
		   (fmt[inx]=='_')) {
	        name[lname++] = fmt[inx++];
	    }
	    name[lname]=0;
	    break;

	//------------------------------  Ditto the repeat count
	} else if (xch>='0' && xch<='9') {
	    rept = 0;
	    while (fmt[inx]>='0' && fmt[inx]<='9') {
	        rept *= 10;
		rept += fmt[inx++]-'0';
	    }

       //-------------------------------  Name defines a name
	} else if (xch == ':') {
	    if (rept >= 0) return syntax(fmt, inx);
	    if (!lname) break;
	    if (mKeepNames) {
		*ptbl++ = o_Name;
		*ptbl++ = lname;
		ptmp = (char *) ptbl;
		for (size_type k=0 ; k<lname ; k++) *ptmp++ = name[k];
		if (lname & 1) *ptmp++ = 0;
		ptbl = (short *) ptmp;
	    }
	    lname=0;
	    inx++;

	//------------------------------  Start a repeat group
	} else if (xch == '(') {
	    if (lname) return syntax(fmt, inx);
	    level++;
	    ptlvl[level] = ptbl;
	    lvtyp[level] = 0;
	    *ptbl++ = o_RepB;
	    if (rept < 0) rept = 1;
	    *ptbl++ = rept;
	    rept = -1;
	    inx++;

	//------------------------------  End of repeat group
	} else if (xch == '(') {
	    if (!level) return syntax(fmt, inx); // verify non-zero level

	    i = fmtcd(name, lname);         /* add format entry */
	    if (i) {
		if (sTabAlign[i] > lvtyp[level]) lvtyp[level] = sTabAlign[i];
		*ptbl++ = i;
		lname = 0;
		if (rept < 0) rept = 1;
		*ptbl++ = rept;
		rept = -1;}
	    else if (lname || (rept>=0)) {
		return syntax(fmt, inx);
	    }

	    switch (lvtyp[level]) {             /* fill in rep operators */
	    case 1:
		*ptbl++ = o_EndS;
		*ptlvl[level] =o_RepS;
		break;
	    case 3:
		*ptbl++ = o_EndI;
		*ptlvl[level] =o_RepI;
		break;
	    case 7:
		*ptbl++ = o_EndD;
		*ptlvl[level] =o_RepD;
		break;
	    default:
		*ptbl++ = o_EndB;
		*ptlvl[level] =o_RepB;}

	    if (lvtyp[level] > lvtyp[level-1]) lvtyp[level-1] = lvtyp[level];
	    level--;
	    inx++;

	//------------------------------  End of format item
	} else if (xch == ',') {
	    i = fmtcd(name, lname);
	    if (i) {
		if (sTabAlign[i] > lvtyp[level]) lvtyp[level] = sTabAlign[i];
		*ptbl++ = i;
		lname = 0;
		if (rept < 0) rept = 1;
		*ptbl++ = rept;
		rept = -1;
	    } else if (lname || (rept>=0)) {
		return syntax(fmt, inx);
	    }
	    inx++;

	//------------------------------  Syntax error
	} else {
	    return syntax(fmt, inx);
	}
    }                   //--------------  End of loop over tokens.

    //----------------------------------  Check that the parens balence.
    if (level) return syntax(fmt, strlen(fmt));

    //----------------------------------  See if there's one more format.
    i = fmtcd(name, lname);
    if (i) {
	if (sTabAlign[i] > lvtyp[level]) lvtyp[level] = sTabAlign[i];
	*ptbl++ = i;
	lname = 0;
	if (rept < 0) rept = 1;
	*ptbl++ = rept;
    } else if (lname) {
	return syntax(fmt, strlen(fmt));
    }
    *ptbl++ = o_End;

    //----------------------------------  Allocate a table. Copy in the data.
    size_type lentbl = (long(ptbl) - long(temp))/sizeof(short);
    mFormat = new short[lentbl];
    memcpy(mFormat, temp, lentbl*sizeof(short));
    return OK;
}

inline unsigned long
ByteOffset(const void* x, unsigned long a) {
    return -long(x) & a;
}

size_type 
Translator::Export(const void* lcl, size_type lclen, Buffer& b) {
    size_type  wc, rc, rep;

    //----------------------------------  Make sure the format is valid
    if (!mFormat) {
        if (lclen > 0) return Invalid;
	else {
	    b.setDataLength(0);
	    return OK;
	}
    }

/*
 *    get pointers to the input and output areas. Copy the message header
 *    in local format! set up length pointers.
 */
    const char* in   = (const char*) lcl;
    size_type      nbyt = lclen;
    size_type      itab = 0;
    bool        cont = true;

    if (isHeaderOK()) {
        nbyt   -= mOffset;
	in     += mOffset;
	itab    = mLvPtr[mLevel];
	rep     = mLvRpt[mLevel];
	mLevel -= 1;
    } else {
        mOffset = 0;
	setHeaderOK();
	mLevel    =0;
	mLvPtr[0] = 0;
	mLvRpt[0] = 0;
	rep       = 0;
	if (mFormat == NO_TRANS) mOpCd = o_UBFmt;
	else                     cont = false;
    }

    char*  out  = b.getDataAddr();
    size_type obyt = b.getDataSize();

/*
 *    Loop over the format entries until a termination condition occurs
 *    Possible conditions are:
 *      loop = 1  end statement in format
 *      loop = 2  output buffer full, incomplete transfer.
 *      loop = 3  mismatch between data and format
 *                - fractional data word left in input.
 *                - insufficient input data for format.
 */
    int  loop = 0;
    bool skip = false;
    while (!loop) {
        //------------------------------  Get the input format type.
	if (!cont) mOpCd = mFormat[itab++];

	//------------------------------  Bump input & output ptrs and adjust
        //                                byte counts for the alignment.
        if (nbyt) {
	    size_type bias  = ByteOffset(out, 3);
	    out  += bias;
	    obyt -= bias;

	    bias  = ByteOffset(in, sTabAlign[mOpCd]);
	    in   += bias;
	    nbyt -= bias;
	}

	//--------------------------------  Calculate the number of words.
	size_type lwd = sTabLength[mOpCd];
	if (lwd) {
	    if (!cont) rep = mFormat[itab++];
	    if (rep <= 0) {
	        if (nbyt <= obyt) wc = nbyt / lwd;
		else              wc = obyt / lwd;
	    } else if (nbyt <= obyt) {
	        wc = nbyt / lwd;
		if (rep <= wc) wc   = rep;
	    } else {
	        wc = obyt / lwd;
		if (rep < wc) wc   = rep;
	    }
	} else {
	    rep = 0;
	    wc = 0;
	}

	switch(mOpCd) {

	//------------------------------  Byte data... check the alignment
	case o_AFmt:
	case o_BFmt:
	case o_UBFmt:
	    memcpy(out, in, wc);
	    in  += wc;
	    out += wc;
	    break;

	//------------------------------  Short data
	case o_SFmt:
	case o_USFmt:
	    export_format_grinder.SwapOut(reinterpret_cast<const short*>(in), 
					  out, wc);
	    break;

	//------------------------------  Unswappped short data
	case o_NSFmt:
	    memcpy(out, in, wc*sizeof(short));
	    in    += wc*sizeof(short);
	    out   += wc*sizeof(short);
	    break;

	//------------------------------  Integer data  (float assumed same)
	case o_IFmt:
	case o_UIFmt:
	case o_FFmt:
	    export_format_grinder.SwapOut(reinterpret_cast<const int*>(in), 
					  out, wc);
	    in    += wc*sizeof(int);
	    out   += wc*sizeof(int);	    
	    break;

	//------------------------------  MsgAddr data
	case o_RFmt:
	    for (size_type i=0; i<wc; ++i) {
		memcpy(out, in, 4);
		out += 4;
		in  += 4;
		export_format_grinder.SwapOut(reinterpret_cast<const short*>(in),
					      out, 2);
		out += 4;
		in  += 4;
	    }
	    break;

	//------------------------------  Unswapped integer data
	case o_NIFmt:
	    memcpy(out, in, wc*sizeof(int));
	    in  += wc*sizeof(int);
	    out += wc*sizeof(int);
	    break;

	//------------------------------  Double floating point data
	case o_DFmt:
	    export_format_grinder.SwapOut(reinterpret_cast<const double*>(in), 
					  out, wc);
	    in    += wc*sizeof(double);
	    out   += wc*sizeof(double);
	    break;

	//------------------------------  Move up a level
	case o_RepB:
	case o_RepS:
	case o_RepI:
	case o_RepD:
	  mLevel++;
	  mLvRpt[mLevel] = mFormat[itab++];
	  mLvPtr[mLevel] = itab;
	  if (!mLvRpt[mLevel] && !nbyt) skip = true;
	  break;

	//------------------------------  End of a level. Do we loop?
	case o_EndB:
	case o_EndS:
	case o_EndI:
	case o_EndD:

	  //----------------------------  Decrement the repeat count.
          //                              down a level if result is 0
	  if (skip && !mLvRpt[mLevel]) {  
	    skip       = false;
	    mLevel -= 1;}

	  else if (!(--mLvRpt[mLevel])) {  
	    mLevel -= 1;}

	  else if ((mLvRpt[mLevel] < 0) && (nbyt <= 0)) {
	    mLevel -= 1;}

	  else {
	    itab = mLvPtr[mLevel];}
	  break;

	//-------------------------------  Ignore a name field
	case o_Name:
	    wc = mFormat[itab++];
	    itab += (wc + 1) >> 1;
	    break;

	//-------------------------------  End of format. Bail out.
	case o_End:
	    loop = 1;
	    break;
	default:
	    return Invalid;
	}

	//-------------------------------  End of the loop over format items.
	//                                 Decrement copied data byte counts.
	if (lwd && !skip) {
	    size_type bias  = wc * lwd;
	    nbyt -= bias;
	    obyt -= bias; // check this!
	    if (rep <= 0) {
	        if (nbyt > 0) 
		    if (obyt > lwd) loop = 3;
		    else            loop = 2;
		else if (mFormat == NO_TRANS) loop = 1;
	    } else if (wc < rep) {
	        if ((nbyt <= 0) || (obyt > lwd)) loop = 3;
		else loop = 2;
	    }
	}
	cont = 0;
    }

    //-----------------------------------  Calculate the export length, 
    //                                     and put it in the local header.
    b.setDataLength(b.getDataSize() - obyt);
    if (loop == 1) 
        if (nbyt > 0)        rc = MisMatch;
        else                 rc = OK;
    else if (loop == 2) 
        if (isMultiBuffer()) rc = Continue;
        else                 rc = MisMatch;
    else                     rc = MisMatch;

    //----------------------------------   Set up the block for a restart
    if (rc == Continue) {
        mOffset = long(in) - long(lcl);
        mLevel += 1;
        mLvPtr[mLevel] = itab;
        mLvRpt[mLevel] = rep - wc;}

    //----------------------------------   Indicate all done.
    else Reset();
    return rc;
}

size_type 
Translator::Import(const Buffer& b, void* lcl, size_type maxlen, 
			 size_type* lclen) {
    size_type wc, rc, rep, itab;

    //----------------------------------  Check that the format is valid
    if (!mFormat) {
        if (b.getDataLength() > 0) return Invalid;
	else {
	    *lclen = 0;
	    return OK;
	}
    }

    //----------------------------------  Get input and output area pointers
    char*       out  = (char *) lcl;
    const char* in   = b.getDataAddr();
    size_type      obyt = maxlen;
    size_type      nbyt = b.getDataLength();
    bool        cont = true;

    if (isHeaderOK()) {
        obyt   -= mOffset;
	out    += mOffset;
	itab    = mLvPtr[mLevel];
	rep     = mLvRpt[mLevel];
	mLevel -= 1;
    } else {
	setHeaderOK();
	itab       = 0;
	mLevel     = 0;
	mLvPtr[0]  = 0;
	mLvRpt[0]  = 0;
	rep        = 0;
	if (mFormat == NO_TRANS) mOpCd = o_UBFmt;
	else                     cont  = false;
    }

/*
 *    Interpret the format table. Loop until an end-condition is raised.
 *    The end conditions are:
 *       1   End token reached in format.
 *       2   Input buffer empty, incomplete transfer.
 *           - variable repeat count used all data, TRN_MULTI set
 *           - not all data from a fixed repeat count were transferred.
 *       3   Output record full.
 *       4   Format mismatch
 *           - Variable repeat count left some data
 *           - Fixed field not completed when data available
 */
    size_type loop = 0;
    bool   skip = false;
    while (!loop) {
        if (nbyt) {
	    size_type bias = ByteOffset(in, 3);  // adjust input alignment
	    in   += bias;
	    nbyt -= bias;
	}

/*
 *    get the op code. Adjust the output pointer and the output byte
 *    count to give the correct alignment to the output field. 
 *    Fill the skipped bytes with zeroes.
 */
	if (!cont) mOpCd = mFormat[itab++];
	size_type lwd = sTabAlign[mOpCd];
	while ((long(out) & lwd) && obyt) {
	    *out++ = 0;
	    obyt--;
	}

	//------------------------------  Calculate number of words.
	lwd = sTabLength[mOpCd];
	if (lwd) {
	  if (!cont) rep = mFormat[itab++];
	  if (rep <= 0) 
	    if (nbyt <= obyt) wc = nbyt / lwd;
	    else              wc = obyt / lwd;
	  else if (nbyt <= obyt) {
	    wc = nbyt / lwd;
	    if (rep <= wc) wc   = rep;}
	  else {
	    wc = obyt / lwd;
	    if (rep < wc) wc   = rep;}
	} else {
	  rep = 0;
	  wc = 0;}

	switch(mOpCd) {

	//------------------------------  Byte data... check the alignment
	case o_AFmt:
	case o_BFmt:
	case o_UBFmt:
	    memcpy(out, in, wc);
	    in  += wc;
	    out += wc;
	    break;

	//------------------------------  Short integer data
	case o_SFmt:
	case o_USFmt:
	    export_format_grinder.SwapIn(in, reinterpret_cast<short*>(out), 
					 wc);
	    in    += wc*sizeof(short);
	    out   += wc*sizeof(short);
	    break;

	//------------------------------  Unswapped Short data
	case o_NSFmt:
	    memcpy(out, in, wc*sizeof(short));
	    in  += wc*sizeof(short);
	    out += wc*sizeof(short);
	    break;

	//------------------------------  Integer data  (float assumed same)
	case o_IFmt:
	case o_UIFmt:
	case o_FFmt:
	    export_format_grinder.SwapIn(in, reinterpret_cast<int*>(out), wc);
	    in    += wc*sizeof(int);
	    out   += wc*sizeof(int);
	    break;

	//------------------------------  Unswapped Integer data
	case o_NIFmt:
	    memcpy(out, in, wc*sizeof(int));
	    in  += wc*sizeof(int);
	    out += wc*sizeof(int);
	    break;

	//------------------------------  MsgAddr data
	case o_RFmt:
	    for (size_type i=0; i<wc; ++i) {
		memcpy(out, in, sizeof(int));
		in    += sizeof(int);
		out   += sizeof(int);
		export_format_grinder.SwapIn(in, reinterpret_cast<short*>(out),
					     2);
		in    += 2*sizeof(short);
		out   += 2*sizeof(short);
	}
	    break;

       //-------------------------------  double data
	case o_DFmt:
	    export_format_grinder.SwapIn(in, reinterpret_cast<double*>(out),
					 wc);
	    in    += wc*sizeof(double);
	    out   += wc*sizeof(double);
	    break;

       //-------------------------------  Move up a level
	case o_RepB:
	case o_RepS:
	case o_RepI:
	case o_RepD:
	    mLevel++;
	    mLvRpt[mLevel] = mFormat[itab++];
	    mLvPtr[mLevel] = itab;
	    if (!nbyt && !mLvRpt[mLevel]) skip = true;
	    break;

	//------------------------------  Move down a level
	case o_EndB:
	case o_EndS:
	case o_EndI:
	case o_EndD:
	    if (skip && !mLvRpt[mLevel]) {
	      skip    = false;
	      mLevel -= 1;}

	    if (!(mLvRpt[mLevel] -= 1)) {
	      mLevel -= 1;}

	    else if ((mLvRpt[mLevel] < 0) &&
		     !isMultiBuffer() && (nbyt <= 0)) {
	      mLevel -= 1;}

	    else {
	      itab = mLvPtr[mLevel];}
	    break;

	//------------------------------  Ignore a name field
	case o_Name:
	    wc = mFormat[itab++];
	    itab += (wc + 1)>>1;
	    break;

	//------------------------------  End of format... bail out
	case o_End:
	    loop = 1;
	    break;
	default:
	    return Invalid;
	}

/*
 *    End of the loop over format items. decrement the byte counts 
 *    of copied data.
 */
	if (!skip && lwd) {
	    size_type bias  = wc * lwd;
	    nbyt -= bias;
	    obyt -= bias;
	    if (rep <= 0) {
	        if (nbyt > 0) {
		    if (obyt < lwd) loop = 3;
		    else            loop = 4;
		} else if (isMultiBuffer()) {
		    loop = 2;
		} else if (mFormat == NO_TRANS) {
		    loop = 1;
		}
	    } else if (wc < rep) {
	        if (nbyt <= 0)       loop = 2;
		else if (obyt < lwd) loop = 3;
	        else                 loop = 4;
	    }
	}
	cont = false;
    }

    //---------------------------------- It's all over but the caterwaling. 
    //                                   Kick the fat lady.
    *lclen = maxlen - obyt;
    if (loop == 1) {
        if (nbyt > 0)             rc = MisMatch;
	else if (isMultiBuffer()) rc = MisMatch;
	else                      rc = OK;
    } else if (loop == 2) {
        if (isMultiBuffer())      rc = Continue;
	else                      rc = MisMatch;
    } else if (loop == 3) {
        rc = SizeError;
    } else {
        rc = MisMatch;
    }

    //----------------------------------   Set up the block for a restart
    if (rc == Continue) {
	mOffset = long(out) - long(lcl);
        mLevel += 1;
        mLvPtr[mLevel] = itab;
        mLvRpt[mLevel] = rep - wc;
    } else {
        Reset();
    }
    return OK;
}

void 
Translator::Reset(void) {
    mFlags = 0;
}

size_type 
Translator::fmtcd(const char *fmt, size_type len) {

    if (len == 1) {
        switch (fmt[0]) {
	case 'A': 
	    return o_AFmt;
	case 'B': 
	    return o_BFmt;
	case 'D': 
	    return o_DFmt;
	case 'F':
	    return o_FFmt;
	case 'I':
	    return o_IFmt;
	case 'R': 
	    return o_RFmt;
	case 'S': 
	    return o_SFmt;
	}
    } else if (len == 2 && fmt[0] == 'N') {
	switch (fmt[1]) {
	case 'I': 
	    return o_NIFmt;
	case 'S': 
	    return o_NSFmt;
	}
    } else if (len == 2 && fmt[0] == 'U') {
	switch (fmt[1]) {
	case 'B': 
	    return o_UBFmt;
	case 'I': 
	    return o_UIFmt;
	case 'S': 
	    return o_USFmt;
	}
    }
    return 0;
}

size_type
Translator::syntax(const char *fmt, size_type npos) {
    std::cerr << "Syntax error in format:" << std::endl;
    std::cerr << fmt << std::endl;
    for (size_type i=0 ; i<npos ; i++) std::cerr << "-";
    std::cerr << std::endl;
    return Syntax;
}
