/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_MSGHANDLER_HH
#define LMSG_MSGHANDLER_HH
//
//   Base class for server message handlers.
//

#include "lmsg/MsgTypes.hh"

namespace lmsg {

    class MsgHeader;
    class Message;
    class AppServer;
    class Buffer;


    /**  The message handler API class defines what processign will be 
      *  performed when a message of a given type is received.
      *  @brief Message handler API.
      *  @author John Zweizig
      *  @version 1.1; Last Modified March 3, 2008
      *  @ingroup IO_lmsg
      */
    class MsgHandler {
    public:
	/**  Default constructor.
	  *  @brief Constructor.
	  */
	MsgHandler(void);

	/**  Destroy a message handler. This is mainly to pass control to 
	  *  derived handler classes.
	  *  @brief Destructor
	  */
	virtual ~MsgHandler(void);

	/**  Perform processing on the message contained in the specified 
	  *  buffer.
	  *  @brief Handle a message.
	  */
	error_type handleBuffer(AppServer& app, Buffer& buf);

	/**  Perform processing on the specified message.
	  *  @brief Handle a message.
	  */
	virtual error_type handleMsg(AppServer& app, const MsgHeader& hdr) = 0;

	/**  Get a reference to the stored message object.
	  */
	virtual Message& getMessage(void) = 0;
    private:
    };

    /**  Handler is a template for clases handling a single kind of 
      *  message. The data portion of the message is passed as the template
      *  argument, and virtual methods of the class perform the message 
      *  processing.
      *  @brief Message handler template class.
      *  @author John Zweizig
      *  @version 1.1; last modified March 4, 2008
      *  @ingroup IO_lmsg
      */
    template<class Msg> 
    class Handler : public MsgHandler {
    public:

	/**  Default constructor.
	  *  @brief Constructor.
	  */
	Handler(void);

	/**  Destructor.
	  *  @brief Destructor.
	  */
	~Handler(void);

	/**  Perform message specific processing.
	  *  @brief Process message.
	  *  @param app Appserver to be used to reply to the message.
	  *  @param hdr Message header information.
	  *  @return Lmsg error code.
	  */
	error_type handleMsg(AppServer& app, const MsgHeader& hdr);

	/**  Get a reference to the input message.
	  *  @brief Get message reference.
	  *  @return Reference to the input message
	  */
	Msg& getMessage(void);

	/**  Get a constant reference to the input message.
	  *  @brief Get message reference.
	  *  @return Constant reference to the input message
	  */
	const Msg& getMessage(void) const;

    private:
	Msg mMessage;
    };

}  // namespace lmsg

//--------------------------------------  Default constructor
template<class Msg>
lmsg::Handler<Msg>::Handler(void) {
}

//--------------------------------------  Default destructor
template<class Msg>
lmsg::Handler<Msg>::~Handler(void) {
}

//--------------------------------------  Get the scratch message
template<class Msg>
Msg&
lmsg::Handler<Msg>::getMessage(void) {
    return mMessage;
}

template<class Msg>
const Msg&
lmsg::Handler<Msg>::getMessage(void) const {
    return mMessage;
}


#endif  //  LMSG_MSGHANDLER_HH


