#include "html/writer.hh"
#include "html/Attrib.hh"
#include <cmath>
#include <cstring>
#include <iostream>

using namespace html;
using namespace std;

//======================================  HTML writer constructor
writer::writer(std::ostream& ostr) 
 : mStream(&ostr), mDepth(0), mCursor(0)
{
    mFont.push_back(attList());
    ostr << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">" 
	 << std::endl;
}

//======================================  HTML writer destructor
writer::~writer() {
    sync();
}

//======================================  Write text
void 
writer::text(const std::string& data) {
    int j=0;
    indent();
    for (int i=0 ; data[i] ; i++) {
        if (data[i] == '\n') {
	    indent();
	    (*mStream) << data.substr(j,i-j);
	    endLine();
	    j=i+1;
	}
    }
    if (data[j]) {
        indent();
	(*mStream) << data.substr(j);
	mCursor += data.size() - j;
    }
}

//======================================  Write text
void 
writer::endLine(void) {
    (*mStream) << std::endl;
    mCursor = 0;
}

//======================================  Write begin tag
void 
writer::tag(const char* tagname, int nattr, const char* attr[], 
		   const char* value[]) {
    indent();
    (*mStream) << "<" << tagname;
    mCursor += 1 + strlen(tagname);
    for (int i=0 ; i<nattr ; i++) {
        if (value[i] && strlen(value[i])) {
	    (*mStream) << " " << attr[i] << "=\"" << value[i] << "\"";
	    mCursor += 4 + strlen(attr[i]) + strlen(value[i]);
	}
    }
    (*mStream) << ">";
    mCursor++;
    mNest[mDepth++] = tagname;
}

//======================================  Write begin tag
void 
writer::tag(const char* tagname, const attList& atl) {
    indent();
    (*mStream) << "<" << tagname;
    mCursor += 1 + strlen(tagname);
    atl.putAttr(*this);
    (*mStream) << ">";
    mCursor++;
    mNest[mDepth++] = tagname;
}

//======================================  Write begin tag
void 
writer::tagNData(const char* tagname, int nattr, 
			const char* attr[], const char* value[]) {
    indent();
    (*mStream) << "<" << tagname;
    mCursor += strlen(tagname);
    for (int i=0 ; i<nattr ; i++) {
        if (*(value[i])) {
	    (*mStream) << " " << attr[i] << "=\"" << value[i] << "\"";
	    mCursor += 4 + strlen(attr[i]) + strlen(value[i]);
	}
    }
    (*mStream) << ">";
    mCursor++;
}

//======================================  Write begin tag
void 
writer::tagNData(const char* tagname, const attList& atl) {
    indent();
    (*mStream) << "<" << tagname;
    atl.putAttr(*this);
    (*mStream) << ">";
    mCursor++;
}

void 
writer::meta(const char* name, const char* value) {
    indent();
    (*mStream) << "<meta http-equiv=\"" << name << "\" content=\""
	       << value << "\">";
}

void 
writer::meta(const char* name, double value) {
    indent();
    (*mStream) << "<meta http-equiv=\"" << name << "\" content=\""
	       << value << "\">";
}


//======================================  Write end tag
void 
writer::endTag(const std::string& tagname) {
    if (!mDepth) {
        std::cerr << "Error extra end tag: </" << tagname << ">" 
		  << std::endl;
	return;
    }
    mDepth--;
    if (tagname != mNest[mDepth]) {
        std::cerr << "Error: end tag </" << tagname 
		  << "> doesn't match tag <" 
		  << mNest[mDepth] << ">." << std::endl;
    }
    indent();
    (*mStream) << "</" << tagname << ">";
    mCursor += 3 + tagname.size();
}

//======================================  Write an integer.
void 
writer::Integer(int N) {
    indent();
    (*mStream) << N;
    mCursor += 1 + int(log10(double(N)));
}

//======================================  Spew everything to the output file
void
writer::sync(void) {
#if defined(__GNUG__) && __GNUC__ < 3
    mStream->rdbuf()->sync();
#endif
}

//======================================  Indent at the start of a line
void
writer::indent(void) {
    if (mCursor > 72) endLine();
}

//======================================  Make a new default text attributes
void 
writer::pushDefault(const attList& x) {
    mFont.push_back(x);

    //  Note that there is always an earlier default as long as push_back()
    //  is used in the constructor.
    mFont.back().merge(mFont[mFont.size()-2]);
}

void 
writer::popDefault(void) {
    mFont.pop_back();
}
