/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_DOCUMENT_HH
#define HTML_DOCUMENT_HH

#include "html/Attrib.hh"
#include "html/compound.hh"
#include "html/style.hh"
#include <string>

namespace html {
  class color;

  /**  The HTML document class defines the top level of an HTML document. 
    *  A document is a compound HTML object, and includes in addition to
    *  the usual list of contained objects and attributes, a style directory
    *  and a few meta-data values (\a e.g. the title, refresh rate, etc.).
    *  @memo HTML document class.
    *  @author J. Zweizig
    *  @version 1.0; Modified January 11, 2001
    *  @ingroup IO_html
    */
  class document : public compound {
  public:
    /**  Construct an empty document.
      *  @memo Default constructor.
      */
    document(void);

    /**  Make a copy of a document object.
      *  @memo copy constructor.
      *  @param x HTML document to be copied.
      */
    document(const document& x);

    /**  Construct an empty document. Preset the title.
      *  @memo Titled document constructor.
      *  @param title  Document title string.
      */
    document(const char* title);

    /**  Destroy a document.
      *  @memo HTML document destructor.
      */
    ~document(void);

    /**  Copy a document into new storage and return the resulting pointer.
      *  @memo clone a document.
      *  @return Pointer to the clone of this object.
      */
    document* clone(void) const;

    /**  Get the tag name to be used. In this case it is document.
      */
    const char* getTag(void) const;

    /**  Write out the document.
     */
    void write(writer& out) const;

    /**  Set the document background color.
     */
    void setBackgroundColor(const color& color);

    /**  Set the document refresh time. A refresh time of zero indicates
      *  that the document is not to be refreshed.
      *  @memo specify a refresh period.
      */
    void setRefresh(double t);

    /**  Add a new class to the style dictionary.
      *  @memo Add a new styl class.
      *  @param name Name of the new style class.
      *  @param c Style class definition.
      */
    void setStyle(const std::string& name, const StyleClass& c);

  private:
    std::string mTitle;
    style       mStyle;
    double      mRefresh;
  };
}
#endif  // HTML_DOCUMENT_HH
