/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif
#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif

#include "PConfig.h"
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <sys/types.h>
#include <sys/socket.h>
#if defined (P__LINUX)
//#include <sys/time.h>
//#include <linux/types.h>
#include <linux/sockios.h>
#include <sys/ioctl.h>
typedef u_int32_t in_addr_t;
#elif defined (P__WIN32)
#include <sys/ioctl.h>
#define IFF_POINTOPOINT 0
#elif defined (P__DARWIN)
#include <sys/ioctl.h>
#else
#include <sys/sockio.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <pthread.h>

#include <memory>
#include <vector>
#include <algorithm>
#include "sockutil.h"
#include "checksum_crc32.hh"
#include "framexmit/matchInterface.hh"
#include "framexmit/framesend.hh"
#include <cerrno>

using namespace std;

namespace framexmit {

   //===================================  Clear statistics structure
   void 
   frameSend::frsend_stats::clear(void) {
      send_requests   = 0;
      send_packets    = 0;
      resend_requests = 0;
      resend_nodata   = 0;
      resend_toomanypkts = 0;
      resend_invalidpkt  = 0;
      resend_puterror = 0;
      resend_packets  = 0;
      total_sent      = 0;
      total_bytes     = 0;
   }

   //===================================  Count rebroadcast  packets.
   /**  The sendlist maintains a list of {sequence, pkt#} for which a 
     *  rebroadcast has been requested in the order of most recent request 
     *  first.
     */
   struct sendlist {
      struct sendcount {
	 sendcount() : _seq(~0), _pkt(~0), _stamp(0), _count(0) {}
	 sendcount(uint32_t seq, uint32_t pkt, uint64_t t) 
	    : _seq(seq), _pkt(pkt), _stamp(t), _count(1) {}
	 bool operator==(const sendcount& x) const {
	    return _seq==x._seq && _pkt==x._pkt;
	 }
	 void swap(sendcount& x);
	 uint32_t _seq;
	 uint32_t _pkt;
	 uint64_t _stamp;
	 uint32_t _count;
      };
      uint32_t _N;
      std::vector<sendcount> _list;
      sendlist(uint32_t N) : _N(N), _list(N) {}
      /**  Test updates the list to move the count structure for the requested 
        *  packet to the front. If the time epoch has expired the count is
	*  reset. The count for the current (first) packet is then updated and 
	*  the test returns true iff the count is less than the specified 
	*  maximum. Time time stamp is updated to the most recent request 
	*  time to pass the test.
	*/
      bool test(uint32_t seq, uint32_t pkt, uint64_t t, uint32_t max, 
		uint64_t tepoch);
   };

   inline void 
   sendlist::sendcount::swap(sendcount& x) {
      std::swap<uint32_t>(_seq, x._seq);
      std::swap<uint32_t>(_pkt, x._pkt);
      std::swap<uint64_t>(_stamp, x._stamp);
      std::swap<uint32_t>(_count, x._count);
   }

   bool
   sendlist::test(uint32_t seq, uint32_t pkt, uint64_t t, uint32_t max,
		  uint64_t tepoch) {
      sendcount tmp(seq, pkt, t);
      for (uint32_t i=0; i<_N; ++i) {
	 tmp.swap(_list[i]);
	 if (tmp == _list[0]) {
             // tmp has the old values, _list[0] has the new values
	    
             // Have tepoch microseconds passed?
             // If so, just keep _list[0] (the new values) and throw 
             // away tmp. This way, _list[0] shows the new count (1)
             // as well as the new time stamp (now)
	     if (_list[0]._stamp >= tmp._stamp + tepoch) return(true);
 
             // Add the count
             _list[0]._count += tmp._count;
 
             // Also copy in the old time stamp
             _list[0]._stamp = tmp._stamp;
 
             // Now we're done:
             // (1) _list[0]._count <= max
             //     This leaves the time stamp as the old time stamp
             //     The counter will have incremented by one
             //     Returning true causes the new rebroadcast packet to
             //         be emitted
             // (2) _list[0]._count > max
             //     This leaves the time stamp as the old time stamp
             //     The counter will have incremented by one, now larger
             //         than max
             //     Returning false blocks the rebroadcast packet from
             //         being emitted
             return (_list[0]._count <= max); 
 
          } // if tmp == _list[0]
       } // for i
       // If we made it this far, (sequence,packet) was not found and
       // was added to _list above
       // So issue the rebroadcast packet
       return true;
   }

   //===================================  Buffer class
   frameSend::buffer::buffer(void)
   : seq (0), own (false), data (0), len (0), used (0), inUseMux (0),
     timestamp (0), duration (0), sofar (0)
   {}

   frameSend::buffer::buffer(const buffer& buf) 
   {
      *this = buf;
   }

   frameSend::buffer::buffer(char* Data, int Len, unsigned int Seq, 
			     bool Own, bool* Used, mutex* InUseMux,
			     unsigned int Timestamp, unsigned int Duration) 
   : seq (Seq), own (Own), data (Data), len (Len), used (Used), 
     inUseMux (InUseMux), timestamp (Timestamp), duration (Duration), 
     sofar (0)
   {}

   frameSend::buffer::~buffer(void) 
   {
      if (own) {
         delete [] data;
      }
      if (inUseMux != 0) {
         inUseMux->lock();
      }
      if (used != 0) {
         *used = false;
      }
      if (inUseMux != 0) {
         inUseMux->unlock();
      }
   }

   frameSend::buffer& 
   frameSend::buffer::operator= (const buffer& buf) 
   {
      if (this != &buf) {
         own = buf.own;
         seq = buf.seq;
         data = buf.data;
         len = buf.len;
         timestamp = buf.timestamp;
         duration = buf.duration;
         sofar = buf.sofar;
         used = buf.used;
         inUseMux = buf.inUseMux;
         buf.own = false;
         buf.used = 0;
      }
      return *this;
   }

   //===================================  Static burst counter instance.
   static frameSend::burst_counter burster;

   //===================================  Pause if necessary
   void
   frameSend::burst_counter::time_check(void) {
      if (!par.packetBurst || _n < par.packetBurst) return;
      timestamp_type expire = _time + par.packetBurstInterval;
      timestamp_type now = get_timestamp();
      if (now < expire) {
	 micro_delay(expire - now);
      }
      _n = 0;
   }

   //===================================  Force delay if packet has been written
   void
   frameSend::burst_counter::force_delay(void) {
      if (_n) {
	 _n = par.packetBurst;
	 time_check();
      }
   }

   //===================================  Bump counter after one packet
   void 
   frameSend::burst_counter::packet_time(void) {
      _n++;
      _time = get_timestamp();
   }


   void xmitDaemonCallback (frameSend& th) 
   {
      th.xmitDaemon();
   }

extern "C" 
   void xmitDaemonCallback2 (frameSend& th) 
   {
      xmitDaemonCallback (th);
   }

   bool frameSend::open (const char* dest_addr, const char* interface, 
                     int port)
   {
      if (sock >= 0) {
         close();
      }
   
      // set destination address  
      name.sin_family = AF_INET;
      name.sin_port = htons (port);
      if (nslookup(dest_addr, &name.sin_addr)) {
	 save_error("frameSend::open Error in nslookup");
         return false;
      }
   
      // open socket
      sock = socket (PF_INET, SOCK_DGRAM, 0);
      if (sock < 0) {
	 save_error("frameSend::open Error in socket");
         return false;
      }
   
      /* set receive buffer size */
      int bufsize = sndInBuffersize;
      if (setsockopt (sock, SOL_SOCKET, SO_RCVBUF, 
                     (char*) &bufsize, sizeof (int)) != 0) {
	 save_error("frameSend::open Error in setsockopt");
         ::close (sock);
         sock = -1;
         return false;
      }
   
      /* set send buffer size */
      bufsize = sndOutBuffersize;
      if (::setsockopt (sock, SOL_SOCKET, SO_SNDBUF, 
                     (char*) &bufsize, sizeof (int)) != 0) {
	 save_error("frameSend::open Error in setsockopt");
         ::close (sock);
         sock = -1;
         return false;
      }
   
      // bind socket
      sockaddr_in name2;
      name2.sin_family = AF_INET;
      name2.sin_port = 0;
      name2.sin_addr.s_addr = htonl (INADDR_ANY);
      if (::bind (sock, (struct sockaddr*) &name2, sizeof (name2))) {
	 save_error("frameSend::open Error in bind");
         ::close (sock); 
         sock = -1;
         return false;
      }
   
      // broadcast/multicast options
      if (IN_MULTICAST (ntohl(name.sin_addr.s_addr))) {
         multicast = true;
         // multicast: set number of hopes
         unsigned char ttl = mcast_TTL;
         if (setsockopt (sock, IPPROTO_IP, IP_MULTICAST_TTL, 
                        (char*) &ttl, sizeof (ttl)) == -1) {
	    save_error("frameSend::open Error in setsockopt");
            ::close (sock);
            sock = -1;
            return false;
         }
         // multicast: disable loopback
         char loopback = 0;
         if (setsockopt (sock, IPPROTO_IP, IP_MULTICAST_LOOP, 
                        (char*) &loopback, sizeof (loopback)) == -1) {
	    save_error("frameSend::open Error in setsockopt");
            ::close (sock);
            sock = -1;
            return false;
         }
      	 // specify interface
         in_addr i_addr;
         if (!matchInterface (sock, interface, i_addr) ||
            (setsockopt (sock, IPPROTO_IP, IP_MULTICAST_IF, 
			 &i_addr, sizeof (i_addr)) == -1)) {
	    save_error("frameSend::open Error in MatchInterface");
	    ::close (sock);
            sock = -1;
            return false;
         }
      }
      else {
         multicast = false;
         // enable broadcast
         int bset = 1;
         if (setsockopt (sock, SOL_SOCKET, SO_BROADCAST, 
                        (char*) &bset, sizeof (bset)) == -1) {
	    save_error("frameSend::open Error in setsockopt");
            ::close (sock);
            sock = -1;
            return false;
         }
      }
   
      // clear buffers
      mux.lock();
      buffers.clear();
      curbuf = -1;
      skippedDataBuffers = 0;
      mux.unlock();
   
      // create transmit daemon
      int attr = PTHREAD_SCOPE_SYSTEM | PTHREAD_CREATE_DETACHED;
      if (taskCreate (attr, daemonPriority, &daemon, "tXmit",
                     (taskfunc_t) xmitDaemonCallback2, (taskarg_t) this) < 0) {
	 save_error("frameSend::open Error in taskCreate");
         ::close (sock);
         sock = -1;
         return false;
      } 
      daemon_running = true;
   
      return true;
   }


   void 
   frameSend::close(void) {
      mux.lock();
      buffers.clear();
      curbuf = -1;
      mux.unlock();
   
      if (daemon_running) {
	 taskCancel (&daemon); 
      } 
   
      if (sock >= 0) {
	 ::close (sock);
	 sock = -1;
      }
   }

   //===================================  Queue a buffer to be multicasted
   bool 
   frameSend::send (buffer& data)
   {
      if (par.minInterBufferTime) {
	 timestamp_type now = get_timestamp();
	 if (last_buffer + par.minInterBufferTime > now) {
	    micro_delay((last_buffer + par.minInterBufferTime) - now);
	 }
      }

      semlock		lockit (mux);
      typedef deque<buffer>::size_type size_type;
   
      // add send buffer to list
      if (curbuf == -1) {
         // no buffer in list
         buffers.push_back (data);
         curbuf = 0;
      }
      else if (curbuf >= maxbuffers / 2) {
         // too many old buffers
         buffers.pop_front ();
         curbuf--;
         buffers.push_back (data);
      }
      else if (buffers.size() < (size_type) maxbuffers) {
         // enough free space
         buffers.push_back (data);
      }
      else {
         // overload! flush pipe
         while (buffers.size() > (size_type) curbuf + 1) {
            buffers.pop_back();
            skippedDataBuffers++;
         }
         buffers.push_back (data);
      #ifdef DEBUG
         cout << "flush pipe" << endl;
      #endif
      }

      last_buffer = get_timestamp();
      return true;
   }


   //===================================  Put data information into a buffer 
   //                                     and queue it to be sent.
   bool 
   frameSend::send(char* data, int len, bool* used, bool copy,
		   unsigned int timestamp, unsigned int duration)
   {
      if ((sock < 0) || (data == 0)) {
         return false;
      }
   
      // copy data if necessary
      char* p;
      if (copy) {
         p = new (std::nothrow) char [len];
         if (p == 0) {
            return false;
         }
         memcpy (p, data, len);
      }
      else  {
         p = data;
      }
   
      // set in use variable
      inUseMux.lock();
      if (used != 0) {
         *used = true;
      }
      inUseMux.unlock();
   
      // fill into buffer
      buffer buf (p, len, seq++, copy, used, &inUseMux, timestamp, duration);
      return send (buf);
   }

   //===================================  Save error information.
   void 
   frameSend::save_error(const char* msg) const {
      saveMsg = msg;
      saveErr = errno;
   }

   //===================================  Receive one retransmit packet.
   bool 
   frameSend::getRetransmitPacket (retransmitpacket& pkt)
   {
      // poll socket
      struct timeval 	wait;		// timeout=0
      wait.tv_sec = 0;
      wait.tv_usec = 0;
      fd_set 		readfds;	
      FD_ZERO (&readfds);
      FD_SET (sock, &readfds);
      int nset = select (FD_SETSIZE, &readfds, 0, 0, &wait);
      if (nset < 0) {
         return false;
      }
      else if (nset == 0) {
         return false;
      }
   
      // receive a packet from socket
      struct sockaddr_in 	from;
      int n;
      socklen_t max = sizeof (from);
      n = recvfrom (sock, (char*) &pkt, sizeof (packet), 0, 
                   (struct sockaddr*) &from, &max);
      if (n < 0) {
         return false;
      }
      // swap if necessary
      pkt.ntoh();
      // check if valid retransmit packet
      if (n < (int)sizeof (packetHeader) || 
         (pkt.header.pktType != PKT_REQUEST_RETRANSMIT) ||
         (n != (int)sizeof (packetHeader) + pkt.header.pktLen)) {
         return false;
      }
      else {
         return true;
      }
   }


   bool frameSend::putPackets (packet pkts[], int n)
   {
      checksum_crc32 crc;
      for (int i = 0; i < n; i++) {
         // swap if necessary
         int sbytes = sizeof (packetHeader) + pkts[i].header.pktLen;
         pkts[i].hton();

	 //-----------------------------  Calculate the checksum.
	 pkts[i].header.checksum = 0;
	 crc.reset();
	 crc.add(reinterpret_cast<unsigned char*>(pkts + i), sbytes);
	 pkts[i].header.checksum = crc.result();

	 //-----------------------------  enforce any required delay
	 burster.time_check();

         //-----------------------------  Send the packet
	 int flags = 0;
#ifdef P__DARWIN
	 int set = 1;
	 setsockopt(sock, SOL_SOCKET, SO_NOSIGPIPE, &set, sizeof(int));
#else
	 flags = MSG_NOSIGNAL;
#endif
         if (sendto(sock, (char*) (pkts + i), sbytes, flags, 
		    (struct sockaddr*) &name, sizeof (struct sockaddr_in)) 
	     != sbytes) 
	 {
	    save_error("frameSend::putPackets: error in sendto");
            ::close (sock);
	    sock = -1;
            return false;
         }
	 stats.total_sent++;
	 stats.total_bytes += sbytes;

	 //-----------------------------  Burst structure accounting.
	 burster.packet_time();
      }
      return true;
   }


   bool 
   frameSend::compSeqeuence (const frameSend::buffer& b, 
			     const retransmitpacket& p) {
      return (b.seq < p.header.seq);
   }


extern "C" {
   typedef void (*cleanup_type)(void*);

   void xmitDaemonCleanup (packet* pkts)
   {
      delete [] pkts;
   }
}

   static void
   copy_packet(packet& p, const frameSend::buffer& b, int pkt_type, 
	       int pktNum, int pktTotal, size_t pktOff, size_t pktLen) {
      memset (&p.header, 0, sizeof (packetHeader));
      p.header.pktType   = pkt_type;
      // p.header.pktType   = PKT_BROADCAST;
      p.header.seq       = b.seq;
      p.header.timestamp = b.timestamp;
      p.header.duration  = b.duration;
      p.header.pktTotal  = pktTotal;
      p.header.pktNum = pktNum;
      p.header.checksum = 0;
      p.header.pktLen = pktLen;
      memcpy (p.payload, b.data + pktOff, pktLen);
   }

   void 
   frameSend::xmitDaemon () {
      sendlist tpkt(64);

      // allocate packet buffer/install cleanup function
      packet*	pkts   = 0;
      int       pktdim = max (par.packetBurst, maximumRetransmit);
      pkts = new (nothrow) packet[pktdim];
      if (pkts == 0) {
         ::close (sock);
         sock = -1;
         return;
      }
      pthread_cleanup_push ((cleanup_type)xmitDaemonCleanup, pkts);
      daemon_running = true;

      // main loop
      while (1) {
         pthread_testcancel();

         //--- Check if buffer is ready. If so, build some packets
         mux.lock();
         if ((curbuf != -1) && (curbuf < (int)buffers.size())) {
	    stats.send_requests++;

            //---------------------------  Get the sequence counters
            int seqLen   = buffers[curbuf].sequenceLength();
            int seqNum   = buffers[curbuf].sofar;
	    int nBurst   = seqLen - seqNum;

	    //---------------------------   Get the raw packet numbers
            int pktTotal = buffers[curbuf].totalPackets();
	    int pktNum   = seqNum;
	    if (pktNum < pktTotal) {
	       nBurst = pktTotal - pktNum;
	    } else {
	       pktNum -= pktTotal;
	    }
	    if (nBurst > par.packetBurst) nBurst = par.packetBurst;

            //--------------------------  assemble packets
            for (int n = 0; n < nBurst; n++) {
	       size_t pktOff = buffers[curbuf].packetOffset(pktNum + n);
	       size_t pktLen = buffers[curbuf].packetLength(pktNum + n);
	       copy_packet(pkts[n], buffers[curbuf], PKT_BROADCAST, 
			   pktNum+n, pktTotal, pktOff, pktLen);
            }
	    buffers[curbuf].sofar += nBurst;

            // check if buffer is done
            if (buffers[curbuf].sofar >= seqLen) {
               curbuf++;
            }
            mux.unlock();
         
            //--------------------------  Now send nBurst packets
	    if (!pktNum) burster.force_delay();
	    stats.send_packets += nBurst;
            if (!putPackets (pkts, nBurst)) {
            #ifdef DEBUG
               cout << "packet error 1" << endl;
            #endif
               break;
            }
         }

	 //----  no current packets... wait a few ticks.
         else {
            mux.unlock();
	    micro_delay(sndDelayTick);
         }
      
      	 // check if retransmit packets are here
         retransmitpacket rpkt;
         if (!getRetransmitPacket (rpkt)) {
            continue;
         }
      #ifdef DEBUG
         cout << "received a retransmit packet "
	      << rpkt.header.pktLen / sizeof (int) << endl;
      #endif
      
      	 // check if buffers are available
         mux.lock();
	 stats.resend_requests++;
         if (curbuf == -1) {
	    stats.resend_nodata++;
            mux.unlock();
            continue;
         }

      	 // find buffer with same sequence
         deque<buffer>::iterator pos = 
            lower_bound (buffers.begin(), buffers.end(), rpkt, compSeqeuence);
         if ((pos == buffers.end()) || (pos->seq != rpkt.header.seq)) {
	    stats.resend_nodata++;
            mux.unlock();
         #ifdef DEBUG
            cout << "old data no longer available" << endl;
         #endif
            continue;
         }

      	 // retransmit
         int n = 0;
         int pktTotal = pos->totalPackets();
         if (rpkt.header.pktLen / (int)sizeof (int) > maximumRetransmit) {
            mux.unlock();
            continue;
         }

	 uint64_t rcv_time = get_timestamp();

         for (int i = rpkt.header.pktLen / sizeof (int) - 1; i >= 0; i--) {
            // assemble retransmit packet
            int pktNum = rpkt.pktResend[i];
            if (pktNum < 0) {
	       stats.resend_invalidpkt++;
               continue;
            }

	    if (!tpkt.test(rpkt.header.seq, pktNum, rcv_time, 
			   par.sndMaxPacketRebroadcast, 
			   par.sndRebroadcastEpoch)
		) {
	       continue;
	    }
	    size_t pktOff = pktNum * packetSize;
            size_t pktLen = pos->packetLength(pktNum);

            if (pktLen <= 0 || pktLen > packetSize) {
	       stats.resend_invalidpkt++;
               continue;
            }

	    copy_packet(pkts[n], *pos, PKT_REBROADCAST, 
			pktNum, pktTotal, pktOff, pktLen);
            n++;
         }
         mux.unlock();

      	 // send packets
      #ifdef DEBUG
         // cout << "send retransmit packet " << n << endl;
      #endif
	 if (n) {
	    stats.resend_packets += n;
	    if (!putPackets (pkts, n)) {
#ifdef DEBUG
	       cout << "packet error 2" << endl;
#endif
	       break;
	    }
	 }
      }
   
      // remove cleanup function
      pthread_cleanup_pop (1);
      daemon_running = false;
   }
}   //  namespace framexmit
