/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Module Name: framefast						     */
/*                                                                           */
/* Module Description:  fast frame reader				     */
/*                                                                           */
/*---------------------------------------------------------------------------*/

#include "framefast/framefast.hh"
#include "Interval.hh"
#include <stdio.h>
#include <cstdlib>
#include <errno.h>
#include <cstring>
#include <iostream>

namespace framefast {

   using namespace std;

   const char* const frameversion_V4 = "FrameFast 3.0 / Frame V4";
   const char* const frameversion_V6 = "FrameFast 3.0 / Frame V6";
   const char* const frameversion_V8 = "FrameFast 3.0 / Frame V8";

   const int kFrameBufferThreshold = 8 * 1024;
   const int kFrameBufferSize = 1024 * 1024;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// guesssframefilename                                                  //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   string guesssframefilename (const detector_t* det, Time& start,
                     Time& end)
   {
      string s;
      // Detector id
      if (!det) {
         s += 'X';
      }
      else if ((strncasecmp (det->fName, "LIGO_1", 6) == 0) ||
              (strncasecmp (det->fName, "LHO", 3) == 0) ||
              (strncasecmp (det->fName, "Hanford", 7) == 0)) {
         s += 'H';
      }
      else if ((strncasecmp (det->fName, "LIGO_2", 6) == 0) ||
              (strncasecmp (det->fName, "LLO", 3) == 0) ||
              (strncasecmp (det->fName, "Livingston", 10) == 0)) {
         s += 'L';
      }
      else if ((strncasecmp (det->fName, "VIRGO", 5) == 0)) {
         s += 'V';
      }
      else if ((strncasecmp (det->fName, "TAMA", 4) == 0)) {
         s += 'T';
      }
      else if ((strncasecmp (det->fName, "GEO", 3) == 0)) {
         s += 'G';
      }
      else {
         s += 'X';
      }
      s += "-R-";
      // start time (GPS) and duration
      Interval duration = end - start;
      char buf[1024];
      sprintf (buf, "%09lu-%li",  start.getS(), duration.GetS());
      s += buf;
      //cerr << "GUESS " << s << endl;
      return s;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// framerstart (write the frame header & dictionary only once)          //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   class framestart {
   public:
      static int write (int version, char* buf, bool swap);
   private:
      framestart (int version, bool swap);
      static framestart*	fStart4;
      static framestart*	fStart4Swap;
      static framestart*	fStart6;
      static framestart*	fStart6Swap;
      static framestart*	fStart8;
      static framestart*	fStart8Swap;
      int		fVersion;
      char		fBuf[16*1024];
      int		fLen;
   };

   int framestart::write (int version, char* buf, bool swap)
   {
      if (version == 4) {
         if (swap) {
            if (!fStart4Swap) fStart4Swap = new framestart (4, true);
            memcpy (buf, fStart4Swap->fBuf, fStart4Swap->fLen);
            return fStart4Swap->fLen;
         }
         else {
            if (!fStart4) fStart4 = new framestart (4, false);
            memcpy (buf, fStart4->fBuf, fStart4->fLen);
            return fStart4->fLen;
         }
      }
      else if (version == 6) {
         if (swap) {
            if (!fStart6Swap) fStart6Swap = new framestart (6, true);
            memcpy (buf, fStart6Swap->fBuf, fStart6Swap->fLen);
            return fStart6Swap->fLen;
         }
         else {
            if (!fStart6) fStart6 = new framestart (6, false);
            memcpy (buf, fStart6->fBuf, fStart6->fLen);
            return fStart6->fLen;
         }
      }
      else if (version == 8) {
         if (swap) {
            if (!fStart8Swap) fStart8Swap = new framestart (8, true);
            memcpy (buf, fStart8Swap->fBuf, fStart8Swap->fLen);
            return fStart8Swap->fLen;
         }
         else {
            if (!fStart8) fStart8 = new framestart (8, false);
            memcpy (buf, fStart8->fBuf, fStart6->fLen);
            return fStart8->fLen;
         }
      }
      else {
         return 0;
      }
   }

   framestart::framestart (int version, bool swap) 
   : fVersion (version), fLen (0)
   {
      // write file header
      fileheader_t   fhead;
      fhead.init (fVersion);
      int len = fhead.write (fBuf, swap);
      if (len <= 0) {
         return;
      }
      fLen += len;
      // write dictionary
      dict_t dict;
      dict.standard (fVersion);
      len = dict.write (fVersion, fBuf + fLen, swap);
      if (len <= 0) {
         fLen = 0;
         return;
      }
      fLen += len;
      if (fLen > 16*1024) {
         printf ("framestart: FATAL ERROR BUFFER TOO SHORT %i\n", fLen);
      }
   };

   framestart* framestart::fStart4 = 0;
   framestart* framestart::fStart4Swap = 0;
   framestart* framestart::fStart6 = 0;
   framestart* framestart::fStart6Swap = 0;
   framestart* framestart::fStart8 = 0;
   framestart* framestart::fStart8Swap = 0;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// framereader                                                          //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   framereader::framereader ()
   : fDebug (true), /*fLen (0), fAddr (0), fOwn (false), fMMap (false), */
   fTOC (0), fHasTOC (kTOCUndefined), fForceScanToc (false), fFileHeader(0), 
   fHasFileHeader (kFileHeaderUndefined), fFrameHeader (0), fSwap (0)
   {
   }

//______________________________________________________________________________
   framereader::~framereader()
   {
      unload();
   }


//______________________________________________________________________________
   bool framereader::loadFile (const char* filename, bool map)
   {
      if (map) {
         return loadFrame (new (nothrow) mmap_frame_storage (filename));
      }
      else {
         return loadFrame (new (nothrow) file_frame_storage (filename));
      }
   }

//______________________________________________________________________________
   bool framereader::loadFrame (const void* fdata, int len, bool ownit)
   {
      return loadFrame (new (nothrow) memory_frame_storage (
                                         (const char*)fdata, len, ownit));
   }

//______________________________________________________________________________
   bool framereader::releaseFrame (frame_storage_ptr& frame)
   {
      frame = fFrame;
      unload();
      return (frame.data() != 0);
   }

//______________________________________________________________________________
   void framereader::unload ()
   {
      fFrame.reset();
      if (fTOC != 0) {
         freeTOC();
      }
      fHasTOC = kTOCUndefined;
      if (fFileHeader != 0) {
         freeFileHeader();
      }
      fHasFileHeader = kFileHeaderUndefined;
   }

//______________________________________________________________________________
   bool framereader::isFrame ()
   {
      if (frame() == 0) {
         return false;
      }
      if (fHasFileHeader == kFileHeaderUndefined) {
         readFileHeader();
      }
      return (fHasFileHeader == kFileHeaderValid);
   }

//______________________________________________________________________________
   bool framereader::getDict (dict_t& dict)
   {
      if (!isFrame()) {
         return false;
      }
      dict.fDictNum = 0;
      const char* p = frame() + 40;
      generic_t g;
      do {
         g.read (fFileHeader->fVersion, p, fSwap);
         if (g.fClass == 1) {
            dict.fDict[dict.fDictNum].read (fFileHeader->fVersion,p, fSwap);
            dict.fDictNum++;
         }
         p += g.fLen;
      } while ((p < frameend()) && (dict.fDictNum < maxDictElements));
      return true;
   }

//______________________________________________________________________________
   bool framereader::getFrameHeader (frameheader_t& h, int fnum)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return false;
      }
      if ((fnum < 0) || (fnum >= (int)fTOC->fNFrame) ||
         (fTOC->fFrames == 0)) {
         return false;
      }
      return (h.read (fFileHeader->fVersion,
                     frame() + fTOC->fFrames[fnum].fPositionH) > 0);
   }

//______________________________________________________________________________
   bool framereader::getDetectorInfo (detector_t& det, int fnum, int dnum)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return false;
      }
      if ((fnum < 0) || (fnum >= (int)fTOC->fNFrame) ||
         (fTOC->fFrames == 0)) {
         return false;
      }
      if (fFileHeader->fVersion < 6) {
         frameheader_t h;
         int skip = h.read (fFileHeader->fVersion, 
                           frame() + fTOC->fFrames[fnum].fPositionH);
         if ((skip <= 0) || 
            ((h.fDir[3].fDataClass == 0) && (h.fDir[3].fDataInstance == 0))) {
            return false;
         }
         const char* p = frame() + fTOC->fFrames[fnum].fPositionH + skip;
         generic_t g;
         do {
            g.read (fFileHeader->fVersion, p, fSwap);
            if ((g.fClass == h.fDir[3].fDataClass) &&
               (g.fInstance == h.fDir[3].fDataInstance)) {
               return (det.read (fFileHeader->fVersion, p, fSwap) > 0);
            }
            p += g.fLen;
         } while (p < frameend());
         return false;
      }
      else {
         if ((dnum < 0) || (dnum >= (int)fTOC->fNDetector)) {
            return false;
         }
         const char* p = frame() + fTOC->fDetInfo[dnum].fPos;
         return (det.read (fFileHeader->fVersion, p, fSwap) > 0);
      }
   }

//______________________________________________________________________________
   int framereader::getDetectorInfoNum()
   {
      if (fFileHeader->fVersion < 6) {
         return 1;
      }
      else {
         if ((frame() == 0) || (fHasTOC == kTOCNo) ||
            ((fTOC == 0) && !readTOC())) {
            return false;
         }
         return fTOC->fNDetector;
      }
   }

//______________________________________________________________________________
   bool framereader::getHistoryInfo (hist_t& hist, int fnum)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return false;
      }
      if ((fnum < 0) || (fnum >= (int)fTOC->fNFrame) ||
         (fTOC->fFrames == 0)) {
         return false;
      }
      frameheader_t h;
      int skip = h.read (fFileHeader->fVersion, 
                        frame() + fTOC->fFrames[fnum].fPositionH);
      if ((skip <= 0) || 
         ((h.fDir[4].fDataClass == 0) && (h.fDir[4].fDataInstance == 0))) {
         return false;
      }
      const char* p = frame() + fTOC->fFrames[fnum].fPositionH + skip;
      generic_t g;
      do {
         g.read (fFileHeader->fVersion, p, fSwap);
         if ((g.fClass == h.fDir[4].fDataClass) &&
            (g.fInstance == h.fDir[4].fDataInstance)) {
            return (hist.read (fFileHeader->fVersion, p, fSwap) > 0);
         }
         p += g.fLen;
      } while (p < frameend());
      return false;
   }

//______________________________________________________________________________
   bool framereader::getRawDataInfo (rawdata_t& raw, int fnum)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return false;
      }
      if ((fnum < 0) || (fnum >= (int)fTOC->fNFrame) ||
         (fTOC->fFrames == 0)) {
         return false;
      }
      frameheader_t h;
      int skip = h.read (fFileHeader->fVersion, 
                        frame() + fTOC->fFrames[fnum].fPositionH);
      if ((skip <= 0) || 
         ((h.fDir[5].fDataClass == 0) && (h.fDir[5].fDataInstance == 0))) {
         return false;
      }
      const char* p = frame() + fTOC->fFrames[fnum].fPositionH + skip;
      generic_t g;
      do {
         g.read (fFileHeader->fVersion, p, fSwap);
         if ((g.fClass == h.fDir[5].fDataClass) &&
            (g.fInstance == h.fDir[5].fDataInstance)) {
            return (raw.read (fFileHeader->fVersion, p, fSwap) > 0);
         }
         p += g.fLen;
      } while (p < frameend());
      return false;
   }

//______________________________________________________________________________
   bool framereader::getData (data_t& dat, int_8u_t pos, 
                     datatype_t dtype, frvect_t::datacopy cpy)
   {
      if ((frame() == 0) || ((int)pos >= length())) {
         return false;
      }
      const char* p = frame() + pos;
      // read FrADCData
      int ofs = dat.fADC.read (fFileHeader->fVersion, dtype, p, fSwap);
      p += ofs;
      //cerr << dat.fADC << endl;
      // read FrVect
      int ret = dat.fVect.read (fFileHeader->fVersion, p, fSwap, cpy);
      // Q: Does it hurt to work for ldas? Who comes up with this shit?!?
      if ((fFileHeader->fVersion >= 6) && (dtype == kProcData)) {
         dat.fADC.fSampleRate = (dat.fVect.fDx[0] > 0) ?
            1./ dat.fVect.fDx[0] : 1.;
      }
      //cerr << dat.fVect << endl;
   
      // ::TEMP:: cout temp
      // if (dat.fVect.data()) {
         // cerr << "GET     Data = ";
         // for (int i = 0; i < 10; i++) {
            // cerr << ((float*)(dat.fVect.data()))[i] << " ";
         // }
         // cerr << endl;
      // }
   
      return (ret > 0);
   }

//______________________________________________________________________________
   bool framereader::getData (data_t& dat, const char* channel, 
                     int nfrm, frvect_t::datacopy cpy)
   {
      if ((channel == 0) || (nfrm < 0)) {
         return false;
      }
      int_8u_t* pos;
      datatype_t dtype;
      int num = lookup (channel, pos, dtype);
      if ((num <= 0) || (nfrm >= num)) {
         return false;
      }
      return getData (dat, pos[nfrm], dtype, cpy);
   }

//______________________________________________________________________________
   int framereader::copy (const char* chnname, real_4_t* x, int max)
   {
      int_8u_t*	pos;
      datatype_t dtype;
   
      int num = lookup (chnname, pos, dtype);
      if (num <= 0) {
         return num;
      }
      int ofs = 0;
      for (int i = 0; i < num; i++) {
         const char* p = frame() + pos[i];
         // skip FrADCData
         adcdata_t adcdata;
         adcdata.read (fFileHeader->fVersion, dtype, p, fSwap);
         p += adcdata.fLen;
         // read FrVect
         frvect_t vec;
         vec.read (fFileHeader->fVersion, p, fSwap);
         // copy data
         int N = (int) vec.fNData;
         if (ofs + N > max) {
            N = max - ofs;
         }
         if (x) vec.get (x + ofs, N);
         ofs += vec.fNData;
      }
      return ofs;
   }

//______________________________________________________________________________
   int framereader::copy (const char* chnname, real_8_t* x, int max)
   {
      int_8u_t*	pos;
      datatype_t dtype;
   
      int num = lookup (chnname, pos, dtype);
      if (num <= 0) {
         return num;
      }
      int ofs = 0;
      for (int i = 0; i < num; i++) {
         const char* p = frame() + pos[i];
         // skip FrADCData
         adcdata_t adcdata;
         adcdata.read (fFileHeader->fVersion, dtype, p, fSwap);
         p += adcdata.fLen;
         // read FrVect
         frvect_t vec;
         vec.read (fFileHeader->fVersion, p, fSwap);
         // copy data
         int N = (int) vec.fNData;
         if (ofs + N > max) {
            N = max - ofs;
         }
         if (x) vec.get (x + ofs, N);
         ofs += vec.fNData;
      }
      return ofs;
   }

//______________________________________________________________________________
   int framereader::copy (const char* chnname, int_2s_t* x, int max)
   {
      int_8u_t*	pos;
      datatype_t dtype;
   
      int num = lookup (chnname, pos, dtype);
      if (num <= 0) {
         return num;
      }
      int ofs = 0;
      for (int i = 0; i < num; i++) {
         const char* p = frame() + pos[i];
         // skip FrADCData
         adcdata_t adcdata;
         adcdata.read (fFileHeader->fVersion, dtype, p, fSwap);
         p += adcdata.fLen;
         // read FrVect
         frvect_t vec;
         vec.read (fFileHeader->fVersion, p, fSwap);
         // copy data
         int N = (int) vec.fNData;
         if (ofs + N > max) {
            N = max - ofs;
         }
         if (x) vec.get (x + ofs, N);
         ofs += vec.fNData;
      }
      return ofs;
   }

//______________________________________________________________________________
   int framereader::copy (const char* chnname, int_4s_t* x, int max)
   {
      int_8u_t*	pos;
      datatype_t dtype;
   
      int num = lookup (chnname, pos, dtype);
      if (num <= 0) {
         return num;
      }
      int ofs = 0;
      for (int i = 0; i < num; i++) {
         const char* p = frame() + pos[i];
         // skip FrADCData
         adcdata_t adcdata;
         adcdata.read (fFileHeader->fVersion, dtype, p, fSwap);
         p += adcdata.fLen;
         // read FrVect
         frvect_t vec;
         vec.read (fFileHeader->fVersion, p, fSwap);
         // copy data
         int N = (int) vec.fNData;
         if (ofs + N > max) {
            N = max - ofs;
         }
         if (x) vec.get (x + ofs, N);
         ofs += vec.fNData;
      }
      return ofs;
   }

//______________________________________________________________________________
   std::string framereader::guessFilename()
   {
      string g = fname() ? fname() : "";
      if (g.empty()) {
         detector_t det;
         if (getDetectorInfo (det)) {
            Time start = starttime();
            Time next = nexttime();
            g = guesssframefilename (&det, start, next);
         }
      }
      return g;
   }

//______________________________________________________________________________
   int framereader::nframe ()
   {
      const toc_t* TOC = getTOC();
      if (TOC == 0) {
         return 0;
      }
      else {
         return TOC->fNFrame;
      }
   }

//______________________________________________________________________________
   Time framereader::starttime (int framenum)
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0) ||
         (framenum < 0) || (framenum >= nframe())) {
         return Time (0);
      }
      else {
         return Time (TOC->fFrames[framenum].fGTimeS, 
                     TOC->fFrames[framenum].fGTimeN);
      }
   }

//______________________________________________________________________________
   Interval framereader::duration (int framenum)
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0) ||
         (framenum < 0) || (framenum >= nframe())) {
         return Interval (0.0);
      }
      else {
         return Interval (TOC->fFrames[framenum].fDt);
      }
   }

//______________________________________________________________________________
   Time framereader::nexttime ()
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0)) {
         return Time (0);
      }
      else {
         int n = nframe() - 1;
         return starttime (n) + duration (n);
      }
   }

//______________________________________________________________________________
   const toc_t* framereader::getTOC()
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return 0;
      }
      return fTOC;
   }

//______________________________________________________________________________
   int framereader::lookup (const char* chnname, int_8u_t*& ofs,
                     datatype_t& dtype)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return -1;
      }
   
      for (int j = 0; j < 4; j++) {
         toc_data_t* found = fTOC->find (chnname, j);
         if (found != 0) {
            ofs = found->fPosition;
            dtype = (datatype_t)j;
            return fTOC->fNFrame;
         }
      }
   
      return 0;
   }

//______________________________________________________________________________
   bool framereader::newTOC ()
   {
      freeTOC();
      fTOC = new toc_t;
      fHasTOC = kTOCUndefined;
      return true;
   }

//______________________________________________________________________________
   bool framereader::freeTOC ()
   {
      fHasTOC = kTOCUndefined;
      if (fTOC == 0) {
         return true;
      }
      delete fTOC;
      fTOC = 0;
      return true;
   }

//______________________________________________________________________________
   bool framereader::readTOC ()
   {
      if ((fHasTOC == kTOCNo) || (frame() == 0) || !newTOC() ||
         !isFrame()) {
         return false;
      }
   
      // get address of TOC
      endof_file_t	eof;
      eof.read (fFileHeader->fVersion, frame(), length(), fSwap);
   
      // read TOC if exists
      if (!fForceScanToc && (eof.fSeekTOC != 0) && 
         ((int)eof.fSeekTOC < length())) {
         if (fTOC->read (fFileHeader->fVersion, frameend() - eof.fSeekTOC, fSwap) <= 0) {
            fHasTOC = kTOCNo;
            delete fTOC; fTOC = 0;
            return false;
         }
         else {
            fHasTOC = kTOCYes;
            return true;
         }
      }
      // otherwise scan TOC
      else {
         static int TOCWarningMax = 3;
         if (TOCWarningMax) {
            cerr << "Warning: rescan table of contents" << endl;
            TOCWarningMax--;
         }
         if (!scanTOC()) {
            fHasTOC = kTOCNo;
            delete fTOC; fTOC = 0;
            return false;
         }
         else {
            // resize TOC
            toc_t* toc = fTOC;
            fTOC = new (nothrow) toc_t (*toc);
            delete toc;
            fHasTOC = fTOC ? kTOCYes : kTOCNo;
            return true;
         }
      }
   }

//______________________________________________________________________________
   bool framereader::scanTOC ()
   {
      if (!isFrame()) {
         return false;
      }
      // scan through file and regenerate TOC
      fTOC->fNFrame = 1024;
      if (!fTOC->init (fFileHeader->fVersion)) {
         return false;
      }
      return fTOC->scan (frame(), length(), fSwap);
   }

//______________________________________________________________________________
   std::ostream& framereader::writeTOC (std::ostream& os)
   {
      if ((frame() == 0) || (fHasTOC == kTOCNo) ||
         ((fTOC == 0) && !readTOC())) {
         return os;
      }
   
      os << *fTOC << endl;
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, framereader& fr)
   {
      return fr.writeTOC (os);
   }

//______________________________________________________________________________
   bool framereader::newFileHeader ()
   {
      freeFileHeader();
      fFileHeader = new (nothrow) fileheader_t;
      if (fFileHeader == 0) {
         fHasFileHeader = kFileHeaderInvalid;
         return false;
      }
      fHasFileHeader = kFileHeaderUndefined;
      return true;
   }

//______________________________________________________________________________
   bool framereader::freeFileHeader ()
   {
      fHasFileHeader = kFileHeaderUndefined;
      if (fFileHeader == 0) {
         return true;
      }
      delete fFileHeader;
      fFileHeader = 0;
      return true;
   }

//______________________________________________________________________________
   bool framereader::readFileHeader ()
   {
      if ((fHasFileHeader == kFileHeaderInvalid) || (frame() == 0) || 
         !newFileHeader()) {
         return false;
      }
      // read header
      if (length() < 40) {
         fHasFileHeader = kFileHeaderInvalid;
         return false;
      }
      fFileHeader->read (frame());
      // check if valid frame of version 4 or higher
      if ((strncmp ((const char*)fFileHeader->fIGWD, "IGWD", 5) != 0) ||
         (fFileHeader->fVersion < 4)) {
         fHasFileHeader = kFileHeaderInvalid;
         return false;
      }
      else {
         fHasFileHeader = kFileHeaderValid;
         fSwap = (fFileHeader->fByteOrder2 != 0x1234);
         return true;
      }
   }

//______________________________________________________________________________
   std::ostream& framereader::writeFileHeader (std::ostream& os)
   {
      if ((frame() == 0) || (fHasFileHeader == kFileHeaderInvalid) ||
         ((fFileHeader == 0) && !readFileHeader())) {
         return os;
      }
      os << *fFileHeader << endl;
      return os;   
   }


//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const framereader& fr)
   {
      if (fr.frame() && (fr.length() > 0)) {
         dumpframe (os, fr.frame(), fr.length());
      }
      return os;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// framebuffer_t                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   framewriter::framebuffer_t::framebuffer_t (int size)
   : fLen (0), fSize (0), fData (0), fOwn (false) 
   {
      allocate (size);
   }

//______________________________________________________________________________
   framewriter::framebuffer_t::framebuffer_t (const framebuffer_t& buf)
   : fLen (0), fSize (0), fData (0), fOwn (false) 
   {
      *this = buf;
   }

//______________________________________________________________________________
   framewriter::framebuffer_t::~framebuffer_t () 
   {
      deallocate ();
   }

//______________________________________________________________________________
   framewriter::framebuffer_t& framewriter::framebuffer_t::operator= (
                     const framebuffer_t& buf)
   {
      if (this != &buf) {
         deallocate ();
         fLen = buf.fLen;
         fSize = buf.fSize;
         fData = buf.fData;
         fOwn = buf.fOwn;
         buf.fOwn = false;
      }
      return *this;
   }

//______________________________________________________________________________
   bool framewriter::framebuffer_t::allocate (int size) 
   {
      deallocate();
      if (size <= 0) {
         return true;
      }
      else {
         fData = new (nothrow) char [size + kFrameBufferThreshold];
         //cerr << "allocate framebuffer_t   @ " << (int) fData << endl;
         fSize = (fData != 0) ? size : 0;
         fLen = 0;
         fOwn = (fData != 0);
         return (fData != 0);
      }
   }

//______________________________________________________________________________
   void framewriter::framebuffer_t::deallocate () 
   {
      if (fOwn && fData) {
         //cerr << "deallocate framebuffer_t @ " << (int) fData << endl;
         delete [] fData;
      }
      fData = 0;
      fSize = 0;
      fLen = 0;
      fOwn = false;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// framewriter                                                          //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   framewriter::framewriter (int nFrameLen, int nFrame, int compress,
                     int version)
   : fSwap (false), fVersion (version), fFramePos (framebegin), 
   fRunNum (0), fFrameNum (0), fFrameLen (1), fNFrame (1), 
   fCompress (0), fN (0), fTotal (0)
   {
      fSwap = littleendian();
      setFormat (nFrameLen, nFrame, compress, version);
   }

//______________________________________________________________________________
   framewriter::~framewriter ()
   {
   }

//______________________________________________________________________________
   bool framewriter::setFormat (int nFrameLen, int nFrame, 
                     int compress, int version)
   { 
      if ((fFramePos == framebegin) || (fFramePos == frameend)) {
         fFrameLen = nFrameLen;
         if (fFrameLen < 1) fFrameLen = 1;
         if (fFrameLen > 16*1024) fFrameLen = 16*1024;
         fNFrame = nFrame;
         if (fNFrame < 1) fNFrame = 1;
         if (fNFrame > 16*1024) fNFrame = 16*1024;
         fCompress = compress & 0X00FF;
         if (fCompress < 0) fCompress = 0;
	 // Current spec (v8) has compression modes 8, 9, 10 defined
         //if (fCompress > 6) fCompress = 3;
	 // Current version doesn't implement zero suppress long
         if (fCompress > 8) fCompress = 3;
         fVersion = version;
         return true;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool framewriter::setTime (const Time& time)
   {
      fNext = time;
      if ((fFramePos == framebegin) || (fFramePos == frameend)) {
         fTime = time;
      }
      return true;
   }

//______________________________________________________________________________
   bool framewriter::setDetectorInfo (const detector_t& det)
   {
      fDetector.clear();
      fDetector.push_back (det);
      return true;
   }

//______________________________________________________________________________
   bool framewriter::addDetectorInfo (const detector_t& det)
   {
      fDetector.push_back (det);
      return true;
   }

//______________________________________________________________________________
   int framewriter::nframe () const
   {
      return fNFrame;
   }

//______________________________________________________________________________
   Time framewriter::starttime (int framenum) const
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0) ||
         (framenum < 0) || (framenum >= nframe())) {
         return Time (0);
      }
      else {
         return Time (TOC->fFrames[framenum].fGTimeS, 
                     TOC->fFrames[framenum].fGTimeN);
      }
   }

//______________________________________________________________________________
   Interval framewriter::duration (int framenum) const
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0) ||
         (framenum < 0) || (framenum >= nframe())) {
         return Interval (0.0);
      }
      else {
         return Interval (TOC->fFrames[framenum].fDt);
      }
   }

//______________________________________________________________________________
   Time framewriter::nexttime () const
   {
      const toc_t* TOC = getTOC();
      if ((TOC == 0) || (TOC->fFrames == 0)) {
         return Time (0);
      }
      else {
         int n = nframe() - 1;
         return starttime (n) + duration (n);
      }
   }
//______________________________________________________________________________
   bool framewriter::addData (const adcdata_t& adcin, 
                     const frvect_t& vectin)
   {
      // ::TEMP:: out temp
      // if (vectin.data()) {
         // cerr << "PUT     Data = ";
         // for (int i = 0; i < 10; i++) {
            // cerr << ((float*)(vectin.data()))[i] << " ";
         // }
         // cerr << endl;
      // }
   
      // check frame condition
      if ((fFramePos == frameend) || (fFramePos == framebegin)) {
         purge();
         fN = 0;
         beginFile();
         beginFrame();
         fFramePos = framebusy;
      }
      else if (fFramePos == framebreak) {
         beginFrame();
         fFramePos = framebusy;
      }
      if ((fN < 0) || (fN >= fNFrame)) {
         return false;
      }
   
      // update TOC
      toc_data_t* chn;
      if (fN == 0) {
         chn = fTOC.add (adcin.fName);
         if (chn) {
            chn->fChannelID = adcin.fChannelNumber;
            chn->fGroupID = adcin.fChannelGroup;
         }
      }
      else {
         chn = fTOC.find (adcin.fName);
      }
      if (chn == 0) {
         // cerr << "unknown channel " << adcin.fName << "++++++++++++++++++++++++++++++++++" << endl;
         return false;
      }
      chn->fPosition[fN] = fTotal;
      //cerr << "add channel " << adcin.fName << " @ " << fTotal << endl;
   
      // get frame buffer
      framebuffer_t* buf = currentBuffer();
      if (buf == 0) {
         return false;
      }
   
      // write adc structure
      adcdata_t adc (adcin);
      adc.fClass = fPtr[4].fDataClass;
      adc.fInstance = fPtr[4].fDataInstance;
      adc.fData[0] = fPtr[5];
      adc.fData[1] = ptr_struct ();
      adc.fData[2] = fPtr[4];
      adc.fData[2]++;
      int len = adc.write (fVersion, buf->pos(), fSwap);
      if (len <= 0) {
         return false;
      }
      buf->expand (len);;
      fTotal += len;
      fPtr[4]++;
   
      // write data vector
      frvect_t vect;
      vect.image (vectin);
      vect.fClass = fPtr[5].fDataClass;
      vect.fInstance = fPtr[5].fDataInstance;
      vect.fCompress = fCompress;
      if (fSwap) {
         vect.fCompress += 256;
      }
      vect.fNext = ptr_struct ();
      // check if enough space
      if (buf->len() + (int)vect.fNBytes < 
         buf->size() + kFrameBufferThreshold / 2) {
         len = vect.write (fVersion, buf->pos(), fSwap);
         if (len <= 0) {
            return false;
         }
         buf->expand (len);;
         fTotal += len;
      }
      // if not, copy vector first and write partial data
      else {
         char* dat = 
            new (nothrow) char [
            (int) (1.1*vect.fNBytes + kFrameBufferThreshold)];
         if (dat == 0) {
            return false; 
         }
         len = vect.write (fVersion, dat, fSwap);
         if (len > (int)vect.fNBytes + kFrameBufferThreshold) {
            cerr << "Danger: intermediate buffer overrun, " <<
               "data may be corrupted" << endl;
         }
         if ((len <= 0) || !appendBuffer (dat, len)) {
            delete [] dat;
            return false;
         }
         delete [] dat;
         fTotal += len;
      }
      fPtr[5]++;
   
      return true;
   }

//______________________________________________________________________________
   bool framewriter::next() 
   {
      //cerr << "next frame" << endl;
      //cerr << "next = " << fNext << " " << fFrameLen << endl;
      Time t = fNext + Interval (fFrameLen, 0);
      return next (t);
   }

//______________________________________________________________________________
   bool framewriter::next (const Time& nextframe)
   {
      if ((fFramePos == framebusy) && (fN < fNFrame)) {
         endFrame();
         fN++;
         if (fN < fNFrame) {
            fFramePos = framebreak;
         }
         else {
            fFramePos = frameend;
            endFile();
         }
         fNext = nextframe;
         if (fFramePos == frameend) {
            fTime = fNext;
         }
         return true;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool framewriter::write (basic_frameout* out)
   {
      int nbuf = buffers();
      if ((out == 0) || (nbuf <= 0)) {
         return true;
      }
      // prepare scatter/gather write
      basic_frameout::src_dest_t* src = new 
         (nothrow) basic_frameout::src_dest_t [nbuf];
      if (src == 0) {
         return false;
      }
      int size = 0;
      int j = 0;
      for (framebufferlist::iterator i = fFrameBuffer.begin();
          i != fFrameBuffer.end(); ++i, ++j) {
         src[j].fAddr = i->data(); 
         src[j].fLen = i->len();
         size += i->len();
      }
      // open/write/close
      bool ret = true;
      if (out->open (size)) {
         ret = out->write (src, nbuf);
         out->close();
      }
      // cleanup
      delete [] src;
      purge();
      return ret;
   }

//______________________________________________________________________________
   std::string framewriter::guessFilename() const
   {
      Time start = starttime();
      Time next = nexttime();
      return guesssframefilename (getDetectorInfo(), start, next);
   }

//______________________________________________________________________________
   framewriter::framebuffer_t* framewriter::currentBuffer()
   {
      // get new buffer if empty queue
      if (fFrameBuffer.empty()) {
         fFrameBuffer.push_back (framebuffer_t (kFrameBufferSize));
      }
      // otherwise check if buffer overrun has occured
      else if (fFrameBuffer.back().len() >= fFrameBuffer.back().size()) {
         int overrun = 
            fFrameBuffer.back().len() - fFrameBuffer.back().size();
         framebuffer_t buf (kFrameBufferSize);
         // copy overrun into new buffer
         if ((overrun > 0) && buf.data()) {
            const char* end = 
               fFrameBuffer.back().data() + fFrameBuffer.back().size();
            memcpy (buf.pos(), end, overrun);
            fFrameBuffer.back().expand (-overrun);
            buf.expand (overrun);
         }
         fFrameBuffer.push_back (buf);
      }
      // now return address to buffer
      framebuffer_t* buf = &(fFrameBuffer.back());
      return buf->data() ? buf : 0;
   }

//______________________________________________________________________________
   bool framewriter::appendBuffer (const char* dat, int len)
   {
      int left = len;
      int ofs = 0;
      while (left > 0) {
         framebuffer_t* buf = currentBuffer();
         if (buf == 0) {
            cerr << "Internal error: append buffer 1" << endl;
            return false;
         }
         int n = buf->size() - buf->len();
         if (n <= 0) {
            cerr << "Internal error: append buffer 2" << endl;
            return false;
         }
         if (left < n) n = left;
         memcpy (buf->pos(), dat + ofs, n);
         left -= n;
         ofs += n;
         buf->expand (n);
      }
      return true;
   }

//______________________________________________________________________________
   bool framewriter::beginFile()
   {
      framebuffer_t* buf = currentBuffer();
      if (buf == 0) {
         return false;
      }
   
      // write file header & dictionary
      int len = framestart::write (fVersion, buf->pos(), fSwap);
      buf->expand (len);;
      fTotal += len;
   //    // write file header
      // fileheader_t   fhead;
      // fhead.init();
      // int len = fhead.write (fVersion, buf->pos(), fSwap);
      // if (len <= 0) {
         // return false;
      // }
      // buf->expand (len);;
      // fTotal += len;
   // 
   //    // write dictionary
      // dict_t dict;
      // dict.standard();
      // len = dict.write (fVersion, buf->pos(), fSwap);
      // if (len <= 0) {
         // return false;
      // }
      // buf->expand (len);;
      // fTotal += len;
   
      // init pointers
      fPtr[0].set (fVersion, kFrameH);
      fPtr[1].set (fVersion, kFrDetector);
      fPtr[2].set (fVersion, kFrHistory);
      fPtr[3].set (fVersion, kFrRawData);
      fPtr[4].set (fVersion, kFrAdcData);
      fPtr[5].set (fVersion, kFrVect);
      fPtr[6].set (fVersion, kFrEndOfFrame);
      fPtr[7].set (fVersion, kFrTOC);
      fPtr[8].set (fVersion, kFrEndOfFile);
   
      // init TOC
      fTOC.fULeapS = LeapS (fNext);
      fTOC.fLocalTime = 0;
      fTOC.fNFrame = fNFrame;
      fTOC.init (fVersion);
   
      return true;
   }

//______________________________________________________________________________
   bool framewriter::endFile()
   {
      framebuffer_t* buf = currentBuffer();
      if (buf == 0) {
         return false;
      }
      // write TOC
      int posTOC = fTotal;
      fTOC.fClass = fPtr[7].fDataClass;
      fTOC.fInstance = fPtr[7].fDataInstance;
      int toclen = fTOC.size (fVersion);
         // check if enough space
      int len = 0;
      if (buf->len() + toclen < buf->size() + kFrameBufferThreshold/2) {
         len = fTOC.write (fVersion, buf->pos(), fSwap);
         buf->expand (len);;
      }
         // if not, copy TOC first and write partial data
      else {
         char* t = new (nothrow) char [toclen + kFrameBufferThreshold];
         if (t == 0) {
            return false; 
         }
         len = fTOC.write (fVersion, t, fSwap);
         if (len > toclen + kFrameBufferThreshold) {
            cerr << "Danger: intermediate buffer overrun, " <<
               "data may be corrupted (toc)" << endl;
         }
         if ((len <= 0) || !appendBuffer (t, len)) {
            delete [] t;
            return false;
         }
         delete [] t;
         buf = currentBuffer();
      }
      if ((buf == 0) || (len <= 0)) {
         return false;
      }
      fTotal += len;
      fPtr[7]++;
   
      // write EOF
      endof_file_t   eof;
      eof.fClass = fPtr[8].fDataClass;
      eof.fInstance = fPtr[8].fDataInstance;
      eof.fNFrames = fNFrame;
      eof.fNBytes = fTotal + endof_file_t::size (fVersion);
      eof.fChkType = 0;
      eof.fChkSum = 0;
      eof.fSeekTOC = eof.fNBytes - posTOC;
      len = eof.write (fVersion, buf->pos(), fSwap);
      if (len != endof_file_t::size (fVersion)) {
         return false;
      }
      buf->expand (len);
      fTotal += len;
      fPtr[8]++;
   
      // this is the last buffer, make sure overrun is handled correctly
      buf = currentBuffer();
      return true;
   }

//______________________________________________________________________________
   bool framewriter::beginFrame()
   {
      framebuffer_t* buf = currentBuffer();
      if (buf == 0) {
         return false;
      }
      // write frame header
      int_8u_t headerpos = fTotal;
      frameheader_t   frhead;
      frhead.fClass = fPtr[0].fDataClass;
      frhead.fInstance = fPtr[0].fDataInstance;
      frhead.fRun = fRunNum;
      frhead.fFrame = fFrameNum;
      frhead.fDataQual = 0;
      frhead.fGTimeS = fNext.getS();
      frhead.fGTimeN = fNext.getN();
      frhead.fULeapS = LeapS (fNext);
      frhead.fLocalTime = 0;
      frhead.fFrameLen = fFrameLen;
      frhead.fDir[3] = fPtr[1];
      frhead.fDir[4] = fPtr[2];
      frhead.fDir[5] = fPtr[3];
      int len = frhead.write (fVersion, buf->pos(), fSwap);
      if (len <= 0) {
         return false;
      }
      buf->expand (len);
      fTotal += len;
      fPtr[0]++;
   
      // write detector data
      for (detector_array_t::iterator i = fDetector.begin();
          i != fDetector.end(); ++i) {
         int_8u_t detpos = fTotal;
         i->fClass = fPtr[1].fDataClass;
         i->fInstance = fPtr[1].fDataInstance;
         fPtr[1]++;
         i->fMore[0] = ptr_struct();
         i->fMore[1] = ptr_struct();
         if (i + 1 != fDetector.end()) {
            i->fMore[2] = fPtr[1];
         }
         else {
            i->fMore[2] = ptr_struct();
         }
         len = i->write (fVersion, buf->pos(), fSwap);
         if (len <= 0) {
            return false;
         }
         buf->expand (len);
         fTotal += len;
         toc_detector_t* det = fTOC.addDetector (i->fName);
         if (det) det->fPos = detpos;
         if (fVersion < 6) {
            break;
         }
      }
   
      // write history data
      hist_t	hist;
      hist.fClass = fPtr[2].fDataClass;
      hist.fInstance = fPtr[2].fDataInstance;
      strcpy (hist.fName, "Frame Writer");
      hist.fTime = Now().getS();
      strcpy (hist.fComment, fVersion < 6 ? 
             frameversion_V4 : frameversion_V6);
      len = hist.write (fVersion, buf->pos(), fSwap);
      if (len <= 0) {
         return false;
      }
      buf->expand (len);
      fTotal += len;
      fPtr[2]++;
   
      // write raw data
      rawdata_t   raw;
      raw.fClass = fPtr[3].fDataClass;
      raw.fInstance = fPtr[3].fDataInstance;
      strcpy (raw.fName, "rawData");
      raw.fData[1] = fPtr[4];
      len = raw.write (fVersion, buf->pos(), fSwap);
      if (len <= 0) {
         return false;
      }
      buf->expand (len);
      fTotal += len;
      fPtr[3]++;
   
      // update TOC
      if ((fN >= 0) && (fN < (int)fTOC.fNFrame) && (fTOC.fFrames != 0)) {
         fTOC.fFrames[fN].fGTimeS = fNext.getS();
         fTOC.fFrames[fN].fGTimeN = fNext.getN();
         fTOC.fFrames[fN].fDt = fFrameLen;
         fTOC.fFrames[fN].fRun = fRunNum;
         fTOC.fFrames[fN].fFrame = fFrameNum;
         fTOC.fFrames[fN].fPositionH = headerpos;
         fTOC.fFrames[fN].fNFirstADC = fTotal;
      }
   
      return true;
   }

//______________________________________________________________________________
   bool framewriter::endFrame()
   {
      framebuffer_t* buf = currentBuffer();
      if (buf == 0) {
         return false;
      }
      // write end of frame
      endof_frame_t   eof;
      eof.fClass = fPtr[6].fDataClass;
      eof.fInstance = fPtr[6].fDataInstance;
      eof.fRun = fRunNum;
      eof.fFrame = fFrameNum;
      int len = eof.write (fVersion, buf->pos(), fSwap);
      if (len <= 0) {
         return false;
      }
      buf->expand (len);;
      fTotal += len;
      fPtr[6]++;
      fFrameNum++;
   
      return true;
   }


}
