/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "GenParam.hh"
#include "rndm.hh"
#include <sstream>
#include <stdexcept>
#include <cmath>
#include <cstdlib>
#include <float.h>

using namespace std;

namespace generator {

  //====================================  Internal translation tables
  static GenParam::DistType lDistTyp[] = {
    GenParam::kStep,  GenParam::kGeom, GenParam::kFlat, GenParam::kGauss, 
    GenParam::kPower, GenParam::kExp
  };

  static const char* lDistNam[] = {
    "step",          "xstep",         "flat",          "gauss", 
    "power",         "exp", 0
  };

  //====================================  Get distribution ID
  GenParam::DistType
  GenParam::getDistID(const std::string& name) {
      for (int i=0; lDistNam[i]; i++) {
	  if (name == lDistNam[i]) return lDistTyp[i];
      }
      throw runtime_error("Undefined distribution type");
  }

  //====================================  Get distribution name
  const char* 
  GenParam::getDistName(DistType id) {
      for (int i=0; lDistNam[i]; i++) {
	  if (lDistTyp[i] == id) return lDistNam[i];
      }
      throw runtime_error("Invalid distribution ID");
  }

  //====================================  Parameter constructor.
  GenParam::GenParam(void) 
    : mType(kConst), mNumValue(0.0)
  {
  }

  //====================================  Parameter destructor.
  GenParam::~GenParam(void) {
  }

  //====================================  Sample a ranedom parameter.
  double 
  GenParam::sampleParameter(void) {
      if (mType != kDistrib) return getNumeric();
      switch (mDistr) {
      case kStep:  //--- Step over range: Par[0]=min, Par[1]=max, Par[2]=Step
	  mNumValue += mDistPar[2];
	  if (mNumValue >= mDistPar[1]) mNumValue = mDistPar[0];
	  break;
      case kGeom:  //--- Step over range: Par[0]=min, Par[1]=max, Par[2]=Step
	  mNumValue *= mDistPar[2];
	  if (mNumValue >= mDistPar[1]) mNumValue = mDistPar[0];
	  break;
      case kFlat:  //--- Flat distribution: Par[0]=min, Par[1]=max
	  mNumValue = mDistPar[0] + (mDistPar[1] - mDistPar[0]) * Rndm();
	  break;
      case kGauss: //--- gaussian distribution: Par[0]=mean, Par[1]=sigma
	  mNumValue = Rannor()*mDistPar[1] + mDistPar[0];
	  break;
      case kPower: //--- x^N distribution: Par[0]=N+1, Par[1]=min^(N+1)
	           //                      Par[2]=(max^(N+1)-min^(N+1))
	           //    or for N=-1     : Par[0]=0, Par[1]=log(min),
	           //                      Par[2]=log(max/min)
	{
	  double z   = Rndm();
	  double pwr = mDistPar[0];
	  if (!pwr) mNumValue = exp(z*mDistPar[2] + mDistPar[1]);
	  else      mNumValue = pow(z*mDistPar[2] + mDistPar[1], 1./pwr);
	  break;
	}
      case kExp:   //--- Exponential: Par[0]=mean, Par[1]=min
	           //                 Par[2]=1-exp((min-max)/mean) 
	{
	  double z = 1.0 - mDistPar[2] * Rndm();
	  if (z <= 0) z = FLT_MIN;
	  mNumValue = mDistPar[1] - mDistPar[0] * log(z);
	  break;
	}
      }
 
      //--------------------------------  Update the string representation
      ostringstream ss;
      ss << mNumValue;
      mStrValue = ss.str();
      return mNumValue;
  }

  //====================================  Set the parameter to distribution
  void 
  GenParam::setDistribution(const std::string& value) {

      //--------------------------------  Parse our distribution name
      int inx = value.find("(");
      string dist = value.substr(0, inx);
      mDistr = getDistID(dist);

      //--------------------------------  Get the distribution parameters
      for (int i=0 ; i<3 ; i++) mDistPar[i] = 0;
      char const * ptr = value.c_str() + inx;
      for (int iarg=0 ; iarg<3 ; iarg++) {
	  mDistPar[iarg] = strtod(++ptr, const_cast<char**>(&ptr));
	  if (*ptr != ',') break;
      }
      if (*ptr != ')') throw runtime_error("Invalid argument list");

      //-------------------------------  Check the parameters
      switch (mDistr) {
      case kStep:
	  if (mDistPar[2] <= 0.0 || mDistPar[0] >= mDistPar[1]) { 
	      throw runtime_error("Invalid argument list");
	  }
	  break;
      case kGeom:
	  if (mDistPar[2] <= 1.0 || mDistPar[0] >= mDistPar[1]) { 
	      throw runtime_error("Invalid argument list");
	  }
	  break;
      case kFlat:
	  if (mDistPar[0] == mDistPar[1]) {
	      throw runtime_error("Invalid argument list");
	  }
	  break;
      case kGauss:
	  if (mDistPar[1] == 0) throw runtime_error("Invalid argument list");
	  break;
      case kPower:
	{
	  double pwr  = mDistPar[0] + 1.0;
	  double ymin = mDistPar[1];
	  double ymax = mDistPar[2];
	  if (pwr == 0.0) {
	      if (ymin <= 0.0) mDistPar[1] = 0; // default min = 1.0
	      else             mDistPar[1] = log(ymin);
	      if (ymax <= 0.0) mDistPar[2] = log(99999999.0);
	      else             mDistPar[2] = log(ymax) - mDistPar[1];
	  } else {
	      if (ymin <= 0.0) mDistPar[1] = 1.0; // default min = 1.0
	      else             mDistPar[1] = pow(ymin, pwr);
	      if (ymax <= 0.0) mDistPar[2] = -mDistPar[1];
	      else             mDistPar[2] = pow(ymax, pwr) - mDistPar[1];
	  } 
	  mDistPar[0] = pwr;
	  break;
	}
      case kExp:
	  if (mDistPar[0] == 0) throw runtime_error("Invalid argument list");
	  if (mDistPar[2] <= mDistPar[1]) mDistPar[2] = 1.0;
	  else mDistPar[2] = 1-exp((mDistPar[1]-mDistPar[2])/mDistPar[0]);
      }
      mType  = kDistrib;
  }

  //====================================  Set a numeric parameter
  void 
  GenParam::setParameter(double value) {
      mType     = kConst;
      mNumValue = value;
      ostringstream ss;
      ss << mNumValue;
      mStrValue = ss.str();
  }

  //====================================  Set a string parameter
  void 
  GenParam::setParameter(const string& value) {
      mType     = kString;
      mStrValue = value;
      istringstream ss(mStrValue);
      ss >> mNumValue;
      if (ss.fail()) {
	 mNumValue = 0.0;
      } else {
	 mType = kConst;
      }
  }

  //====================================  Dump the parameter type value, etc
  ostream& 
  GenParam::dumpDefinition(ostream& out) const {
      switch (mType) {
      case kString:
	  out << "'" << mStrValue << "'";
	  break;
      case kConst:
	  out << mNumValue;
	  break;
      case kDistrib:
	  out << getDistName(mDistr) << "(" << mDistPar[0] << "," 
	      << mDistPar[1] << "," << mDistPar[2] << ")";
	  break;
      }
      return out;
  }

  ostream& 
  GenParam::dump(ostream& out) const {
      switch (mType) {
      case kString:
	  out << "Type: String        Value: '" << mStrValue << "'" << endl;
	  break;
      case kConst:
	  out << "Type: Numeric       Value: " << mNumValue << endl;
	  break;
      case kDistrib:
	  out << "Type: Distribution  Value: ";
	  dumpDefinition(out) << " = " << mNumValue << endl;
	  break;
      }
      return out;
  }

} // namespace generator
